package com.bizunited.empower.business.payment.repository;

import com.bizunited.empower.business.payment.entity.DefrayInfo;
import com.bizunited.empower.business.payment.repository.internal.DefrayInfoRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.math.BigDecimal;

import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * DefrayInfo业务模型的数据库方法支持
 * @author saturn
 */
@Repository("_DefrayInfoRepository")
public interface DefrayInfoRepository
    extends
      JpaRepository<DefrayInfo, String>
      ,JpaSpecificationExecutor<DefrayInfo>
      , DefrayInfoRepositoryCustom
  {
  /**
   * 按照应收账款编号进行详情查询（包括关联信息）
   * @param id 应收账款编号
   * */
  @Query("select distinct defrayInfo from DefrayInfo defrayInfo "
      + " left join fetch defrayInfo.paymentInfo defrayInfo_paymentInfo "
       + " where defrayInfo_paymentInfo.id = :id order by defrayInfo.defrayTime desc")
  Set<DefrayInfo> findDetailsByPaymentInfo(@Param("id") String id);

  /**
   * 按照主键进行详情查询（包括关联信息）
   * @param id 主键
   * */
  @Query("select distinct defrayInfo from DefrayInfo defrayInfo "
      + " left join fetch defrayInfo.paymentInfo defrayInfo_paymentInfo "
      + " left join fetch defrayInfo.fundsAccount defrayInfo_fundsAccount "
      + " left join fetch defrayInfo.defrayAttachments defrayInfo_Attachments "
      + " where defrayInfo.id=:id ")
  DefrayInfo findDetailsById(@Param("id") String id);
  /**
   * 按照收款单流水编号进行查询
   * @param defrayCode 收款单流水编号
   * */
  @Query(" from DefrayInfo f where f.defrayCode=:defrayCode and f.tenantCode=:tenantCode and f.defrayStatus<>0")
  DefrayInfo findByDefrayCode(@Param("defrayCode") String defrayCode, @Param("tenantCode") String tenantCode);

  /**
   * 按照收款单流水编号进行查询明细
   * @param defrayCode 收款单流水编号
   * */
  @Query("select distinct defrayInfo from DefrayInfo defrayInfo "
       + " left join fetch defrayInfo.paymentInfo"
       + " where defrayInfo.defrayCode=:defrayCode and defrayInfo.tenantCode=:tenantCode and defrayInfo.defrayStatus<>0")
  DefrayInfo findDetailsByDefrayCode(@Param("defrayCode") String defrayCode, @Param("tenantCode") String tenantCode);

  /**
   * 按人员和指定日期，统计付款总金额
   * @param account 创建人账号(付款人业务员账号)
   * @param defrayTime 付款日期
   * @param tenantCode 租户编码
   */
  @Query(value = "select sum(ifnull(defray_amount,0)) from defray_info where create_account=:account and date_format(defray_time,'%Y-%m-%d')=date_format(:defrayTime,'%Y-%m-%d') and tenant_code=:tenantCode and defray_status = 1",nativeQuery = true)
  BigDecimal sumTotalDefrayAmountByAccountAndDefrayTimeAndTenantCode(@Param("account") String account, @Param("defrayTime") Date defrayTime, @Param("tenantCode") String tenantCode);

  /**
   * 按人员和指定日期，统计付款总单数
   * @param account 创建人账号(付款人业务员账号)
   * @param defrayTime 付款日期
   */
  @Query(value = "select count(id) from defray_info where create_account=:account and date_format(defray_time,'%Y-%m-%d')=date_format(:defrayTime,'%Y-%m-%d') and tenant_code=:tenantCode and defray_status = 1",nativeQuery = true)
  long countTotalDefrayNumByAccountAndDefrayTimeAndTenantCode(@Param("account") String account, @Param("defrayTime") Date defrayTime, @Param("tenantCode") String tenantCode);
  /**
  * 按照客户编号以及开始结束时间查询应付账单
  * @param customerCode
  * @param tstatus
  * @param startTime
  * @param endTime
  * @param tenantCode
  */
  @Query("select defrayInfo from DefrayInfo defrayInfo " +
          "left join fetch defrayInfo.paymentInfo " +
          "where defrayInfo.customerCode=:customerCode " +
          "and defrayInfo.defrayStatus=:tstatus " +
          "and defrayInfo.createTime between :startTime and :endTime " +
          "and defrayInfo.tenantCode=:tenantCode order by defrayInfo.createTime desc")
  List<DefrayInfo> findByCustomerCodeAndTstatusAndTimesAndTenantCode(@Param("customerCode") String customerCode, @Param("tstatus") Integer tstatus, @Param("startTime") Date startTime, @Param("endTime") Date endTime, @Param("tenantCode") String tenantCode);

    /**
     * 按照付款单编号查询付款单信息
     *
     * @param defrayCodes
     * @return
     */
  @Query("select defrayInfo from DefrayInfo defrayInfo " +
          "left join fetch defrayInfo.paymentInfo " +
          "where defrayInfo.defrayCode in (:defrayCodes) " +
          "order by defrayInfo.createTime desc")
  List<DefrayInfo> findByDefrayCodes(@Param("defrayCodes") List<String> defrayCodes);
}