package com.bizunited.empower.business.payment.repository;

import com.bizunited.empower.business.payment.entity.PaymentInfo;
import com.bizunited.empower.business.payment.repository.internal.PaymentInfoRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.math.BigDecimal;
import java.util.Map;

/**
 * PaymentInfo业务模型的数据库方法支持
 *
 * @author saturn
 */
@Repository("_PaymentInfoRepository")
public interface PaymentInfoRepository
    extends
    JpaRepository<PaymentInfo, String>
    , JpaSpecificationExecutor<PaymentInfo>
    , PaymentInfoRepositoryCustom {

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  @Query("select distinct paymentInfo from PaymentInfo paymentInfo "
      + " where paymentInfo.id=:id ")
  PaymentInfo findDetailsById(@Param("id") String id);

  /**
   * 按照应收账款流水编号进行查询
   *
   * @param paymentCode 应收账款流水编号
   */
  @Query(" from PaymentInfo f "
      + " left join fetch f.defrayInfos "
      + " where f.paymentCode = :paymentCode and f.tenantCode = :tenantCode and f.tstatus=1")
  PaymentInfo findByPaymentCodeAndTenantCode(@Param("paymentCode") String paymentCode, @Param("tenantCode") String tenantCode);

  /**
   * 按照订单编号进行查询
   *
   * @param associatedCode 订单编号
   */
  @Query(" from PaymentInfo f "
      + " left join fetch f.defrayInfos "
      + " where f.associatedCode = :associatedCode and f.tenantCode = :tenantCode and f.tstatus=1")
  PaymentInfo findByAssociatedCodeAndTenantCode(@Param("associatedCode") String associatedCode, @Param("tenantCode") String tenantCode);

  /**
   * 按照租户统计资金情况
   *
   * @param tenantCode
   * @return
   */
  @Query("select sum(f.payAmount) as payAmount,sum(f.waitPayAmount) as waitPayAmount,sum(f.waitConfirmAmount) as waitConfirmAmount,sum(f.payedAmount) as payedAmount from PaymentInfo f where f.tenantCode = :tenantCode and f.tstatus=1")
  Map<String, Object> statistics(@Param("tenantCode") String tenantCode);

  /**
   * 统计应付金额
   * @param tenantCode
   * @return
   */
  @Query("select sum(p.waitPayAmount) from PaymentInfo p where p.tenantCode = :tenantCode and p.tstatus = 1")
  BigDecimal sumWaitPayAmountByTenantCode(@Param("tenantCode") String tenantCode);

  /**
   * 根据客户编号统计客户应付账款数据
   * @param customerCode
   * @param tenantCode
   * @return
   */
  @Query(value = "select c.customer_code as customerCode,count(c.payment_code) as paymentNum,sum(c.pay_amount) as payAmount,sum(c.payed_amount) as payedAmount,sum(c.wait_pay_amount) as waitPayAmount,sum(c.wait_confirm_amount) as waitConfirmAmount from payment_info c where c.tstatus=1 and c.customer_code=:customerCode and c.tenant_code=:tenantCode group by c.customer_code", nativeQuery = true)
  Map<String,Object> findAggregationByCustomerCode(@Param("customerCode") String customerCode, @Param("tenantCode") String tenantCode);
}