package com.bizunited.empower.business.payment.repository;

import com.bizunited.empower.business.payment.entity.ReceiptInfo;
import com.bizunited.empower.business.payment.repository.internal.ReceiptInfoRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * ReceiptInfo业务模型的数据库方法支持
 * @author saturn
 */
@Repository("_ReceiptInfoRepository")
public interface ReceiptInfoRepository
    extends
      JpaRepository<ReceiptInfo, String>
      ,JpaSpecificationExecutor<ReceiptInfo>
      ,ReceiptInfoRepositoryCustom 
  {
  /**
   * 按照应收账款编号进行详情查询（包括关联信息）
   * @param id 应收账款编号
   * */
  @Query("select distinct receiptInfo from ReceiptInfo receiptInfo "
      + " left join fetch receiptInfo.receivableInfo receiptInfo_receivableInfo "
       + " where receiptInfo_receivableInfo.id = :id order by receiptInfo.receiptTime desc")
  Set<ReceiptInfo> findDetailsByReceivableInfo(@Param("id") String id);

  /**
   * 按照主键进行详情查询（包括关联信息）
   * @param id 主键
   * */
  @Query("select distinct receiptInfo from ReceiptInfo receiptInfo "
      + " left join fetch receiptInfo.receivableInfo receiptInfo_receivableInfo "
      + " left join fetch receiptInfo.fundsAccount receiptInfo_fundsAccount "
      + " left join fetch receiptInfo.receiptAttachments receiptInfo_receiptAttachments "
      + " where receiptInfo.id=:id ")
  ReceiptInfo findDetailsById(@Param("id") String id);
  /**
   * 按照收款单流水编号进行查询
   * @param receiptCode 收款单流水编号
   * */
  @Query(" from ReceiptInfo f where f.receiptCode = :receiptCode and f.tenantCode=:tenantCode and f.receiptStatus<>0")
  ReceiptInfo findByReceiptCode(@Param("receiptCode") String receiptCode, @Param("tenantCode") String tenantCode);

  /**
   * 按照收款单流水编号进行查询
   * @param receiptCode 收款单流水编号
   * */
  @Query("select distinct receiptInfo from ReceiptInfo receiptInfo "
      + " left join fetch receiptInfo.receivableInfo"
      + " where receiptInfo.receiptCode = :receiptCode and receiptInfo.tenantCode=:tenantCode and receiptInfo.receiptStatus<>0")
  ReceiptInfo findDetailsByReceiptCode(@Param("receiptCode") String receiptCode, @Param("tenantCode") String tenantCode);

  /**
  * 按人员和指定日期，统计收款总金额
  * @param account 创建人账号(收款人业务员账号)
  * @param receiptTime 收款日期
  * @param tenantCode 租户编码
  */
  @Query(value = "select ifnull(sum(ifnull(receipt_amount,0)),0) from receipt_info where create_account=:account and date_format(receipt_time,'%Y-%m-%d')=date_format(:receiptTime,'%Y-%m-%d') and tenant_code=:tenantCode and receipt_status = 2",nativeQuery = true)
  BigDecimal sumTotalReceiptAmountByAccountAndReceiptTimeAndTenantCode(@Param("account")String account, @Param("receiptTime")Date receiptTime, @Param("tenantCode") String tenantCode);

  /**
   * 按人员和指定日期，统计收款总金额
   * @param account 创建人账号(收款人业务员账号)
   * @param receiptTime 收款日期
   * @param tenantCode 租户编码
   */
  @Query(value = "select count(id) from receipt_info where create_account=:account and date_format(receipt_time,'%Y-%m-%d')=date_format(:receiptTime,'%Y-%m-%d') and tenant_code=:tenantCode and receipt_status = 2",nativeQuery = true)
  long countTotalReceiptNumByAccountAndReceiptTimeAndTenantCode(@Param("account")String account, @Param("receiptTime")Date receiptTime, @Param("tenantCode") String tenantCode);
  /**
   * 按照客户编号以及开始结束时间查询应收账单
   * @param customerCode
   * @param tstatus
   * @param startTime
   * @param endTime
   * @param tenantCode
   * @return
   */
  @Query("select receiptInfo from ReceiptInfo receiptInfo " +
          "left join fetch receiptInfo.receivableInfo " +
          "where receiptInfo.customerCode=:customerCode " +
          "and receiptInfo.receiptStatus=:tstatus " +
          "and receiptInfo.createTime between :startTime and :endTime " +
          "and receiptInfo.tenantCode=:tenantCode order by receiptInfo.createTime desc")
  List<ReceiptInfo> findByCustomerCodeAndTstatusAndTimesAndTenantCode(@Param("customerCode") String customerCode, @Param("tstatus") Integer tstatus, @Param("startTime") Date startTime, @Param("endTime") Date endTime, @Param("tenantCode") String tenantCode);

    /**
     * 按照客户编号查询应收账单
     * @param customerCode 客户编号
     * @param tstatus 状态
     * @param fundsChannels 资金渠道数组
     * @param tenantCode 客户编号
     * @return
     */
    @Query("select receiptInfo from ReceiptInfo receiptInfo " +
            "left join fetch receiptInfo.receivableInfo " +
            "where receiptInfo.customerCode=:customerCode " +
            "and receiptInfo.receiptStatus=:tstatus " +
            "and receiptInfo.fundsChannel in (:fundsChannels) " +
            "and receiptInfo.tenantCode=:tenantCode order by receiptInfo.createTime desc")
    List<ReceiptInfo> findWaitForByCustomerCodeAndFundsChannelsAndTenantCode(@Param("customerCode") String customerCode,@Param("fundsChannels") List<Integer> fundsChannels, @Param("tstatus") Integer tstatus, @Param("tenantCode") String tenantCode);

    /**
     * 按照收款单号集合查询收款单
     * @param receiptCodes
     * @return
     */
    @Query("select receiptInfo from ReceiptInfo receiptInfo " +
            "left join fetch receiptInfo.receivableInfo " +
            "where receiptInfo.receiptCode in (:receiptCodes) " +
            "order by receiptInfo.createTime desc")
    List<ReceiptInfo> findByReceiptCodes(@Param("receiptCodes")List<String> receiptCodes);
  }