package com.bizunited.empower.business.payment.repository;

import com.bizunited.empower.business.payment.entity.ReceivableInfo;
import com.bizunited.empower.business.payment.repository.internal.ReceivableInfoRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * ReceivableInfo业务模型的数据库方法支持
 * @author saturn
 */
@Repository("_ReceivableInfoRepository")
public interface ReceivableInfoRepository
    extends
      JpaRepository<ReceivableInfo, String>
      ,JpaSpecificationExecutor<ReceivableInfo>
      ,ReceivableInfoRepositoryCustom 
  {

  /**
   * 按照主键进行详情查询（包括关联信息）
   * @param id 主键
   * */
  @Query("select distinct receivableInfo from ReceivableInfo receivableInfo "
      + " where receivableInfo.id=:id ")
  ReceivableInfo findDetailsById(@Param("id") String id);
  /**
   * 按照应收账款流水编号进行查询
   * @param receivableCode 应收账款流水编号
   * */
  @Query(" from ReceivableInfo f "
       + " left join fetch f.receiptInfos "
       + " where f.receivableCode = :receivableCode and f.tenantCode = :tenantCode and f.tstatus=1")
  ReceivableInfo findByReceivableCodeAndTenantCode(@Param("receivableCode") String receivableCode,@Param("tenantCode") String tenantCode);
  /**
   * 按照订单编号进行查询
   * @param associatedCode 订单编号
   * */
  @Query(" from ReceivableInfo f "
       + " left join fetch f.receiptInfos "
       + " where f.associatedCode = :associatedCode and f.tenantCode = :tenantCode and f.tstatus=1")
  ReceivableInfo findByAssociatedCodeAndTenantCode(@Param("associatedCode") String associatedCode,@Param("tenantCode") String tenantCode);

  /**
   * 按照订单编号集合进行查询
   * @param associatedCodes 订单编号集合
   * */
  @Query("select distinct f from ReceivableInfo f "
          + " left join fetch f.receiptInfos receiptInfos"
          + " left join fetch receiptInfos.fundsAccount fundsAccount"
          + " where f.associatedCode in (:associatedCodes) and f.tenantCode = :tenantCode and f.tstatus=1" +
          " order by receiptInfos.receiptCode,receiptInfos.receiptTime desc")
  List<ReceivableInfo> findByAssociatedCodesAndTenantCode(@Param("associatedCodes") Set<String> associatedCodes, @Param("tenantCode") String tenantCode);
  /**
   * 按照租户统计资金情况
   * @param tenantCode
   * @return
   */
  @Query("select sum(f.receivableAmount) as receivableAmount,sum(f.waitReceiveAmount) as waitReceiveAmount,sum(f.waitConfirmAmount) as waitConfirmAmount,sum(f.receivedAmount) as receivedAmount from ReceivableInfo f where f.tenantCode = :tenantCode and f.tstatus=1")
  Map<String , Object> statistics(@Param("tenantCode") String tenantCode);

  /**
   * 统计应收金额
   * @param tenantCode
   * @return
   */
  @Query("select sum(r.waitReceiveAmount) from ReceivableInfo r where r.tenantCode = :tenantCode and r.tstatus = 1")
  BigDecimal sumWaitReceiveAmountByTenantCode(@Param("tenantCode") String tenantCode);

  /**
   * 查询客户所有的应收账单
   * @return
   */
  @Query("select receivableInfo from ReceivableInfo receivableInfo " +
          "where receivableInfo.customerCode=:customerCode " +
          "and receivableInfo.tenantCode=:tenantCode order by receivableInfo.createTime desc")
  List<ReceivableInfo> findByCustomerAndTenantCode(@Param("customerCode") String customerCode, @Param("tenantCode") String tenantCode);

  /**
   * 根据客户编号统计客户应收账款数据
   * @param customerCode
   * @param tenantCode
   * @return
   */
  @Query(value = "select c.customer_code as customerCode,count(c.receivable_code) as receivableNum,sum(c.receivable_amount) as receivableAmount,sum(c.received_amount) as receivedAmount,sum(c.wait_receive_amount) as waitReceiveAmount,sum(c.wait_confirm_amount) as waitConfirmAmount from receivable_info c where c.tstatus=1 and c.customer_code=:customerCode and c.tenant_code=:tenantCode group by c.customer_code", nativeQuery = true)
  Map<String,Object> findAggregationByCustomerCode(@Param("customerCode") String customerCode, @Param("tenantCode") String tenantCode);
}