package com.bizunited.empower.business.payment.repository.internal;

import com.bizunited.empower.business.payment.vo.AssociatedPaymentVo;
import com.bizunited.platform.common.repository.PageRepositoryImpl;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * PaymentInfo模型的数据层自定义接口实现，可以由程序员根据实际情况完善
 *
 * @author saturn
 */
public class PaymentInfoRepositoryImpl implements PaymentInfoRepositoryCustom, PageRepositoryImpl {
  /**
   * 租户编码key
   */
  private final String TENANT_CODE = "tenantCode";
  /**
   * 客户编号
   */
  private final String CUSTOMER_CODE = "customerCode";
  /**
   * 应收账款状态 {@link com.bizunited.empower.business.payment.common.enums.ReceivableStatus}
   */
  private final String PAYMENT_STATUS = "paymentStatus";
  /**
   * 单据开始时间
   */
  private final String START_TIME = "startTime";
  /**
   * 单据结束时间
   */
  private final String END_TIME = "endTime";
  /**
   * 应收账款编号
   */
  private final String PAYMENT_CODE = "paymentCode";
  /**
   * 创建人账号（业务员账号）
   */
  private final String CREATE_ACCOUNT = "account";

  @Autowired
  @PersistenceContext
  private EntityManager entityManager;

  @Override
  public Page<AssociatedPaymentVo> queryPageForAssociated(Pageable pageable, InvokeParams conditions) {
    StringBuilder hql = new StringBuilder("from PaymentInfo c where 1=1 and c.tstatus=1");
    StringBuilder countHql = new StringBuilder("select count(*) FROM PaymentInfo c where 1=1 and c.tstatus=1");
    StringBuilder condition = new StringBuilder();
    Map<String, Object> parameters = new HashMap<>();

    Map<String, Object> params = conditions.getInvokeParams();
    String tenantCode = (String) params.get(TENANT_CODE);
    String paymentStatus = (String) params.get(PAYMENT_STATUS);
    String startTime = (String) params.get(START_TIME);
    String endTime = (String) params.get(END_TIME);
    String paymentCode = (String) params.get(PAYMENT_CODE);
    String account = (String) params.get(CREATE_ACCOUNT);
    String customerCode = (String) params.get(CUSTOMER_CODE);

    Validate.notBlank(tenantCode, "租户编号为空");

    condition.append(" and c.tenantCode=:tenantCode");
    parameters.put("tenantCode", tenantCode);

    if (conditions != null) {
      if (StringUtils.isNotBlank(account)) {
        condition.append(" and c.createAccount=:account");
        parameters.put("account", account);
      }
      if (StringUtils.isNotBlank(customerCode)) {
        condition.append(" and c.customerCode=:customerCode");
        parameters.put("customerCode", customerCode);
      }
      if (StringUtils.isNotBlank(paymentStatus)) {
        condition.append(" and c.paymentStatus=:paymentStatus");
        parameters.put("paymentStatus", Integer.parseInt(paymentStatus));
      }
      if (StringUtils.isNotBlank(startTime)) {
        condition.append(" and c.createTime >=:startTime");
        parameters.put("startTime", parseDate(startTime));
      }
      if (StringUtils.isNotBlank(endTime)) {
        condition.append(" and c.createTime <=:endTime");
        parameters.put("endTime", parseDate(endTime));
      }
      if (StringUtils.isNotBlank(paymentCode)) {
        condition.append(" and c.paymentCode like concat(:paymentCode,'%')");
        parameters.put("paymentCode", paymentCode);
      }
    }
    hql.append(condition).append(" order by c.createTime desc");
    countHql.append(condition);
    return queryByConditions(entityManager, hql.toString(), countHql.toString(), parameters, pageable, false, AssociatedPaymentVo.class);
  }

  @SuppressWarnings("unchecked")
  @Override
  public Page<Object[]> queryPageForCustomer(Pageable pageable, InvokeParams conditions) {
    StringBuilder sql = new StringBuilder("select c.customer_code as customerCode,c.customer_name as customerName,cl.level_name as customerLevel,cc.`name` as customerCategory,sa.sales_area_name as salesName,count(c.payment_code) as paymentNum,sum(c.pay_amount) as payAmount,sum(c.payed_amount) as payedAmount,sum(c.wait_pay_amount) as waitpayAmount,sum(c.wait_confirm_amount) as waitConfirmAmount");
    StringBuilder countSql = new StringBuilder("select count(c.customer_code)");
    StringBuilder joinSql = new StringBuilder();
    joinSql.append(" from payment_info c ");
    joinSql.append(" left join customer as cu on c.customer_code=cu.customer_code and c.tenant_code=cu.tenant_code ");
    joinSql.append(" left join customer_level cl on cu.customer_level=cl.id ");
    joinSql.append(" left join customer_category cc on cc.id=cu.customer_category ");
    joinSql.append(" left join sales_area sa on sa.id=cu.sales_area_id ");

    StringBuilder condition = new StringBuilder();
    Map<String, Object> parameters = new HashMap<>();

    Map<String, Object> params = conditions.getInvokeParams();
    String tenantCode = (String) params.get(TENANT_CODE);
    String paymentStatus = (String) params.get(PAYMENT_STATUS);
    String startTime = (String) params.get(START_TIME);
    String endTime = (String) params.get(END_TIME);
    String paymentCode = (String) params.get(PAYMENT_CODE);
    String account = (String) params.get(CREATE_ACCOUNT);

    Validate.notBlank(tenantCode, "租户编号为空");
    condition.append("where 1=1 and c.tstatus=1 ");

    condition.append("and c.tenant_code=:tenantCode ");
    parameters.put("tenantCode", tenantCode);

    if (conditions != null) {
      if (StringUtils.isNotBlank(account)) {
        condition.append(" and c.create_account=:account");
        parameters.put("account", account);
      }
      if (StringUtils.isNotBlank(paymentStatus)) {
        condition.append(" and c.payment_status=:paymentStatus");
        parameters.put("paymentStatus", Integer.parseInt(paymentStatus));
      }
      if (StringUtils.isNotBlank(startTime)) {
        condition.append(" and c.create_time >=:startTime");
        parameters.put("startTime", parseDate(startTime));
      }
      if (StringUtils.isNotBlank(endTime)) {
        condition.append(" and c.create_time <=:endTime");
        parameters.put("endTime", parseDate(endTime));
      }
      if (StringUtils.isNotBlank(paymentCode)) {
        condition.append(" and c.payment_code like concat(:paymentCode,'%')");
        parameters.put("paymentCode", paymentCode);
      }
    }
    sql.append(joinSql).append(condition).append(" group by c.customer_code,c.customer_name,cl.level_name,cc.`name`,sa.sales_area_name");
    countSql.append(joinSql).append(condition).append(" group by c.customer_code");

    Query query = entityManager.createNativeQuery(sql.toString());
    Query countQuery = entityManager.createNativeQuery(countSql.toString());
    parameters.forEach((k, v) -> {
      query.setParameter(k, v);
      countQuery.setParameter(k, v);
    });

    // 构造分页信息
    query.setFirstResult(pageable.getPageNumber() * pageable.getPageSize());
    query.setMaxResults(pageable.getPageSize());
    List<Object[]> resutls = query.getResultList();
    // 查询总数量信息
    Number count = null;
    if (!CollectionUtils.isEmpty(countQuery.getResultList())) {
      count = (Number) countQuery.getResultList().get(0);
    }
    return new PageImpl<>(resutls, pageable, count == null ? 0L : count.longValue());
  }

  /**
   * 日期格式转换
   *
   * @param time
   * @return
   */
  private Date parseDate(String time) {
    SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    Date date = null;
    try {
      date = df.parse(time);
    } catch (ParseException e) {
      throw new IllegalArgumentException("日期格式转换错误", e);
    }
    return date;
  }
}
