package com.bizunited.empower.business.payment.repository.internal;

import com.bizunited.platform.common.repository.PageRepositoryImpl;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * ReceiptInfo模型的数据层自定义接口实现，可以由程序员根据实际情况完善
 *
 * @author saturn
 */
public class ReceiptInfoRepositoryImpl implements ReceiptInfoRepositoryCustom, PageRepositoryImpl {
  /**
   * 租户编码key
   */
  private final String TENANT_CODE = "tenantCode";
  /**
   * 客户编号
   */
  private final String CUSTOMER_CODE = "customerCode";
  /**
   * 收款单状态 {@link com.bizunited.empower.business.payment.common.enums.ReceiptStatus}
   */
  private final String RECEIPT_STATUS = "receiptStatus";
  /**
   * 单据开始时间
   */
  private final String START_TIME = "startTime";
  /**
   * 单据结束时间
   */
  private final String END_TIME = "endTime";
  /**
   * 应收账款编号
   */
  private final String RECEIPT_CODE = "receiptCode";
  /** 创建人（业务员） */
  private final String CREATE_ACCOUNT = "account";

  @Autowired
  @PersistenceContext
  private EntityManager entityManager;

  @SuppressWarnings("unchecked")
  @Override
  public Page<Object> queryPage(Pageable pageable, InvokeParams conditions) {
    StringBuilder sql = new StringBuilder("select c.id from receipt_info c where 1=1");
    StringBuilder countSql = new StringBuilder("select count(*) FROM receipt_info c where 1=1");
    StringBuilder condition = new StringBuilder();
    Map<String, Object> parameters = new HashMap<>(6);
    Map<String, Object> params = conditions.getInvokeParams();
    String tenantCode = (String) params.get(TENANT_CODE);
    String receiptStatus = (String) params.get(RECEIPT_STATUS);
    String startTime = (String) params.get(START_TIME);
    String endTime = (String) params.get(END_TIME);
    String receiptCode = (String) params.get(RECEIPT_CODE);
    String account = (String) params.get(CREATE_ACCOUNT);
    String customerCode = (String) params.get(CUSTOMER_CODE);

    Validate.notBlank(tenantCode, "租户编号为空");
    condition.append(" and c.tenant_code=:tenantCode ");
    parameters.put("tenantCode", tenantCode);
    if (conditions != null) {
      if (StringUtils.isNotBlank(account)) {
        condition.append(" and c.create_account=:account");
        parameters.put("account", account);
      }
      if (StringUtils.isNotBlank(customerCode)) {
        condition.append(" and c.customer_code=:customerCode");
        parameters.put("customerCode", customerCode);
      }
      if (StringUtils.isNotBlank(receiptStatus)) {
        condition.append(" and c.receipt_status=:receiptStatus");
        parameters.put("receiptStatus", Integer.parseInt(receiptStatus));
      }
      if (StringUtils.isNotBlank(startTime)) {
        condition.append(" and c.receipt_time >=:startTime");
        parameters.put("startTime", parseDate(startTime));
      }
      if (StringUtils.isNotBlank(endTime)) {
        condition.append(" and c.receipt_time <=:endTime");
        parameters.put("endTime", parseDate(endTime));
      }
      if (StringUtils.isNotBlank(receiptCode)) {
        condition.append(" and c.receipt_code like concat(:receiptCode,'%')");
        parameters.put("receiptCode", receiptCode);
      }
    }
    sql.append(condition).append(" order by c.receipt_time desc");
    countSql.append(condition);

    Query query = entityManager.createNativeQuery(sql.toString());
    Query countQuery = entityManager.createNativeQuery(countSql.toString());
    parameters.forEach((k, v) -> {
      query.setParameter(k, v);
      countQuery.setParameter(k, v);
    });

    // 构造分页信息
    query.setFirstResult(pageable.getPageNumber() * pageable.getPageSize());
    query.setMaxResults(pageable.getPageSize());
    List<Object> resutls = query.getResultList();
    // 查询总数量信息
    Number count = (Number) countQuery.getResultList().get(0);
    return new PageImpl<>(resutls, pageable, count == null ? 0L : count.longValue());
  }

  /**
   * 日期格式转换
   * @param time
   * @return
   */
  private Date parseDate(String time) {
    SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    Date date = null;
    try {
      date = df.parse(time);
    } catch (ParseException e) {
      throw new IllegalArgumentException("日期格式转换错误", e);
    }
    return date;
  }
}
