package com.bizunited.empower.business.payment.repository.internal;

import com.bizunited.empower.business.payment.vo.AssociatedReceivableVo;
import com.bizunited.platform.common.repository.PageRepositoryImpl;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * ReceivableInfo模型的数据层自定义接口实现，可以由程序员根据实际情况完善
 *
 * @author saturn
 */
public class ReceivableInfoRepositoryImpl implements ReceivableInfoRepositoryCustom, PageRepositoryImpl {
  /**
   * 租户编码key
   */
  private final String TENANT_CODE = "tenantCode";
  /**
   * 应收账款状态 {@link com.bizunited.empower.business.payment.common.enums.ReceivableStatus}
   */
  private final String RECEIVABLE_STATUS = "receivableStatus";
  /**
   * 单据开始时间
   */
  private final String START_TIME = "startTime";
  /**
   * 单据结束时间
   */
  private final String END_TIME = "endTime";
  /**
   * 应收账款编号
   */
  private final String RECEIVABLE_CODE = "receivableCode";

  /** 客户编号 */
  private final String CUSTOMER_CODE = "customerCode";
  /** 客户名称 */
  private final String CUSTOMER_NAME = "customerName";
  /** 最小订单金额 */
  private final String MIN_ORDER_AMOUNT = "minOrderAmount";
  /** 最大订单金额 */
  private final String MAX_ORDER_AMOUNT = "maxOrderAmount";
  /** 创建人（业务员） */
  private final String CREATE_ACCOUNT = "account";
  /** 关联的订单编号 */
  private final String ASSOCIATE_CODE = "associatedCode";

  @Autowired
  @PersistenceContext
  private EntityManager entityManager;

  @Override
  public Page<AssociatedReceivableVo> queryPageForAssociated(Pageable pageable, InvokeParams conditions) {
    StringBuilder hql = new StringBuilder("from ReceivableInfo c where 1=1 and c.tstatus=1");
    StringBuilder countHql = new StringBuilder("select count(*) FROM ReceivableInfo c where 1=1 and c.tstatus=1");
    StringBuilder condition = new StringBuilder();
    Map<String, Object> parameters = new HashMap<>();

    Map<String, Object> params = conditions.getInvokeParams();
    String tenantCode = (String) params.get(TENANT_CODE);
    String receivableStatus = (String) params.get(RECEIVABLE_STATUS);
    String startTime = (String) params.get(START_TIME);
    String endTime = (String) params.get(END_TIME);
    String receivableCode = (String) params.get(RECEIVABLE_CODE);

    String customerCode = (String) params.get(CUSTOMER_CODE);
    String customerName = (String) params.get(CUSTOMER_NAME);
    String minOrderAmount = (String) params.get(MIN_ORDER_AMOUNT);
    String maxOrderAmount = (String) params.get(MAX_ORDER_AMOUNT);
    String account = (String) params.get(CREATE_ACCOUNT);
    String associatedCode = (String) params.get(ASSOCIATE_CODE);


    Validate.notBlank(tenantCode, "租户编号为空");

    condition.append(" and c.tenantCode=:tenantCode");
    parameters.put("tenantCode", tenantCode);

    if (conditions != null) {
      if (StringUtils.isNotBlank(account)) {
        condition.append(" and c.createAccount=:account");
        parameters.put("account", account);
      }
      if (StringUtils.isNotBlank(associatedCode)) {
        condition.append(" and c.associatedCode =:associatedCode");
        parameters.put("associatedCode", associatedCode);
      }
      if (StringUtils.isNotBlank(minOrderAmount)) {
        condition.append(" and c.receivableAmount>=:minOrderAmount");
        parameters.put("minOrderAmount", new BigDecimal(minOrderAmount));
      }
      if (StringUtils.isNotBlank(maxOrderAmount)) {
        condition.append(" and c.receivableAmount <=:maxOrderAmount");
        parameters.put("maxOrderAmount", new BigDecimal(maxOrderAmount));
      }

      if (StringUtils.isNotBlank(customerCode)) {
        condition.append(" and c.customerCode=:customerCode");
        parameters.put("customerCode", customerCode);
      }
      if (StringUtils.isNotBlank(customerName)) {
        condition.append(" and c.customerName like concat('%',:customerName,'%')");
        parameters.put("customerName", customerName);
      }
      if (StringUtils.isNotBlank(receivableStatus)) {
        condition.append(" and c.receivableStatus=:receivableStatus");
        parameters.put("receivableStatus", Integer.parseInt(receivableStatus));
      }
      if (StringUtils.isNotBlank(startTime)) {
        condition.append(" and c.createTime >=:startTime");
        parameters.put("startTime", parseDate(startTime));
      }
      if (StringUtils.isNotBlank(endTime)) {
        condition.append(" and c.createTime <=:endTime");
        parameters.put("endTime", parseDate(endTime));
      }
      if (StringUtils.isNotBlank(receivableCode)) {
        condition.append(" and c.receivableCode like concat(:receivableCode,'%')");
        parameters.put("receivableCode", receivableCode);
      }
    }
    hql.append(condition).append(" order by c.createTime desc");
    countHql.append(condition);
    return queryByConditions(entityManager, hql.toString(), countHql.toString(), parameters, pageable, false, AssociatedReceivableVo.class);
  }

  @SuppressWarnings("unchecked")
  @Override
  public Page<Object[]> queryPageForCustomer(Pageable pageable, InvokeParams conditions) {
    StringBuilder sql = new StringBuilder("select c.customer_code as customerCode,c.customer_name as customerName,cl.level_name as customerLevel,cc.`name` as customerCategory,sa.sales_area_name as salesName,count(c.receivable_code) as receivableNum,sum(c.receivable_amount) as receivableAmount,sum(c.received_amount) as receivedAmount,sum(c.wait_receive_amount) as waitReceiveAmount,sum(c.wait_confirm_amount) as waitConfirmAmount,c.id");
    StringBuilder countSql = new StringBuilder("select count(c.customer_code) ");
    StringBuilder joinSql = new StringBuilder();
    joinSql.append(" from receivable_info c ");
    joinSql.append(" left join customer as cu on c.customer_code=cu.customer_code and cu.tenant_code=c.tenant_code ");
    joinSql.append(" left join customer_level cl on cu.customer_level=cl.id ");
    joinSql.append(" left join customer_category cc on cc.id=cu.customer_category ");
    joinSql.append(" left join sales_area sa on sa.id=cu.sales_area_id ");

    StringBuilder condition = new StringBuilder();
    Map<String, Object> parameters = new HashMap<>();

    Map<String, Object> params = conditions.getInvokeParams();
    String tenantCode = (String) params.get(TENANT_CODE);
    String receivableStatus = (String) params.get(RECEIVABLE_STATUS);
    String startTime = (String) params.get(START_TIME);
    String endTime = (String) params.get(END_TIME);
    String receivableCode = (String) params.get(RECEIVABLE_CODE);

    String customerCode = (String) params.get(CUSTOMER_CODE);
    String customerName = (String) params.get(CUSTOMER_NAME);
    String minOrderAmount = (String) params.get(MIN_ORDER_AMOUNT);
    String maxOrderAmount = (String) params.get(MAX_ORDER_AMOUNT);
    String account = (String) params.get(CREATE_ACCOUNT);
    String associatedCode = (String) params.get(ASSOCIATE_CODE);

    Validate.notBlank(tenantCode, "租户编号为空");
    condition.append("where 1=1 and c.tstatus=1 ");

    condition.append("and c.tenant_code=:tenantCode  ");
    parameters.put("tenantCode", tenantCode);

    if (conditions != null) {
      if (StringUtils.isNotBlank(account)) {
        condition.append(" and c.create_account=:account");
        parameters.put("account", account);
      }
      if (StringUtils.isNotBlank(associatedCode)) {
        condition.append(" and c.associated_code =:associatedCode");
        parameters.put("associatedCode", associatedCode);
      }
      if (StringUtils.isNotBlank(minOrderAmount)) {
        condition.append(" and c.receivable_amount>=:minOrderAmount");
        parameters.put("minOrderAmount", new BigDecimal(minOrderAmount));
      }
      if (StringUtils.isNotBlank(maxOrderAmount)) {
        condition.append(" and c.receivable_amount <=:maxOrderAmount");
        parameters.put("maxOrderAmount", new BigDecimal(maxOrderAmount));
      }

      if (StringUtils.isNotBlank(customerCode)) {
        condition.append(" and c.customer_code=:customerCode");
        parameters.put("customerCode", customerCode);
      }
      if (StringUtils.isNotBlank(customerName)) {
        condition.append(" and c.customer_name like concat('%',:customerName,'%')");
        parameters.put("customerName", customerName);
      }
      if (StringUtils.isNotBlank(receivableStatus)) {
        condition.append(" and c.receivable_status=:receivableStatus");
        parameters.put("receivableStatus", Integer.parseInt(receivableStatus));
      }
      if (StringUtils.isNotBlank(startTime)) {
        condition.append(" and c.create_time >=:startTime");
        parameters.put("startTime", parseDate(startTime));
      }
      if (StringUtils.isNotBlank(endTime)) {
        condition.append(" and c.create_time <=:endTime");
        parameters.put("endTime", parseDate(endTime));
      }
      if (StringUtils.isNotBlank(receivableCode)) {
        condition.append(" and c.receivable_code like concat(:receivableCode,'%')");
        parameters.put("receivableCode", receivableCode);
      }
    }
    sql.append(joinSql).append(condition).append(" group by c.customer_code,c.customer_name,cl.level_name,cc.`name`,sa.sales_area_name");
    countSql.append(joinSql).append(condition).append(" group by c.customer_code");

    Query query = entityManager.createNativeQuery(sql.toString());
    Query countQuery = entityManager.createNativeQuery(countSql.toString());
    parameters.forEach((k, v) -> {
      query.setParameter(k, v);
      countQuery.setParameter(k, v);
    });

    // 构造分页信息
    query.setFirstResult(pageable.getPageNumber() * pageable.getPageSize());
    query.setMaxResults(pageable.getPageSize());
    List<Object[]> resutls = query.getResultList();
    // 查询总数量信息
    Number count = null;
    if (!CollectionUtils.isEmpty(countQuery.getResultList())){
      count = (Number) countQuery.getResultList().get(0);
    }
    return new PageImpl<>(resutls, pageable, count == null ? 0L : count.longValue());
  }

  /**
   * 日期格式转换
   *
   * @param time
   * @return
   */
  private Date parseDate(String time) {
    SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    Date date = null;
    try {
      date = df.parse(time);
    } catch (ParseException e) {
      throw new IllegalArgumentException("日期格式转换错误", e);
    }
    return date;
  }
}
