package com.bizunited.empower.business.payment.service;


import com.bizunited.empower.business.payment.dto.CustomerCreditDto;
import com.bizunited.empower.business.payment.entity.CustomerCredit;
import com.bizunited.empower.business.payment.vo.CustomerCreditVo;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.math.BigDecimal;
import java.util.List;

/**
 * CustomerCredit业务模型的服务层接口定义
 *
 * @author saturn paul
 */
public interface CustomerCreditService {

  /**
   * 初始化客户的信用信息
   *
   * @param customerCode
   * @param initialAmount
   * @param remark
   * @return
   */
  CustomerCredit init(String customerCode, BigDecimal initialAmount, String remark);

  /**
   * 初始化客户的信用信息
   *
   * @param customerCreditDtos
   * @return
   */
  List<CustomerCredit> init(List<CustomerCreditDto> customerCreditDtos);

  /**
   * 更新客户信用总额度
   *
   * @param customerCode
   * @param amount
   * @param remark
   * @return
   */
  CustomerCredit updateAmount(String customerCode, BigDecimal amount, String remark);

  /**
   * 更新客户信用状态
   * 正常：1、禁用：0
   *
   * @param customerCode
   * @param tstatis
   * @return
   */
  CustomerCredit updateStatis(String customerCode,Integer tstatis);

  /**
   * 该接口功能负责生成基于“授信额度”的应收账款以及相应的收款凭证，在此基础上付款中心中该经销商的信用额度将被立即扣减。
   * 注意：如果信用额度不足，则该方法会抛出异常
   *
   * @param customerCode 指定的经销商客户业务编号
   * @param amount       需要减少的信用额度
   * @param businessNo   业务单据的业务编号，需要保持唯一性
   * @param remark       备注信息
   * @return
   */
  CustomerCredit receipt(String customerCode, BigDecimal amount, String businessNo, String remark);

  /**
   * 该接口功能负责验证在扣减信用额度前对信用额度支付的额度校验，如果信用额度不足则返回false，能够正常扣减信用则返回true
   *
   * @param customerCode 指定的经销商客户业务编号
   * @param amount       需要减少的信用额度
   * @return
   */
  boolean validReceipt(String customerCode, BigDecimal amount);

  /**
   * 恢复客户信用的额度
   *
   * @param customerCode 指定的经销商客户业务编号
   * @param amount       需要恢复的信用额度
   * @param businessNo   业务单据的业务编号，需要保持唯一性
   * @param remark       备注信息
   * @return
   */
  CustomerCredit recover(String customerCode, BigDecimal amount, String businessNo, String remark);

  /**
   * 根据客户编码获取信用信息, 如果客户没有信用信息，则返回一个null
   *
   * @param customerCode
   * @return
   */
  CustomerCredit findByCustomerCode(String customerCode);

  /**
   * 按照查询条件进行用户信用的查询
   *
   * @param pageable
   * @param conditions
   */
  Page<CustomerCreditVo> findByConditions(Pageable pageable, InvokeParams conditions);

}