package com.bizunited.empower.business.payment.service;

import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.entity.DefrayInfo;
import com.bizunited.empower.business.payment.vo.DefrayInfoVo;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * DefrayInfo业务模型的服务层接口定义
 *
 * @author saturn
 */
public interface DefrayInfoService {
  /**
   * 创建一个新的DefrayInfo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）
   */
  DefrayInfo create(DefrayInfo defrayInfo);

  /**
   * 创建一个新的ReceiptInfo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）(无需确认收款)
   */
  DefrayInfo createWithoutConfirm(DefrayInfo defrayInfo);

  /**
   * 创建一个新的StudentEntity模型对象
   * 该代码由satrun骨架生成，默认不包括任何可能第三方系统调用、任何复杂逻辑处理等，主要应用场景为前端表单数据的暂存功能</br>
   * 该方法与本接口中的updateFrom方法呼应
   */
  DefrayInfo createForm(DefrayInfo defrayInfo);

  /**
   * 按照关联的 应付账款编号进行详情查询（包括关联信息）
   *
   * @param paymentInfo 关联的 应收账款编号
   */
  Set<DefrayInfo> findDetailsByPaymentInfo(String paymentInfo);

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  DefrayInfo findDetailsById(String id);

  /**
   * 按照DefrayInfo的主键编号，查询指定的数据信息（不包括任何关联信息）
   *
   * @param id 主键
   */
  DefrayInfo findById(String id);


  /**
   * 按照付款单流水编号进行查询
   *
   * @param defrayCode 付款单流水编号
   */
  DefrayInfo findDetailsByDefrayCode(String defrayCode);

  /**
   * 按照付款单流水编号进行查询
   *
   * @param defrayCode 付款单流水编号
   */
  DefrayInfo findByDefrayCode(String defrayCode);

  /**
   * 按照付款单编号，确认付款单金额
   *
   * @param defrayCode 付款单号
   */
  DefrayInfo confirm(String defrayCode);

  /**
   * 通过关联单据创建付款单
   *
   * @param associatedCode   关联单据号
   * @param amount           金额
   * @param fundsChannelType 资金渠道
   */
  DefrayInfo createByAssociatedCode(String associatedCode, BigDecimal amount, FundsChannelType fundsChannelType);

  /**
   * 取消付款单
   *
   * @param defrayCode
   * @return
   */
  DefrayInfo cancelByDefrayCode(String defrayCode);

  /**
   * 付款单查询列表
   *
   * @param pageable   分页参数
   * @param conditions 条件
   * @return
   */
  Page<DefrayInfoVo> findByConditions(Pageable pageable, InvokeParams conditions);

  /**
   * 按人员和指定日期，统计付款总金额
   *
   * @param account    创建人账号(付款人业务员账号)
   * @param defrayTime 付款日期
   */
  BigDecimal sumTotalDefrayAmountByAccountAndDefrayTime(String account, Date defrayTime);

  /**
   * 按人员和指定日期，统计付款总单数
   *
   * @param account    创建人账号(付款人业务员账号)
   * @param defrayTime 付款日期
   */
  BigInteger countTotalDefrayNumByAccountAndDefrayTime(String account, Date defrayTime);

  /**
   * 根据筛选条件查询已付金额统计
   *
   * @param pageable
   * @param conditions
   * @return
   */
  BigDecimal findTotalByConditions(Pageable pageable, InvokeParams conditions);

  /**
   * 根据付款单号查询付款单集合
   *
   * @param defrayCodes
   * @return
   */
  List<DefrayInfo> findByDefrayCodes(List<String> defrayCodes);
}