package com.bizunited.empower.business.payment.service;

import com.bizunited.empower.business.payment.common.enums.ElAccountBillBusinessType;
import com.bizunited.empower.business.payment.common.enums.ElAccountBillChargeRateType;
import com.bizunited.empower.business.payment.common.enums.ElAccountBillState;
import com.bizunited.empower.business.payment.dto.TransferNoticeDto;
import com.bizunited.empower.business.payment.entity.ElectronicAccountBill;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 电子账户流水记录服务
 *
 * @Author: Paul Chan
 * @Date: 2021/4/20 上午11:47
 */
public interface ElectronicAccountBillService {

  /**
   * 创建流水单据
   *
   * @param accountId    电子账户ID
   * @param state        流水状态，0：处理中 1：成功，2：失败
   * @param businessType 业务类型，1：提现、2：应收、3：退货退款
   * @param businessNo   业务单号
   * @param amount       交易金额
   * @param chargeRate   手续费率
   * @param realAmount   实际收支金额
   * @param dealFrom     交易来源
   * @return
   */
  ElectronicAccountBill create(String accountId, ElAccountBillState state, ElAccountBillBusinessType businessType, String businessNo, BigDecimal amount,
                               ElAccountBillChargeRateType chargeRateType, BigDecimal chargeRate, BigDecimal chargeAmount, BigDecimal realAmount, String dealFrom, String remark);

  /**
   * 根据业务单号更新流水状态,一个业务单号可能会找到多个流水单
   * 如果要更精准的更新流水单的状态，请使用方法：updateStateByBillNo
   *
   * @param businessNo
   * @param state
   * @return
   */
  List<ElectronicAccountBill> updateStateByBusinessNo(String businessNo, ElAccountBillState state);

  /**
   * 根据流水号号更新流水状态
   *
   * @param billNo
   * @param state
   * @return
   */
  ElectronicAccountBill updateStateByBillNo(String billNo, ElAccountBillState state);

  /**
   * 处理提现mq通知
   * @param transferNoticeDto
   */
  void handleExtractCashMq(TransferNoticeDto transferNoticeDto);

  /**
   * 多条件分页查询
   *
   * @param conditions
   * @param pageable
   * @return
   */
  Page<ElectronicAccountBill> findByConditions(Map<String, Object> conditions, Pageable pageable);

  /**
   * 根据类型统计金额
   *
   * @param type
   * @param startDate
   * @param endDate
   * @return
   */
  BigDecimal sumRealAmountByTypeAndModifyTimeBetween(Integer type, Date startDate, Date endDate);

  /**
   * 根据类型统计金额
   *
   * @param accountId
   * @param type
   * @param startDate
   * @param endDate
   * @return
   */
  BigDecimal sumRealAmountByAccountIdAndTypeAndModifyTimeBetween(String accountId, Integer type, Date startDate, Date endDate);
}
