package com.bizunited.empower.business.payment.service;

import com.bizunited.empower.business.payment.entity.ElectronicAccount;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

/**
 * ElectronicAccount业务模型的服务层接口定义
 *
 * @author Paul Chan
 */
public interface ElectronicAccountService {
  /**
   * 创建一个新的ElectronicAccount模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）
   */
  ElectronicAccount create(ElectronicAccount electronicAccount);

  /**
   * 创建一个新的StudentEntity模型对象
   * 该代码由satrun骨架生成，默认不包括任何可能第三方系统调用、任何复杂逻辑处理等，主要应用场景为前端表单数据的暂存功能</br>
   * 该方法与本接口中的updateFrom方法呼应
   */
  ElectronicAccount createForm(ElectronicAccount electronicAccount);

  /**
   * 更新一个已有的ElectronicAccount模型对象，其主键属性必须有值(1.1.4-release版本调整)。
   * 这个方法实际上一共分为三个步骤（默认）：</br>
   * 1、调用updateValidation方法完成表单数据更新前的验证</br>
   * 2、调用updateForm方法完成表单数据的更新</br>
   * 3、完成开发人员自行在本update方法中书写的，进行第三方系统调用（或特殊处理过程）的执行。</br>
   * 这样做的目的，实际上是为了保证updateForm方法中纯粹是处理表单数据的，在数据恢复表单引擎默认调用updateForm方法时，不会影响任何第三方业务数据
   * （当然，如果系统有特别要求，可由开发人员自行完成代码逻辑调整）
   */
  ElectronicAccount update(ElectronicAccount electronicAccount);

  /**
   * 该方法只用于处理业务表单信息，包括了主业务模型、其下的关联模型、分组信息和明细细信息等
   * 该方法非常重要，因为如果进行静态表单的数据恢复，那么表单引擎将默认调用主业务模型（服务层）的这个方法。</br>
   * 这样一来保证了数据恢复时，不会涉及任何第三方系统的调用（当然，如果开发人员需要涉及的，可以自行进行修改）
   */
  ElectronicAccount updateForm(ElectronicAccount electronicAccount);

  /**
   * 更新绑卡状态
   *
   * @param electronicAccountId
   * @param tiedCardStatus
   */
  void updateTiedCardStatusById(String electronicAccountId, Integer tiedCardStatus);

  /**
   * 更新申请状态
   *
   * @param id
   * @param applyState
   * @param failureReason
   */
  void updateApplyStateById(String id, Integer applyState, String failureReason);

  /**
   * 根据ID更新状态
   *
   * @param id
   * @param state
   */
  void updateStateById(String id, Integer state);

  /**
   * 更新电子账户的时间
   *
   * @param id
   * @param preApplyTime      预审核时间
   * @param repeatedApplyTime 复审核时间
   * @param signTime          签约时间
   * @param openTime          开通时间
   * @param expireTime        过期时间
   */
  void updateTimesById(String id, Date preApplyTime, Date repeatedApplyTime, Date signTime, Date openTime, Date expireTime);

  /**
   * 修改是否显示余额
   *
   * @param showBalance
   */
  void updateShowBalance(Boolean showBalance);

  /**
   * 更新电子账户的商户信息
   *
   * @param id
   * @param merchantCode
   * @param merchantName
   */
  void updateMerchantInfoById(String id, String merchantCode, String merchantName, String billingCycle);

  /**
   * 更新账户绑定的唯一标识
   *
   * @param id
   * @param userId
   */
  void updateUserIdById(String id, String userId);

  /**
   * 收入,增加电子账户余额
   *
   * @param amount
   * @param chargeRate
   * @param chargeAmount
   * @param realAmount
   * @param businessNo
   * @param remark
   */
  void receipt(BigDecimal amount, BigDecimal chargeRate, BigDecimal chargeAmount, BigDecimal realAmount, String businessNo, String remark, String dealFrom);

  /**
   * 提现
   *
   * @param amount
   * @param validCode
   */
  void withdraw(BigDecimal amount, String validCode);

  /**
   * 给电子账户添加余额
   *
   * @param id
   * @param realAmount
   */
  void addBalanceByAccountId(String id, BigDecimal realAmount);

  /**
   * 给电子账户安全手机号发送手机验证码
   *
   * @param mobile       手机号码，只有当短信业务类型为[4,5]时需要传入
   * @param businessType
   */
  void sendValidCode(String mobile, Integer businessType);

  /**
   * 验证手机验证码
   *
   * @param mobile
   * @param validCode
   * @param businessType
   */
  void verifyValidCode(String mobile, String validCode, Integer businessType);

  /**
   * 绑定账号安全手机号码
   *
   * @param mobile
   * @param validCode
   */
  void bindSecurityMobile(String mobile, String validCode);

  /**
   * 重新绑定账号安全手机号码
   *
   * @param mobile
   * @param oldMobileValidCode
   * @param newMobileValidCode
   */
  void rebindSecurityMobile(String mobile, String oldMobileValidCode, String newMobileValidCode);

  /**
   * 获取签约验证码
   *
   * @param isUpdate
   * @param name
   * @param idCardNo
   * @param mobile
   */
  void sendSignValidCode(Boolean isUpdate, String name, String idCardNo, String mobile);

  /**
   * 获取经销商的签约地址
   *
   * @param validCode 验证码
   * @return
   */
  String getSignUrlByTenant(String validCode);

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  ElectronicAccount findDetailsById(String id);

  /**
   * 按照ElectronicAccount的主键编号，查询指定的数据信息（不包括任何关联信息）
   *
   * @param id 主键
   */
  ElectronicAccount findById(String id);

  /**
   * 获取当前经销商的电子账户资料详情
   *
   * @return
   */
  ElectronicAccount findDetailsByTenant();

  /**
   * 获取当前经销商的电子账户
   *
   * @return
   */
  ElectronicAccount findByTenant();

  /**
   * 根据经销商编码获取
   *
   * @param tenantCode
   * @return
   */
  ElectronicAccount findByTenantCode(String tenantCode);

  /**
   * 获取所有电子账户信息
   *
   * @return
   */
  List<ElectronicAccount> findAll();

  /**
   * 通过账号查询电子账户
   *
   * @param merchantCode
   * @return
   */
  ElectronicAccount findByMerchantCode(String merchantCode);
}