package com.bizunited.empower.business.payment.service;

import com.bizunited.empower.business.payment.common.enums.PaymentType;
import com.bizunited.empower.business.payment.entity.DefrayInfo;
import com.bizunited.empower.business.payment.entity.PaymentInfo;
import com.bizunited.empower.business.payment.vo.AssociatedPaymentVo;
import com.bizunited.empower.business.payment.vo.CustomerPaymentVo;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.math.BigDecimal;

/**
 * PaymentInfo业务模型的服务层接口定义
 *
 * @author saturn
 */
public interface PaymentInfoService {

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  PaymentInfo findDetailsById(String id);

  /**
   * 按照ReceivableInfo的主键编号，查询指定的数据信息（不包括任何关联信息）
   *
   * @param id 主键
   */
  PaymentInfo findById(String id);

  /**
   * 按照应付账款流水编号进行查询
   *
   * @param paymentCode 应付账款流水编号
   */
  PaymentInfo findByPaymentCode(String paymentCode);

  /**
   * 按照关联单编号进行查询
   *
   * @param associatedCode 关联单
   */
  PaymentInfo findByAssociatedCode(String associatedCode);

  /**
   * 根据订单创建应付账款
   *
   * @param customerCode   客户编号
   * @param associatedCode 关联单据号
   * @param amount         金额
   * @param paymentType    付款类型 {@link PaymentType}
   */
  PaymentInfo createByAssociatedCode(String customerCode, String associatedCode, BigDecimal amount, PaymentType paymentType);

  /**
   * 根据客户编号查看客户应付账款数据
   * @param customerCode
   * @return
   */
  CustomerPaymentVo findAggregationByCustomerCode(String customerCode);

  /**
   * 根据订单修改应付账款信息
   *
   * @param associatedCode 关联单据号
   * @param amount         金额
   * @return
   */
  PaymentInfo updateByAssociatedCode(String associatedCode, BigDecimal amount);

  /**
   * 根据关联单号取消应付账款
   *
   * @param associatedCode 关联单据号
   */
  PaymentInfo cancelByAssociatedCode(String associatedCode);

  /**
   * 应付账款查询列表 (按照关联单据)
   *
   * @param pageable
   * @param conditions
   */
  Page<AssociatedPaymentVo> findByConditionsForAssociated(Pageable pageable, InvokeParams conditions);

  /**
   * 应付账款查询列表 (按照客户)
   *
   * @param pageable
   * @param conditions
   */
  Page<CustomerPaymentVo> findByConditionsForCustomer(Pageable pageable, InvokeParams conditions);

  /**
   * 待确认付款（线下支付）
   *
   * @param paymentCode 应付账款单号
   * @param amount      金额
   * @return
   */
  PaymentInfo waitPay(String paymentCode, BigDecimal amount);

  /**
   * 取消待确认付款金额
   *
   * @param paymentCode
   * @param amount
   * @return
   */
  PaymentInfo cancelWaitPay(String paymentCode, BigDecimal amount);

  /**
   * 确认付款（线下支付）
   *
   * @param paymentCode 应付账款单号
   * @param amount      金额
   * @param defrayInfo  付款单号
   * @return
   */
  PaymentInfo confirmPay(String paymentCode, BigDecimal amount, DefrayInfo defrayInfo);

  /**
   * 直接付款（信用支付/余额支付）
   *
   * @param paymentCode 应付账款单号
   * @param amount         金额
   * @param defrayInfo    收款单号
   * @return
   */
  PaymentInfo give(String paymentCode, BigDecimal amount, DefrayInfo defrayInfo);

  /**
   * 统计应付金额
   *
   * @return
   */
  BigDecimal sumWaitPayAmount();

}