package com.bizunited.empower.business.payment.service;

import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.dto.ReceiptDto;
import com.bizunited.empower.business.payment.dto.TransferNoticeDto;
import com.bizunited.empower.business.payment.entity.ReceiptInfo;
import com.bizunited.empower.business.payment.vo.ReceiptInfoVo;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * ReceiptInfo业务模型的服务层接口定义
 *
 * @author saturn
 */
public interface ReceiptInfoService {
  /**
   * 创建一个新的ReceiptInfo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）
   */
  ReceiptInfo create(ReceiptInfo receiptInfo);

  /**
   * 创建一个新的ReceiptInfo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）
   */
  ReceiptInfo create(ReceiptInfo receiptInfo, InvokeParams params);

  /**
   * 创建一个新的ReceiptInfo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）(无需确认收款)
   */
  ReceiptInfo createWithoutConfirm(ReceiptInfo receiptInfo);

  /**
   * 创建一个新的StudentEntity模型对象
   * 该代码由satrun骨架生成，默认不包括任何可能第三方系统调用、任何复杂逻辑处理等，主要应用场景为前端表单数据的暂存功能</br>
   * 该方法与本接口中的updateFrom方法呼应
   */
  ReceiptInfo createForm(ReceiptInfo receiptInfo);

  /**
   * 创建一个新的StudentEntity模型对象
   * 该代码由satrun骨架生成，默认不包括任何可能第三方系统调用、任何复杂逻辑处理等，主要应用场景为前端表单数据的暂存功能</br>
   * 该方法与本接口中的updateFrom方法呼应
   */
  ReceiptInfo createForm(ReceiptInfo receiptInfo, InvokeParams params);

  /**
   * 按照关联的 应收账款编号进行详情查询（包括关联信息）
   *
   * @param receivableInfo 关联的 应收账款编号
   */
  Set<ReceiptInfo> findDetailsByReceivableInfo(String receivableInfo);

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  ReceiptInfo findDetailsById(String id);

  /**
   * 按照ReceiptInfo的主键编号，查询指定的数据信息（不包括任何关联信息）
   *
   * @param id 主键
   */
  ReceiptInfo findById(String id);


  /**
   * 按照收款单流水编号进行查询
   *
   * @param receiptCode 收款单流水编号
   */
  ReceiptInfo findByReceiptCode(String receiptCode);

  /**
   * 按照收款单流水编号进行查询明细
   *
   * @param receiptCode 收款单流水编号
   */
  ReceiptInfo findDetailsByReceiptCode(String receiptCode);

  /**
   * 按照收款单号再次支付
   *
   * @param receiptCode
   * @param fundsChannel
   * @return
   */
  ReceiptInfoVo redoByReceiptCode(String receiptCode, Integer fundsChannel, InvokeParams params);

  /**
   * 按照收款单编号，确认收款单金额
   *
   * @param receiptCode 收款单号
   */
  ReceiptInfo confirm(String receiptCode);

  /**
   * 通过关联单据创建收款单
   *
   * @param associatedCode   关联单据号
   * @param amount           金额
   * @param fundsChannelType 资金渠道
   * @param params           扩展参数
   */
  ReceiptInfoVo createByAssociatedCode(String associatedCode, BigDecimal amount, FundsChannelType fundsChannelType, InvokeParams params);

  /**
   * 取消收款单
   *
   * @param receiptCode
   * @return
   */
  ReceiptInfo cancelByReceiptCode(String receiptCode);

  /**
   * 失败的收款单
   *
   * @param receiptCode
   * @return
   */
  ReceiptInfo failedByReceiptCode(String receiptCode);

  /**
   * 收款单查询列表
   *
   * @param pageable   分页参数
   * @param conditions 条件
   * @return
   */
  Page<ReceiptInfoVo> findByConditions(Pageable pageable, InvokeParams conditions);

  /**
   * 按人员和指定日期，统计收款总金额
   *
   * @param account     创建人账号(收款人业务员账号)
   * @param receiptTime 收款日期
   */
  BigDecimal sumTotalReceiptAmountByAccountAndReceiptTime(String account, Date receiptTime);

  /**
   * 按人员和指定日期，统计收款总单数
   *
   * @param account     创建人账号(收款人业务员账号)
   * @param receiptTime 收款日期
   */
  BigInteger countTotalReceiptNumByAccountAndReceiptTime(String account, Date receiptTime);

  /**
   * 通过参数dto创建ReceiptInfo模型对象
   */
  ReceiptInfoVo create(ReceiptDto receiptDto);

  /**
   * 处理转账业务
   *
   * @return
   */
  void handlerTransfer(TransferNoticeDto transferNoticeDto);
  
  /**
   * 根据筛选条件查询已收金额统计
   * @param pageable
   * @param conditions
   * @return
   */
  BigDecimal findTotalByConditions(Pageable pageable, InvokeParams conditions);

  /**
   * 根据客户编码以及资金渠道查询客户所有待支付的收款单
   * @param customerCode
   * @param fundsChannels
   * @return
   */
  List<ReceiptInfo> findWaitForByCustomerCodeAndFundsChannels(String customerCode, List<Integer> fundsChannels);

  /**
   * 根据客户编号创建应收账款以及收款单
   * @param customerCode
   * @param amount
   * @param fundsChannel
   * @param params
   * @return
   */
  ReceiptInfoVo createByCustomerCodeAndFundsChannels(String customerCode,BigDecimal amount, Integer fundsChannel, InvokeParams params);

  /**
   * 根据收款单号集合查询对应的收款单信息
   * @param receiptCodes
   * @return
   */
  List<ReceiptInfo> findByReceipts(List<String> receiptCodes);
}