package com.bizunited.empower.business.payment.service;

import com.bizunited.empower.business.payment.common.enums.ReceivableType;
import com.bizunited.empower.business.payment.entity.ReceiptInfo;
import com.bizunited.empower.business.payment.entity.ReceivableInfo;
import com.bizunited.empower.business.payment.vo.AssociatedReceivableVo;
import com.bizunited.empower.business.payment.vo.CustomerReceivableVo;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.math.BigDecimal;
import java.util.List;
import java.util.Set;

/**
 * ReceivableInfo业务模型的服务层接口定义
 *
 * @author saturn
 */
public interface ReceivableInfoService {

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  ReceivableInfo findDetailsById(String id);

  /**
   * 按照ReceivableInfo的主键编号，查询指定的数据信息（不包括任何关联信息）
   *
   * @param id 主键
   */
  ReceivableInfo findById(String id);

  /**
   * 按照应收账款流水编号进行查询
   *
   * @param receivableCode 应收账款流水编号
   */
  ReceivableInfo findByReceivableCode(String receivableCode);

  /**
   * 按照关联单编号进行查询
   *
   * @param associatedCode 关联单
   */
  ReceivableInfo findByAssociatedCode(String associatedCode);

  /**
   * 按照集合关联单编号进行查询
   * @param associatedCodes 关联单集合
   */
  List<ReceivableInfo> findByAssociatedCodes(Set<String> associatedCodes);

  /**
   * 根据订单创建应收账款
   *
   * @param customerCode   客户编号
   * @param associatedCode 关联单据号
   * @param amount         金额
   * @param receivableType 收款类型 {@link ReceivableType}
   */
  ReceivableInfo createByAssociatedCode(String customerCode, String associatedCode, BigDecimal amount, ReceivableType receivableType);

  /**
   * 根据客户创建应收账款
   * @param customerCode
   * @param amount
   * @param receivableType
   * @return
   */
  ReceivableInfo createByCustomerCode(String customerCode, BigDecimal amount, ReceivableType receivableType);
  /**
   * 根据订单修改应收账款信息
   *
   * @param associatedCode 关联单据号
   * @param amount         金额
   * @return
   */
  ReceivableInfo updateByAssociatedCode(String associatedCode, BigDecimal amount);

  /**
   * 根据关联单号取消应收账款
   *
   * @param associatedCode 关联单据号
   */
  ReceivableInfo cancelByAssociatedCode(String associatedCode);

  /**
   * 应收账款查询列表 (按照关联单据)
   *
   * @param pageable
   * @param conditions
   */
  Page<AssociatedReceivableVo> findByConditionsForAssociated(Pageable pageable, InvokeParams conditions);

  /**
   * 应收账款查询列表 (按照客户)
   *
   * @param pageable
   * @param conditions
   */
  Page<CustomerReceivableVo> findByConditionsForCustomer(Pageable pageable, InvokeParams conditions);

  /**
   * 根据客户编号查看客户应收账款数据
   * @param customerCode
   * @return
   */
  CustomerReceivableVo findAggregationByCustomerCode(String customerCode);

  /**
   * 待确认收款（线下支付）
   *
   * @param receivableCode 应收账款单号
   * @param amount         金额
   * @return
   */
  ReceivableInfo waitReceive(String receivableCode, BigDecimal amount);

  /**
   * 取消待确认收款金额
   *
   * @param receivableCode
   * @param amount
   * @return
   */
  ReceivableInfo cancelWaitReceive(String receivableCode, BigDecimal amount);

  /**
   * 确认收款（线下支付/小程序支付/扫码支付）
   *
   * @param receivableCode 应收账款单号
   * @param amount         金额
   * @param receiptInfo    收款单号
   * @return
   */
  ReceivableInfo confirmReceive(String receivableCode, BigDecimal amount, ReceiptInfo receiptInfo);

  /**
   * 直接收款（信用支付/余额支付/转账支付）
   *
   * @param receivableCode 应收账款单号
   * @param amount         金额
   * @param receiptInfo    收款单号
   * @return
   */
  ReceivableInfo receive(String receivableCode, BigDecimal amount, ReceiptInfo receiptInfo);

  /**
   * 统计待收金额
   * @return
   */
  BigDecimal sumWaitReceiveAmount();

  /**
   * 按照客户编号与租户编号查找应收账款
   * @param customerCode
   * @param tenantCode
   * @return
   */
  List<ReceivableInfo> findByCustomerAndTenantCode(String customerCode, String tenantCode);
}