package com.bizunited.empower.business.payment.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.empower.business.common.service.SignService;
import com.bizunited.empower.business.common.util.ResponseModelUtils;
import com.bizunited.empower.business.payment.entity.CustomerBank;
import com.bizunited.empower.business.payment.feign.BankCardFeignClient;
import com.bizunited.empower.business.payment.repository.CustomerBankRepository;
import com.bizunited.empower.business.payment.service.CustomerBankService;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * CustomerBank业务模型的服务层接口实现
 *
 * @author Keller
 */
@Service("CustomerBankServiceImpl")
public class CustomerBankServiceImpl implements CustomerBankService {
  @Autowired
  private CustomerBankRepository customerBankRepository;
  @Autowired
  private BankCardFeignClient bankCardFeignClient;
  @Autowired
  private SignService signService;

  @Override
  @Transactional
  public CustomerBank create(CustomerBank customerBank) {
    customerBank.setTenantCode(TenantUtils.getTenantCode());
    customerBank.setTstatus(1);
    Date now = new Date();
    customerBank.setCreateTime(now);
    customerBank.setModifyTime(now);
    // 验证参数
    this.createValidation(customerBank);
    customerBank = customerBankRepository.save(customerBank);
    JSONObject jsonObject = new JSONObject();
    jsonObject.put("cardNum", customerBank.getCardNumber());
    jsonObject.put("bankName", customerBank.getCardHolder());
    jsonObject.put("tenantCode", TenantUtils.getTenantCode());
    String json = JSONObject.toJSONString(jsonObject);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = bankCardFeignClient.addToWhiteList(appId, sign, json);
    ResponseModelUtils.validSuccess(responseModel, "用户添加失败，请重试");
    // 10=待审核 20=启用 30=失效
    String status = (String) responseModel.getData();
    Validate.isTrue("20".equals(status), "用户添加失败，请重试");
    return customerBank;
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    Validate.notBlank(id, "删除数据主键为空，请检查！");
    CustomerBank customerBank = customerBankRepository.findById(id).orElse(null);
    if (customerBank != null) {
      JSONObject jsonObject = new JSONObject();
      jsonObject.put("cardNum", customerBank.getCardNumber());
      jsonObject.put("bankName", customerBank.getCardHolder());
      jsonObject.put("tenantCode", TenantUtils.getTenantCode());
      String json = JSONObject.toJSONString(jsonObject);
      String appId = signService.getCurrentAppId();
      String sign = signService.sign(json);
      ResponseModel responseModel = bankCardFeignClient.removeFromWhiteList(appId, sign, json);
      ResponseModelUtils.validSuccess(responseModel, "用户注销失败，请重试");
      String status = (String) responseModel.getData();
      Validate.isTrue("30".equals(status), "用户注销失败，请重试");
      customerBank.setTstatus(0);
      customerBankRepository.save(customerBank);
    }
  }

  private void createValidation(CustomerBank customerBank) {
    Validate.notNull(customerBank, "添加数据不能为空，请检查！");
    Validate.notBlank(customerBank.getCustomerCode(), "客户编码不能为空，请检查!");
    Validate.notBlank(customerBank.getCardHolder(), "持卡人不能为空，请检查!");
    Validate.notBlank(customerBank.getCardNumber(), "银行卡密码不能为空，请检查!");
    Validate.notBlank(customerBank.getBankId(), "开户行不能为空，请检查!");
    Validate.notBlank(customerBank.getBankName(), "银行名称不能为空，请检查!");
    Validate.notBlank(customerBank.getCardType(), "银行卡类型不能为空，请检查!");
    CustomerBank current = customerBankRepository.findByTenantCodeAndCardNumberAndTstatus(customerBank.getTenantCode(), customerBank.getCardNumber(), 1);
    Validate.isTrue(current == null, "该银行卡已经被绑定，请换一张支付银行卡");
  }

  @Override
  public Page<CustomerBank> findByCustomerCode(String customerCode, Integer tstatus, Pageable pageable) {
    String tenantCode = TenantUtils.getTenantCode();
    Page<CustomerBank> page = null;
    if (tstatus != null) {
      page = customerBankRepository.findByTenantCodeAndCustomerCodeAndTstatus(tenantCode, customerCode, tstatus, pageable);
    } else {
      page = customerBankRepository.findByTenantCodeAndCustomerCode(tenantCode, customerCode, pageable);
    }
    List<CustomerBank> data = page.getContent();
    List<CustomerBank> customerBanks = Lists.newArrayList();
    if (data != null) {
      customerBanks = data.stream().map(cb -> {
        int length = StringUtils.length(cb.getCardNumber());
        // 脱敏操作
        cb.setCardNumber(StringUtils.leftPad(StringUtils.right(cb.getCardNumber(), length - 4), length, "*"));
        return cb;
      }).collect(Collectors.toList());
    }
    return new PageImpl<>(customerBanks, pageable, page.getTotalElements());
  }

  @Override
  public CustomerBank findByCardNumAndTenantCode(String cardNum, String tenantCode) {
    return customerBankRepository.findByTenantCodeAndCardNumberAndTstatus(tenantCode, cardNum, 1);
  }
}
