package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.payment.common.enums.CustomerCreditBillType;
import com.bizunited.empower.business.payment.entity.CustomerCredit;
import com.bizunited.empower.business.payment.entity.CustomerCreditBill;
import com.bizunited.empower.business.payment.repository.CustomerCreditBillRepository;
import com.bizunited.empower.business.payment.service.CustomerCreditBillService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.constant.Constants.CUSTOMER_CREDIT_BILL_NO_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.CUSTOMER_CREDIT_BILL_NO_KEY_PREFIX;
import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * CustomerCreditBill业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("CustomerCreditBillServiceImpl")
public class CustomerCreditBillServiceImpl implements CustomerCreditBillService {

  @Autowired
  private CustomerCreditBillRepository customerCreditBillRepository;

  @Autowired
  private RedisMutexService redisMutexService;

  @Override
  @Transactional
  public CustomerCreditBill create(CustomerCredit customerCredit, String businessNo, BigDecimal amount, BigDecimal balance, CustomerCreditBillType type, String remark) {
    Validate.notNull(type, "流水单类型不能为空");
    Date now = new Date();
    CustomerCreditBill bill = new CustomerCreditBill();
    bill.setCustomerCredit(customerCredit);
    bill.setBusinessNo(businessNo);
    bill.setAmount(amount);
    bill.setBalance(balance);
    bill.setType(type.getType());
    bill.setRemark(remark);
    bill.setCreateTime(now);
    bill.setModifyTime(now);
    bill.setTenantCode(TenantUtils.getTenantCode());
    this.createValidation(bill);
    bill.setBillNo(this.generateBillNo());
    customerCreditBillRepository.save(bill);
    return bill;
  }

  @Override
  public Page<CustomerCreditBill> findByCustomerCode(String customerCode, Pageable pageable) {
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    if (StringUtils.isBlank(customerCode)) {
      return Page.empty(pageable);
    }
    String tenantCode = TenantUtils.getTenantCode();
    return customerCreditBillRepository.findByTenantCodeAndCustomerCode(tenantCode, customerCode, pageable);
  }

  /**
   * 生成流水单单号
   *
   * @return
   */
  private String generateBillNo() {
    Date now = new Date();
    String tenantCode = TenantUtils.getTenantCode();
    SimpleDateFormat dayFormat = new SimpleDateFormat("yyyyMMdd");
    String redisKey = String.format(CUSTOMER_CREDIT_BILL_NO_KEY_PREFIX, tenantCode, dayFormat.format(now));
    String index = redisMutexService.getAndIncrement(redisKey, 1, 6, 25, TimeUnit.HOURS);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(CUSTOMER_CREDIT_BILL_NO_PREFIX, format.format(now), index);
  }

  /**
   * 创建前校验
   *
   * @param bill
   */
  private void createValidation(CustomerCreditBill bill) {
    Validate.notNull(bill, "流水单不能为空");
    CustomerCredit customerCredit = bill.getCustomerCredit();
    Validate.notNull(customerCredit, "客户信用不能为空");
    Validate.notBlank(customerCredit.getId(), "客户信用主键不能为空");
    Validate.notBlank(bill.getBusinessNo(), "业务单号不能为空");
    Validate.notNull(bill.getAmount(), "流水金额不能为空");
    Validate.notNull(bill.getType(), "流水单类型不能为空");
    Validate.notBlank(bill.getTenantCode(), "租户编码不能为空");
    CustomerCreditBillType type = CustomerCreditBillType.valueOfType(bill.getType());
    Validate.notNull(type, "非法的流水单类型：%s", bill.getType());
    switch (type) {
      case RECEIPT:
        Validate.isTrue(bill.getAmount().compareTo(BigDecimal.ZERO) < 0, "收款流水单的金额须为负数");
        break;
      case RECOVER:
        Validate.isTrue(bill.getAmount().compareTo(BigDecimal.ZERO) > 0, "恢复额度流水单的金额须为正数");
        break;
      default:
        break;
    }
    long count = customerCreditBillRepository.countByTenantCodeAndBusinessNo(bill.getTenantCode(), bill.getBusinessNo());
    Validate.isTrue(count == 0, "业务单号已存在，请更换单号");
  }

}
