package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.payment.dto.CustomerCreditDto;
import com.bizunited.empower.business.payment.entity.CustomerCredit;
import com.bizunited.empower.business.payment.repository.CustomerCreditRepository;
import com.bizunited.empower.business.payment.service.CustomerCreditBillService;
import com.bizunited.empower.business.payment.service.CustomerCreditRecordService;
import com.bizunited.empower.business.payment.service.CustomerCreditService;
import com.bizunited.empower.business.payment.service.notifier.CustomerCreditEventListener;
import com.bizunited.empower.business.payment.vo.CustomerCreditVo;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.enums.CustomerCreditBillType.RECEIPT;
import static com.bizunited.empower.business.payment.common.enums.CustomerCreditBillType.RECOVER;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.CUSTOMER_CREDIT_LOCK_PREFIX;

/**
 * CustomerCredit业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("CustomerCreditServiceImpl")
public class CustomerCreditServiceImpl implements CustomerCreditService {
  @Autowired
  private CustomerCreditRepository customerCreditRepository;

  @Autowired
  private CustomerService customerService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private CustomerCreditBillService customerCreditBillService;
  @Autowired
  private CustomerCreditRecordService customerCreditRecordService;
  @Autowired(required = false)
  private List<CustomerCreditEventListener> customerCreditEventListeners;

  @Override
  @Transactional
  public CustomerCredit init(String customerCode, BigDecimal initialAmount, String remark) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    Validate.notNull(initialAmount, "初始信用额度不能为空");
    Validate.isTrue(initialAmount.compareTo(BigDecimal.ZERO) >= 0, "初始信用额度不能是负数");
    String tenantCode = TenantUtils.getTenantCode();
    Customer customer = customerService.findDetailsByTenantCodeAndCustomerCode(tenantCode, customerCode);
    Validate.notNull(customer, "客户【%s】不存在， 请检查！", customerCode);
    String lockKey = String.format(CUSTOMER_CREDIT_LOCK_PREFIX, tenantCode, customerCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "操作频繁，请稍后再试！");
    try {
      CustomerCredit customerCredit = customerCreditRepository.findByTenantCodeAndCustomerCode(tenantCode, customerCode);
      Validate.isTrue(customerCredit == null, "客户信用已初始化!");
      remark = StringUtils.defaultIfBlank(remark, "初始化");
      Date now = new Date();
      String userAccount = SecurityUtils.getUserAccount();
      customerCredit = new CustomerCredit();
      customerCredit.setTenantCode(tenantCode);
      customerCredit.setCustomerCode(customerCode);
      customerCredit.setTotalAmount(initialAmount.setScale(4, RoundingMode.HALF_UP));
      customerCredit.setUsedAmount(BigDecimal.ZERO);
      customerCredit.setBalance(initialAmount.setScale(4, RoundingMode.HALF_UP));
      customerCredit.setTstatus(NormalStatusEnum.ENABLE.getStatus());
      customerCredit.setCreateTime(now);
      customerCredit.setModifyTime(now);
      customerCredit.setCreateAccount(userAccount);
      customerCredit.setModifyAccount(userAccount);
      customerCreditRepository.save(customerCredit);
      customerCreditRecordService.create(customerCredit, initialAmount, BigDecimal.ZERO, initialAmount, remark);
      return customerCredit;
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public List<CustomerCredit> init(List<CustomerCreditDto> customerCreditDtos) {
    Validate.isTrue(CollectionUtils.isNotEmpty(customerCreditDtos), "用户信用初始化参数为空，请检查！");
    List<CustomerCredit> results = Lists.newArrayList();
    for (CustomerCreditDto dto : customerCreditDtos) {
      results.add(this.init(dto.getCustomerCode(), dto.getInitialAmount(), null));
    }
    return results;
  }

  @Override
  @Transactional
  public CustomerCredit updateAmount(String customerCode, BigDecimal amount, String remark) {
    /*
     * 根据产品的需求进行修改信用额度的逻辑
     * 1、修改的金额为最终用户的信用额度
     * 2、计算是需要新增还是减少额度
     * 3、需要验证额度是否足够
     * 4、创建修改记录
     */
    Validate.notBlank(customerCode, "客户编码不能为空");
    Validate.notNull(amount, "调整额度不能为空");
    Validate.isTrue(amount.compareTo(BigDecimal.ZERO) >= 0, "调整额度不能为负");
    String tenantCode = TenantUtils.getTenantCode();
    Customer customer = customerService.findDetailsByTenantCodeAndCustomerCode(tenantCode, customerCode);
    Validate.notNull(customer, "客户【%s】不存在， 请检查！", customerCode);
    String lockKey = String.format(CUSTOMER_CREDIT_LOCK_PREFIX, tenantCode, customerCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "操作频繁，请稍后再试！");
    try {
      CustomerCredit customerCredit = customerCreditRepository.findByTenantCodeAndCustomerCode(tenantCode, customerCode);
      Validate.notNull(customerCredit, "客户【%s】未开通信用额度，请联系业务！", customerCode);
      BigDecimal beforeAmount = customerCredit.getTotalAmount();
      BigDecimal afterAmount = amount;
      BigDecimal changeAmount = afterAmount.subtract(beforeAmount).setScale(4,RoundingMode.HALF_UP);
      Validate.isTrue(beforeAmount.compareTo(afterAmount) != 0,"调整信用总额后与现有信用总额相同，请确认调整额度！");
      // 余额的计算始终是：总信用额度 - 已使用额度
      // 如果计算后的余额小于0，则设置为0
      BigDecimal balance = afterAmount.subtract(customerCredit.getUsedAmount());
      Validate.isTrue(balance.compareTo(BigDecimal.ZERO)>=0,"调整信用总额度后，余额小于0，请更换调整额度！");
      customerCredit.setTotalAmount(afterAmount.setScale(4, RoundingMode.HALF_UP));
      customerCredit.setBalance(balance.setScale(4, RoundingMode.HALF_UP));
      customerCreditRepository.save(customerCredit);
      customerCreditRecordService.create(customerCredit, changeAmount, beforeAmount, afterAmount, remark);
      return customerCredit;
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public CustomerCredit updateStatis(String customerCode, Integer tstatis) {
    Validate.notBlank(customerCode, "客户编码为空，请检查！");
    Validate.notNull(tstatis, "客户信用状态不能为空，请检查！");
    CustomerCredit customerCredit = this.findByCustomerCode(customerCode);
    Validate.notNull(customerCredit, "客户编码异常【%s】请检查!", customerCode);
    customerCredit.setTstatus(tstatis);
    customerCreditRepository.saveAndFlush(customerCredit);
    // 状态变化通知
    this.customerCreditEvent(customerCredit);
    return customerCredit;
  }

  @Override
  @Transactional
  public CustomerCredit receipt(String customerCode, BigDecimal amount, String businessNo, String remark) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "收款金额不能为空并且为正数");
    Validate.notBlank(businessNo, "业务单号不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    Customer customer = customerService.findDetailsByTenantCodeAndCustomerCode(tenantCode, customerCode);
    Validate.notNull(customer, "客户【%s】不存在， 请检查！", customerCode);
    String lockKey = String.format(CUSTOMER_CREDIT_LOCK_PREFIX, tenantCode, customerCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "操作频繁，请稍后再试！");
    try {
      CustomerCredit customerCredit = customerCreditRepository.findByTenantCodeAndCustomerCode(tenantCode, customerCode);
      Validate.notNull(customerCredit, "客户【%s】未开通信用额度，请联系业务！", customerCode);
      Validate.isTrue(customerCredit.getBalance().compareTo(amount) >= 0, "信用余额不足！");
      BigDecimal balance = customerCredit.getBalance().subtract(amount);
      BigDecimal usedAmount = customerCredit.getUsedAmount().add(amount);
      customerCredit.setBalance(balance.setScale(4, RoundingMode.HALF_UP));
      customerCredit.setUsedAmount(usedAmount.setScale(4, RoundingMode.HALF_UP));
      customerCreditRepository.save(customerCredit);
      // 保存流水单
      customerCreditBillService.create(customerCredit, businessNo, amount.negate(), balance, RECEIPT, remark);
      return customerCredit;
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  public boolean validReceipt(String customerCode, BigDecimal amount) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "支付金额不能为空并且为正数");
    Customer customer = customerService.findDetailsByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
    Validate.notNull(customer, "客户【%s】不存在， 请检查！", customerCode);
    CustomerCredit customerCredit = customerCreditRepository.findByTenantCodeAndCustomerCode(customer.getTenantCode(), customerCode);
    Validate.notNull(customerCredit, "客户【%s】未开通信用额度，请联系业务！", customerCode);
    return customerCredit.getBalance().compareTo(amount) >= 0;
  }

  @Override
  @Transactional
  public CustomerCredit recover(String customerCode, BigDecimal amount, String businessNo, String remark) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "恢复额度金额不能为空并且为正数");
    Validate.notBlank(businessNo, "业务单号不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    Customer customer = customerService.findDetailsByTenantCodeAndCustomerCode(tenantCode, customerCode);
    Validate.notNull(customer, "客户【%s】不存在， 请检查！", customerCode);
    String lockKey = String.format(CUSTOMER_CREDIT_LOCK_PREFIX, tenantCode, customerCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "操作频繁，请稍后再试！");
    try {
      CustomerCredit customerCredit = customerCreditRepository.findByTenantCodeAndCustomerCode(tenantCode, customerCode);
      Validate.notNull(customerCredit, "客户【%s】未开通信用额度，请联系业务！", customerCode);
      Validate.isTrue(amount.compareTo(customerCredit.getUsedAmount()) <= 0, "恢复额度金额大于欠款金额");
      BigDecimal usedAmount = customerCredit.getUsedAmount().subtract(amount);
      // 余额的计算始终是：总信用额度 - 已使用额度
      // 如果计算后的余额小于0，则设置为0
      BigDecimal balance = customerCredit.getTotalAmount().subtract(usedAmount);
      balance = balance.compareTo(BigDecimal.ZERO) < 0 ? BigDecimal.ZERO : balance;
      customerCredit.setBalance(balance.setScale(4, RoundingMode.HALF_UP));
      customerCredit.setUsedAmount(usedAmount.setScale(4, RoundingMode.HALF_UP));
      customerCreditRepository.save(customerCredit);
      // 保存流水单
      customerCreditBillService.create(customerCredit, businessNo, amount, balance, RECOVER, remark);
      return customerCredit;
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  public CustomerCredit findByCustomerCode(String customerCode) {
    if (StringUtils.isBlank(customerCode)) {
      return null;
    }
    return customerCreditRepository.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
  }

  @Override
  public Page<CustomerCreditVo> findByConditions(Pageable pageable, InvokeParams conditions) {
    conditions.getInvokeParams().put("tenantCode", TenantUtils.getTenantCode());
    Page<Object[]> page = customerCreditRepository.queryPage(pageable, conditions);
    List<CustomerCreditVo> list = Lists.newArrayList();
    if (page != null && page.getTotalElements() > 0L) {
      page.getContent().stream().forEach(data -> {
        CustomerCreditVo customerCreditVo = new CustomerCreditVo();
        customerCreditVo.setCustomerCode((String) data[0]);
        customerCreditVo.setCustomerName((String) data[1]);
        customerCreditVo.setCustomerLevel((String) data[2]);
        customerCreditVo.setCustomerCategory((String) data[3]);
        customerCreditVo.setSalesAreaName((String) data[4]);
        customerCreditVo.setTstatus((Integer) data[5]);
        customerCreditVo.setTotalAmount((BigDecimal) data[6]);
        customerCreditVo.setUsedAmount((BigDecimal) data[7]);
        customerCreditVo.setBalance((BigDecimal) data[8]);
        list.add(customerCreditVo);
      });
    } else {
      return null;
    }
    return new PageImpl<>(list, pageable, page.getTotalElements());
  }

  /**
   * 用户信用状态通知
   *
   * @param customerCredit 用户信用
   */
  private void customerCreditEvent(CustomerCredit customerCredit) {
    if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(customerCreditEventListeners)) {
      for (CustomerCreditEventListener eventListener : customerCreditEventListeners) {
        eventListener.onChange(customerCredit);
      }
    }
  }

}
