package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.payment.common.enums.WalletBillBusinessType;
import com.bizunited.empower.business.payment.entity.CustomerWallet;
import com.bizunited.empower.business.payment.entity.CustomerWalletBill;
import com.bizunited.empower.business.payment.repository.CustomerWalletBillRepository;
import com.bizunited.empower.business.payment.service.CustomerWalletBillService;
import com.bizunited.platform.common.service.CodeGeneratorService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.constant.Constants.WALLET_BILL_NO_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.CUSTOMER_WALLET_BILL_NO_INDEX;
import static com.bizunited.empower.business.payment.common.enums.CustomerCreditBillType.RECEIPT;
import static com.bizunited.empower.business.payment.common.enums.CustomerCreditBillType.RECOVER;
import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * CustomerWalletBill业务模型的服务层接口实现
 * @author saturn
 */
@Service("CustomerWalletBillServiceImpl")
public class CustomerWalletBillServiceImpl implements CustomerWalletBillService {

  @Autowired
  private CustomerWalletBillRepository customerWalletBillRepository;

  @Autowired
  private CodeGeneratorService codeGeneratorService;

  @Override
  @Transactional
  public CustomerWalletBill create(CustomerWallet wallet, BigDecimal amount, WalletBillBusinessType businessType, String businessNo, String remark) {
    Validate.notNull(wallet, "钱包信息不能为空");
    Validate.notNull(amount, "金额不能为空");
    Validate.notNull(businessType, "业务类型不能为空");
    Validate.notNull(businessNo, "业务编号不能为空");
    Validate.isTrue(amount.compareTo(BigDecimal.ZERO) != 0, "金额不能为0");
    Integer type = amount.compareTo(BigDecimal.ZERO) > 0 ? RECOVER.getType() : RECEIPT.getType();
    CustomerWalletBill bill = new CustomerWalletBill();
    bill.setType(type);
    bill.setBillNo(this.generateBillNo(wallet));
    bill.setBusinessType(businessType.getType());
    bill.setBusinessNo(businessNo);
    bill.setAmount(amount);
    bill.setBalance(wallet.getBalance());
    bill.setRemark(remark);
    bill.setCreateAccount(SecurityUtils.getUserAccount());
    bill.setCreateTime(new Date());
    bill.setWallet(wallet);
    return customerWalletBillRepository.save(bill);
  }

  @Override
  public Page<CustomerWalletBill> findByCustomerCodeAndConditions(String customerCode, Pageable pageable, Map<String, Object> conditions) {
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    if(StringUtils.isBlank(customerCode)) {
      return Page.empty(pageable);
    }
    conditions = ObjectUtils.defaultIfNull(conditions, Maps.newHashMap());
    String tenantCode = TenantUtils.getTenantCode();
    conditions.put("tenantCode", tenantCode);
    return customerWalletBillRepository.findByCustomerCodeAndConditions(customerCode, pageable, conditions);
  }

  @Override
  public BigDecimal sumAmountByTypeAndTimeBetween(Integer type, Date start, Date end) {
    if(!ObjectUtils.allNotNull(type, start, end)) {
      return BigDecimal.ZERO;
    }
    String tenantCode = TenantUtils.getTenantCode();
    BigDecimal amount = customerWalletBillRepository.sumAmountByTenantCodeAndTypeAndCreateTimeBetween(tenantCode, type, start, end);
    return amount == null ? BigDecimal.ZERO : amount;
  }

  /**
   * 生成流水编号
   * @param wallet
   * @return
   */
  private String generateBillNo(CustomerWallet wallet) {
    String date = new SimpleDateFormat("yyyyMMdd").format(new Date());
    String key = String.format(CUSTOMER_WALLET_BILL_NO_INDEX, wallet.getTenantCode(), date);
    return codeGeneratorService.generate(key, WALLET_BILL_NO_PREFIX + date, null, 1, 6, 25, TimeUnit.HOURS);
  }

}
