package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.customer.common.enums.CustomerSmsBusinessType;
import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.customer.service.CustomerSmsService;
import com.bizunited.empower.business.payment.common.enums.WalletBillBusinessType;
import com.bizunited.empower.business.payment.entity.CustomerWallet;
import com.bizunited.empower.business.payment.repository.CustomerWalletRepository;
import com.bizunited.empower.business.payment.service.CustomerWalletBillService;
import com.bizunited.empower.business.payment.service.CustomerWalletService;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.Date;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.constant.Constants.WALLET_BILL_BUSINESS_TYPE_RECEIPT;
import static com.bizunited.empower.business.payment.common.constant.Constants.WALLET_BILL_BUSINESS_TYPE_RECOVER;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.CUSTOMER_WALLET_LOCK_PREFIX;
import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * CustomerWallet业务模型的服务层接口实现
 * @author saturn
 */
@Service("CustomerWalletServiceImpl")
public class CustomerWalletServiceImpl implements CustomerWalletService {

  @Autowired
  private CustomerWalletRepository customerWalletRepository;

  @Autowired
  private CustomerService customerService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private CustomerSmsService customerSmsService;
  @Autowired
  private CustomerWalletBillService customerWalletBillService;

  @Override
  @Transactional
  public CustomerWallet init(String customerCode) {
    Validate.notBlank(customerCode, "客户编码不能为空！");
    String tenantCode = TenantUtils.getTenantCode();
    Customer customer = customerService.findByTenantCodeAndCustomerCode(tenantCode, customerCode);
    Validate.notNull(customer, "未找到客户：%s", customerCode);
    Validate.isTrue(customer.getEnabledState(), "客户未启用");
    String lockKey = String.format(CUSTOMER_WALLET_LOCK_PREFIX, tenantCode, customerCode);
    boolean locked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(locked, "操作繁忙，请稍后再试！");
    try {
      CustomerWallet wallet = customerWalletRepository.findByTenantCodeAndCustomerCode(tenantCode, customerCode);
      Validate.isTrue(wallet == null, "客户电子钱包已初始化！");
      String userAccount = SecurityUtils.getUserAccount();
      Date now = new Date();
      wallet = new CustomerWallet();
      wallet.setTenantCode(tenantCode);
      wallet.setBalance(BigDecimal.ZERO);
      wallet.setCustomerCode(customerCode);
      wallet.setCustomerName(customer.getCustomerName());
      wallet.setState(NormalStatusEnum.ENABLE.getStatus());
      wallet.setCreateAccount(userAccount);
      wallet.setCreateTime(now);
      wallet.setModifyAccount(userAccount);
      wallet.setModifyTime(now);
      return customerWalletRepository.save(wallet);
    } finally {
      if(redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public CustomerWallet receipt(String customerCode, BigDecimal amount, Integer businessType, String businessNo, String remark) {
    Validate.notBlank(customerCode, "客户编码不能为空！");
    Validate.notNull(businessType, "业务类型不能为空！");
    Validate.notBlank(businessNo, "业务编码不能为空！");
    Validate.notNull(amount, "金额不能为空");
    WalletBillBusinessType billBusinessType = WalletBillBusinessType.valueOfType(businessType);
    Validate.notNull(billBusinessType, "不支持的业务类型:%s", businessType);
    Validate.isTrue(WALLET_BILL_BUSINESS_TYPE_RECEIPT.contains(billBusinessType), "不支持的业务类型:%s", businessType);
    Validate.isTrue(amount.compareTo(BigDecimal.ZERO) > 0, "扣款金额必须大于0");
    String tenantCode = TenantUtils.getTenantCode();
    String lockKey = String.format(CUSTOMER_WALLET_LOCK_PREFIX, tenantCode, customerCode);
    boolean locked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(locked, "操作繁忙，请稍后再试！");
    try {
      CustomerWallet wallet = this.findByCustomerCode(customerCode);
      Validate.notNull(wallet, "当前客户未开通电子钱包");
      Validate.isTrue(NormalStatusEnum.ENABLE.getStatus().equals(wallet.getState()), "客户电子钱包已冻结");
      Validate.isTrue(wallet.getBalance().compareTo(amount) >= 0, "余额不足");
      BigDecimal balance = wallet.getBalance().subtract(amount);
      wallet.setBalance(balance);
      wallet.setModifyTime(new Date());
      wallet.setModifyAccount(SecurityUtils.getUserAccount());
      customerWalletRepository.save(wallet);
      customerWalletBillService.create(wallet, amount.negate(), billBusinessType, businessNo, remark);
      return wallet;
    } finally {
      if(redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public CustomerWallet receipt(String customerCode, BigDecimal amount, Integer businessType, String businessNo, String validCode, String remark) {
    Validate.notBlank(customerCode, "客户编码不能为空！");
    Validate.notBlank(validCode, "验证码不能为空");
    Customer customer = customerService.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
    Validate.notNull(customer, "未找到客户：%s", customerCode);
    customerSmsService.verifyValidCode(customerCode, customer.getSecurityMobile(), CustomerSmsBusinessType.ORDER_RECEIPT, validCode, 0L);
    return this.receipt(customerCode, amount, businessType, businessNo, remark);
  }

  @Override
  @Transactional
  public CustomerWallet recover(String customerCode, BigDecimal amount, Integer businessType, String businessNo, String remark) {
    Validate.notBlank(customerCode, "客户编码不能为空！");
    Validate.notNull(businessType, "业务类型不能为空！");
    Validate.notBlank(businessNo, "业务编码不能为空！");
    Validate.notNull(amount, "金额不能为空");
    WalletBillBusinessType billBusinessType = WalletBillBusinessType.valueOfType(businessType);
    Validate.notNull(billBusinessType, "不支持的业务类型:%s", businessType);
    Validate.isTrue(WALLET_BILL_BUSINESS_TYPE_RECOVER.contains(billBusinessType), "不支持的业务类型:%s", businessType);
    String tenantCode = TenantUtils.getTenantCode();
    Validate.isTrue(amount.compareTo(BigDecimal.ZERO) > 0, "恢复金额必须大于0");
    String lockKey = String.format(CUSTOMER_WALLET_LOCK_PREFIX, tenantCode, customerCode);
    boolean locked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(locked, "操作繁忙，请稍后再试！");
    try {
      CustomerWallet wallet = this.findByCustomerCode(customerCode);
      Validate.notNull(wallet, "当前客户未开通电子钱包");
      Validate.isTrue(NormalStatusEnum.ENABLE.getStatus().equals(wallet.getState()), "客户电子钱包已冻结");
      BigDecimal balance = wallet.getBalance().add(amount);
      wallet.setBalance(balance);
      wallet.setModifyTime(new Date());
      wallet.setModifyAccount(SecurityUtils.getUserAccount());
      customerWalletRepository.save(wallet);
      customerWalletBillService.create(wallet, amount, billBusinessType, businessNo, remark);
      return wallet;
    } finally {
      if(redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public CustomerWallet withdraw(String customerCode, BigDecimal amount, String remark) {
    Validate.notBlank(customerCode, "客户编码不能为空！");
    Validate.isTrue(amount == null || amount.compareTo(BigDecimal.ZERO) > 0, "提现金额不能小于0");
    String tenantCode = TenantUtils.getTenantCode();
    String lockKey = String.format(CUSTOMER_WALLET_LOCK_PREFIX, tenantCode, customerCode);
    boolean locked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(locked, "操作繁忙，请稍后再试！");
    try {
      CustomerWallet wallet = this.findByCustomerCode(customerCode);
      Validate.notNull(wallet, "当前客户未开通电子钱包");
      Validate.isTrue(wallet.getBalance().compareTo(BigDecimal.ZERO) > 0, "没有余额可提现");
      BigDecimal withdrawAmount = wallet.getBalance();
      if(amount != null) {
        Validate.isTrue(wallet.getBalance().compareTo(amount) >= 0, "余额不足！");
        withdrawAmount = amount;
      }
      BigDecimal balance = wallet.getBalance().subtract(withdrawAmount);
      wallet.setBalance(balance);
      wallet.setModifyTime(new Date());
      wallet.setModifyAccount(SecurityUtils.getUserAccount());
      customerWalletRepository.save(wallet);
      customerWalletBillService.create(wallet, withdrawAmount.negate(), WalletBillBusinessType.WITHDRAW, "", remark);
      return wallet;
    } finally {
      if(redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  public boolean validReceipt(String customerCode, BigDecimal amount) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "支付金额不能为空并且为正数");
    Customer customer = customerService.findDetailsByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
    Validate.notNull(customer, "客户【%s】不存在， 请检查！", customerCode);
    CustomerWallet wallet = customerWalletRepository.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
    Validate.notNull(wallet, "客户【%s】未开通电子钱包", customerCode);
    return wallet.getBalance().compareTo(amount) >= 0;
  }

  @Override
  public CustomerWallet findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<CustomerWallet> op = customerWalletRepository.findById(id);
    return op.orElse(null); 
  }

  @Override
  public CustomerWallet findByCustomerCode(String customerCode) {
    if(StringUtils.isBlank(customerCode)) {
      return null;
    }
    CustomerWallet wallet = customerWalletRepository.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
    if(wallet == null) {
      // 如果客户没有电子钱包，则初始化一个
      CustomerWalletService customerWalletService = applicationContext.getBean(CustomerWalletService.class);
      customerWalletService.init(customerCode);
    }
    return wallet;
  }

  @Override
  public Page<CustomerWallet> findByConditions(Map<String, Object> conditions, Pageable pageable) {
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    conditions = ObjectUtils.defaultIfNull(conditions, Maps.newHashMap());
    conditions.put("tenantCode", TenantUtils.getTenantCode());
    return customerWalletRepository.findByConditions(conditions, pageable);
  }

  @Override
  public BigDecimal sumBalance() {
    String tenantCode = TenantUtils.getTenantCode();
    BigDecimal balance = customerWalletRepository.sumBalanceByTenantCode(tenantCode);
    return balance == null ? BigDecimal.ZERO : balance;
  }

}
