package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.entity.SalesArea;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.payment.entity.CustomerWallet;
import com.bizunited.empower.business.payment.service.CustomerWalletService;
import com.bizunited.empower.business.payment.service.CustomerWalletVoService;
import com.bizunited.empower.business.payment.vo.CustomerWalletVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * 客户电子钱包Vo服务
 * @Author: Paul Chan
 * @Date: 2021/4/13 下午3:14
 */
@Service
public class CustomerWalletVoServiceImpl implements CustomerWalletVoService {

  @Autowired
  private CustomerService customerService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private CustomerWalletService customerWalletService;

  @SuppressWarnings({"unchecked", "rawtypes"})
  @Override
  public Page<CustomerWalletVo> findByConditions(Map<String, Object> conditions, Pageable pageable) {
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    conditions = ObjectUtils.defaultIfNull(conditions, Maps.newHashMap());
    Map<String, Object> customerConditions = this.getCustomerConditions(conditions);
    List<String> customerCodes = Lists.newArrayList();
    if(!CollectionUtils.isEmpty(customerConditions)) {
      Page<Customer> customerPage = customerService.findByConditions(PageRequest.of(0, Integer.MAX_VALUE), customerConditions);
      if(customerPage.isEmpty()) {
        return Page.empty(pageable);
      }
      customerCodes = customerPage.stream().map(Customer::getCustomerCode).collect(Collectors.toList());
    }
    conditions.put("customerCodes", customerCodes);
    Page<CustomerWallet> page = customerWalletService.findByConditions(conditions, pageable);
    if(page.isEmpty()) {
      return Page.empty(pageable);
    }
    Collection<CustomerWalletVo> collection = nebulaToolkitService.copyCollectionByWhiteList(page.getContent(), CustomerWallet.class, CustomerWalletVo.class, HashSet.class, ArrayList.class);
    List<CustomerWalletVo> list = Lists.newArrayList(collection);
    this.loadDetails(list);
    return new PageImpl(list, pageable, page.getTotalElements());
  }

  /**
   * 加载明细
   * @param list
   */
  private void loadDetails(List<CustomerWalletVo> list) {
    String tenantCode = TenantUtils.getTenantCode();
    for (CustomerWalletVo wallet : list) {
      Customer customer = customerService.findByTenantCodeAndCustomerCode(tenantCode, wallet.getCustomerCode());
      if(customer != null) {
        wallet.setContactPerson(customer.getContactPerson());
        wallet.setCustomerName(customer.getCustomerName());
        wallet.setPhone(customer.getPhone());
        SalesArea salesArea = customer.getSalesArea();
        if(salesArea != null) {
          wallet.setSalesAreaName(salesArea.getSalesAreaName());
        }
      }
    }
  }

  /**
   * 获取客户的条件
   * @param conditions
   * @return
   */
  private Map<String, Object> getCustomerConditions(Map<String, Object> conditions) {
    Map<String, Object> customerConditions = Maps.newHashMap();
    String keyword = (String) conditions.get("keyword");
    if(StringUtils.isNotBlank(keyword)) {
      customerConditions.put("keyword", conditions.get("keyword"));
    }
    return customerConditions;
  }

}
