package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.payment.common.enums.DefrayStatus;
import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.entity.DefrayInfo;
import com.bizunited.empower.business.payment.entity.PaymentInfo;
import com.bizunited.empower.business.payment.repository.DefrayInfoRepository;
import com.bizunited.empower.business.payment.service.DefrayInfoService;
import com.bizunited.empower.business.payment.service.DefrayStrategy;
import com.bizunited.empower.business.payment.service.PaymentInfoService;
import com.bizunited.empower.business.payment.vo.DefrayInfoVo;
import com.bizunited.empower.business.payment.vo.PaymentInfoVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.bizunited.empower.business.payment.common.constant.Constants.DEFRAY_INFO_NO_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.DEFRAY_INFO_NO_KEY_PREFIX;

/**
 * DefrayInfo业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("DefrayInfoServiceImpl")
public class DefrayInfoServiceImpl implements DefrayInfoService {
  @Autowired
  private DefrayInfoRepository defrayInfoRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private PaymentInfoService paymentInfoService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private CustomerService customerService;
  @Autowired(required = false)
  private List<DefrayStrategy> defrayStrategies;

  @Transactional
  @Override
  public DefrayInfo create(DefrayInfo defrayInfo) {
    DefrayInfo current = this.createForm(defrayInfo);
    return current;
  }

  @Override
  public DefrayInfo createWithoutConfirm(DefrayInfo defrayInfo) {
    DefrayInfo current = this.createForm(defrayInfo);
    return this.confirm(current.getDefrayCode());
  }

  @Transactional
  @Override
  public DefrayInfo createForm(DefrayInfo defrayInfo) {
    Date now = new Date();
    defrayInfo.setCreateAccount(SecurityUtils.getUserAccount());
    defrayInfo.setCreateTime(now);
    defrayInfo.setModifyAccount(SecurityUtils.getUserAccount());
    defrayInfo.setModifyTime(now);

    // 设置编号、租户、用户等信息
    defrayInfo.setDefrayCode(generateNo());
    defrayInfo.setTenantCode(TenantUtils.getTenantCode());

    this.createValidation(defrayInfo);

    FundsChannelType fundsChannelType = FundsChannelType.valueOfType(defrayInfo.getFundsChannel());
    DefrayStrategy currentStrategy = getDefrayStrategy(fundsChannelType);
    currentStrategy.handler(defrayInfo);
    return defrayInfo;
  }

  /**
   * 在创建一个新的DefrayInfo模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(DefrayInfo defrayInfo) {
    Validate.notNull(defrayInfo, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(defrayInfo.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    defrayInfo.setId(null);
    Validate.notBlank(defrayInfo.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(defrayInfo.getDefrayCode(), "添加信息时，付款单流水编号不能为空！");
    Validate.notNull(defrayInfo.getDefrayAmount(), "添加信息时，付款金额不能为空！");
    Validate.notNull(defrayInfo.getFundsChannel(), "添加信息时，付款资金渠道不能为空!");
    Validate.notNull(defrayInfo.getDefrayTime(), "添加信息时，付款时间不能为空");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(defrayInfo.getDefrayAmount().compareTo(BigDecimal.ZERO) >= 0, "添加信息时，金额不能为负!");
    Validate.isTrue(defrayInfo.getTenantCode() == null || defrayInfo.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(defrayInfo.getDefrayCode() == null || defrayInfo.getDefrayCode().length() < 64, "付款单流水编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.notNull(defrayInfo.getPaymentInfo(), "添加信息时，应付账款信息不能为空!");
    PaymentInfo paymentInfo = this.paymentInfoService.findByPaymentCode(defrayInfo.getPaymentInfo().getPaymentCode());
    Validate.notNull(paymentInfo, "应付账款信息不能为空");
    defrayInfo.setPaymentInfo(paymentInfo);
    defrayInfo.setCustomerCode(paymentInfo.getCustomerCode());
    DefrayInfo currentDefrayInfo = this.findByDefrayCode((defrayInfo.getDefrayCode()));
    Validate.isTrue(currentDefrayInfo == null, "付款单流水编号已存在,请检查");
  }

  @Override
  public Set<DefrayInfo> findDetailsByPaymentInfo(String paymentInfo) {
    if (StringUtils.isBlank(paymentInfo)) {
      return Sets.newHashSet();
    }
    return this.defrayInfoRepository.findDetailsByPaymentInfo(paymentInfo);
  }

  @Override
  public DefrayInfo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.defrayInfoRepository.findDetailsById(id);
  }

  @Override
  public DefrayInfo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Optional<DefrayInfo> op = defrayInfoRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public DefrayInfo findDetailsByDefrayCode(String defrayCode) {
    if (StringUtils.isBlank(defrayCode)) {
      return null;
    }
    return this.defrayInfoRepository.findDetailsByDefrayCode(defrayCode, TenantUtils.getTenantCode());
  }

  @Override
  public DefrayInfo findByDefrayCode(String defrayCode) {
    if (StringUtils.isBlank(defrayCode)) {
      return null;
    }
    return this.defrayInfoRepository.findByDefrayCode(defrayCode, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public DefrayInfo confirm(String defrayCode) {
    Validate.notBlank(defrayCode, "业务付款单号不能为空");
    DefrayInfo defrayInfo = this.defrayInfoRepository.findByDefrayCode(defrayCode, TenantUtils.getTenantCode());
    Validate.notNull(defrayInfo, "付款单数据错误，请联系供应商");
    Validate.isTrue(defrayInfo.getDefrayStatus().intValue() == DefrayStatus.WAITFOR.getValue().intValue(), "付款单状态不是待确认，请检查");
    defrayInfo.setDefrayStatus(DefrayStatus.CONFIRMED.getValue());
    this.defrayInfoRepository.saveAndFlush(defrayInfo);
    // 确认付款单后 进行应付账款的金额扣除工作
    this.paymentInfoService.confirmPay(defrayInfo.getPaymentInfo().getPaymentCode(), defrayInfo.getDefrayAmount(), defrayInfo);
    return defrayInfo;
  }

  @Override
  @Transactional
  public DefrayInfo createByAssociatedCode(String associatedCode, BigDecimal amount, FundsChannelType fundsChannelType) {
    Validate.notBlank(associatedCode, "关联单据号为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "付款金额不能为负或者0");
    DefrayInfo defrayInfo = new DefrayInfo();
    defrayInfo.setDefrayAmount(amount);
    defrayInfo.setFundsChannel(fundsChannelType.getValue());
    defrayInfo.setDefrayTime(new Date());
    PaymentInfo paymentInfo = this.paymentInfoService.findByAssociatedCode(associatedCode);
    defrayInfo.setPaymentInfo(paymentInfo);
    this.create(defrayInfo);
    return defrayInfo;
  }

  @Override
  @Transactional
  public DefrayInfo cancelByDefrayCode(String defrayCode) {
    Validate.notBlank(defrayCode, "付务付款单号不能为空");
    DefrayInfo defrayInfo = this.defrayInfoRepository.findByDefrayCode(defrayCode, TenantUtils.getTenantCode());
    Validate.notNull(defrayInfo, "付款单数据错误，请联系供应商");
    Validate.isTrue(defrayInfo.getDefrayStatus().intValue() == DefrayStatus.WAITFOR.getValue().intValue(), "付款单状态不是待确认，请检查");
    defrayInfo.setDefrayStatus(DefrayStatus.CANNEL.getValue());
    defrayInfo = this.defrayInfoRepository.saveAndFlush(defrayInfo);
    this.paymentInfoService.cancelWaitPay(defrayInfo.getPaymentInfo().getPaymentCode(), defrayInfo.getDefrayAmount());
    return defrayInfo;
  }

  @Override
  public Page<DefrayInfoVo> findByConditions(Pageable pageable, InvokeParams conditions) {
    conditions.getInvokeParams().put("tenantCode", TenantUtils.getTenantCode());
    Page<Object> page = this.defrayInfoRepository.queryPage(pageable, conditions);
    List<DefrayInfoVo> list = Lists.newArrayList();
    if (page != null && page.getTotalElements() > 0L) {
      page.getContent().forEach(data -> {
        DefrayInfo defrayInfo = this.defrayInfoRepository.findDetailsById((String) data);
        DefrayInfoVo vo = nebulaToolkitService.copyObjectByWhiteList(defrayInfo, DefrayInfoVo.class, HashSet.class, ArrayList.class, "paymentInfo");
        Customer customer = customerService.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), vo.getCustomerCode());
        Validate.notNull(customer, "客户编号为【%s】信息异常, 请检查!", vo.getCustomerCode());
        vo.setCustomerName(customer.getCustomerName());
        list.add(vo);
      });
    } else {
      return Page.empty(pageable);
    }
    return new PageImpl<>(list, pageable, page.getTotalElements());
  }

  /**
   * 按人员和指定日期，统计付款总金额
   *
   * @param account    创建人账号(付款人业务员账号)
   * @param defrayTime 付款日期
   */
  @Override
  public BigDecimal sumTotalDefrayAmountByAccountAndDefrayTime(String account, Date defrayTime) {
    if (StringUtils.isAnyBlank(account, TenantUtils.getTenantCode()) || defrayTime == null) {
      return BigDecimal.ZERO;
    }
    return defrayInfoRepository.sumTotalDefrayAmountByAccountAndDefrayTimeAndTenantCode(account, defrayTime, TenantUtils.getTenantCode());
  }

  /**
   * 按人员和指定日期，统计付款总单数
   *
   * @param account    创建人账号(付款人业务员账号)
   * @param defrayTime 付款日期
   */
  @Override
  public BigInteger countTotalDefrayNumByAccountAndDefrayTime(String account, Date defrayTime) {
    if (StringUtils.isAnyBlank(account, TenantUtils.getTenantCode()) || defrayTime == null) {
      return BigInteger.ZERO;
    }
    return BigInteger.valueOf(defrayInfoRepository.countTotalDefrayNumByAccountAndDefrayTimeAndTenantCode(account, defrayTime, TenantUtils.getTenantCode()));
  }

  @Override
  public BigDecimal findTotalByConditions(Pageable pageable, InvokeParams conditions) {
    BigDecimal payedAmountTotal = new BigDecimal(BigInteger.ZERO);
    pageable = PageRequest.of(0,Integer.MAX_VALUE);
    Page<DefrayInfoVo> byConditions = this.findByConditions(pageable, conditions);
    if (null!=byConditions&&!CollectionUtils.isEmpty(byConditions.getContent())){
      List<DefrayInfoVo> content = byConditions.getContent();
      List<PaymentInfoVo> paymentInfoVoList = content.stream().filter(o -> o.getPaymentInfo() != null)
              .map(DefrayInfoVo::getPaymentInfo).collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(paymentInfoVoList)){
        payedAmountTotal = paymentInfoVoList.stream().map(PaymentInfoVo::getPayedAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
      }
    }
    return payedAmountTotal;
  }

  @Override
  public List<DefrayInfo> findByDefrayCodes(List<String> defrayCodes) {
    return this.defrayInfoRepository.findByDefrayCodes(defrayCodes);
  }

  /**
   * 生成业务编号
   *
   * @return
   */
  private String generateNo() {
    Date now = new Date();
    SimpleDateFormat dayFormat = new SimpleDateFormat("yyyyMMdd");
    String redisKey = String.format(DEFRAY_INFO_NO_KEY_PREFIX, TenantUtils.getTenantCode(), dayFormat.format(now));
    // 设置key有效期为一天
    String index = redisMutexService.getAndIncrement(redisKey, 1, 8, 86401, TimeUnit.SECONDS);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(DEFRAY_INFO_NO_PREFIX, format.format(now), index);
  }

  /**
   * 获取当前执行策略
   *
   * @param type
   * @return
   */
  private DefrayStrategy getDefrayStrategy(FundsChannelType type) {
    Optional<DefrayStrategy> defrayStrategyOp = defrayStrategies.stream().filter(item -> type.getValue() == item.getFundsChannel().intValue()).findFirst();
    DefrayStrategy defrayStrategy = defrayStrategyOp.orElse(null);
    Validate.notNull(defrayStrategy, String.format("类型【%d】没有对应的处理策略，请检查！", type.getValue()));
    return defrayStrategy;
  }
}
