package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.payment.common.enums.ElectronicAccountBankState;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountTiedCardStatus;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountType;
import com.bizunited.empower.business.payment.dto.CpcnResultDto;
import com.bizunited.empower.business.payment.dto.ElectronicAccountDto;
import com.bizunited.empower.business.payment.entity.ElectronicAccount;
import com.bizunited.empower.business.payment.entity.ElectronicAccountBank;
import com.bizunited.empower.business.payment.repository.ElectronicAccountBankRepository;
import com.bizunited.empower.business.payment.service.ElectronicAccountBankService;
import com.bizunited.empower.business.payment.service.ElectronicAccountDtoService;
import com.bizunited.empower.business.payment.service.ElectronicAccountService;
import com.bizunited.empower.business.tenant.common.enums.TenantSmsBusinessType;
import com.bizunited.empower.business.tenant.service.TenantSmsService;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * ElectronicAccountBank业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ElectronicAccountBankServiceImpl")
public class ElectronicAccountBankServiceImpl implements ElectronicAccountBankService {

  @Autowired
  private ElectronicAccountBankRepository electronicAccountBankRepository;

  @Autowired
  private TenantSmsService tenantSmsService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ElectronicAccountService electronicAccountService;
  @Autowired
  private ElectronicAccountDtoService electronicAccountDtoService;

  @Override
  @Transactional
  public ElectronicAccountBank create(ElectronicAccountBank bank) {
    this.createValidation(bank);
    bank.setState(ElectronicAccountBankState.WAITING_VALID.getState());
    ElectronicAccountBank accountBank = electronicAccountBankRepository.saveAndFlush(bank);
    // 同步到saas平台
    ElectronicAccount account = electronicAccountService.findDetailsByTenant();
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(account, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    electronicAccountDtoService.updateForTiedCard(electronicAccountDto);
    electronicAccountService.updateTiedCardStatusById(bank.getElectronicAccount().getId(), ElectronicAccountTiedCardStatus.TIED_PROCESSING.getStatus());
    return accountBank;
  }

  @Override
  @Transactional
  public ElectronicAccountBank update(ElectronicAccountBank bank) {
    this.updateValidation(bank);
    ElectronicAccountBank dbBank = electronicAccountBankRepository.findById(bank.getId()).orElse(null);
    Validate.notNull(dbBank, "更新对象不存在！");
    Validate.isTrue(ElectronicAccountBankState.WAITING_VALID.getState().equals(dbBank.getState()), "待审核的银行卡才能修改");
    ElectronicAccount account = dbBank.getElectronicAccount();
    Validate.notNull(account, "未找到电子账户");
    this.update(bank, dbBank);
    ElectronicAccountBank accountBank = electronicAccountBankRepository.saveAndFlush(dbBank);
    // 同步到saas平台
    account = electronicAccountService.findDetailsById(account.getId());
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(account, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    electronicAccountDtoService.updateForTiedCard(electronicAccountDto);
    return accountBank;
  }

  /**
   * 更新验证
   *
   * @param bank
   */
  private void updateValidation(ElectronicAccountBank bank) {
    Validate.notNull(bank, "银行卡信息不能为空");
    Validate.notBlank(bank.getId(), "银行卡信息ID不能为空");
    ElectronicAccountBank dbBank = electronicAccountBankRepository.findById(bank.getId()).orElse(null);
    Validate.notNull(dbBank, "更新对象不存在！");
    ElectronicAccount account = dbBank.getElectronicAccount();
    Validate.notNull(account, "未找到电子账户");
    this.saveValidation(account, bank);
    long count = electronicAccountBankRepository.countByElectronicAccountIdAndState(account.getId(), ElectronicAccountBankState.ENABLE.getState());
    Validate.isTrue(count == 0, "当前电子账户已绑定银行卡");
  }

  /**
   * 创建验证
   *
   * @param bank
   */
  private void createValidation(ElectronicAccountBank bank) {
    Validate.notNull(bank, "银行卡信息不能为空");
    Validate.notNull(bank.getElectronicAccount(), "账户信息不能为空");
    Validate.notBlank(bank.getElectronicAccount().getId(), "账户信息ID不能为空");
    ElectronicAccount account = electronicAccountService.findById(bank.getElectronicAccount().getId());
    Validate.notNull(account, "未找到对应的电子账户:%s", bank.getElectronicAccount().getId());
    this.saveValidation(account, bank);
    long count = electronicAccountBankRepository.countByElectronicAccountIdAndStateNot(account.getId(), ElectronicAccountBankState.UNBOUND.getState());
    Validate.isTrue(count == 0, "电子账户已绑定银行卡");
  }

  @Override
  @Transactional
  public Set<ElectronicAccountBank> save(ElectronicAccount electronicAccount, Set<ElectronicAccountBank> banks) {
    Validate.notEmpty(banks, "银行信息不能为空");
    banks = banks.stream()
            .filter(b -> b.getState() == null || !ElectronicAccountBankState.UNBOUND.getState().equals(b.getState()))
            .collect(Collectors.toSet());
    Validate.isTrue(banks.size() == 1, "银行卡信息只能有一个");
    ElectronicAccountBank dbBank = electronicAccountBankRepository.findByElectronicAccountIdAndState(electronicAccount.getId(), ElectronicAccountBankState.ENABLE.getState());
    if (dbBank != null) {
      // 如果已经有正常状态的银行卡，则直接返回，
      // 可能存在：银行卡已经验证通过，但是需要修改其他电子账户资料
      return Sets.newHashSet(dbBank);
    }
    ElectronicAccountBank bank = banks.iterator().next();
    this.saveValidation(electronicAccount, bank);
    dbBank = electronicAccountBankRepository.findByElectronicAccountIdAndState(electronicAccount.getId(),
            ElectronicAccountBankState.WAITING_VALID.getState());
    if (dbBank == null) {
      dbBank = bank;
      dbBank.setState(ElectronicAccountBankState.WAITING_VALID.getState());
      dbBank.setElectronicAccount(electronicAccount);
    } else {
      this.update(bank, dbBank);
    }
    electronicAccountBankRepository.save(dbBank);
    return Sets.newHashSet(bank);
  }

  /**
   * 更新基础数据
   *
   * @param bank
   * @param dbBank
   */
  private void update(ElectronicAccountBank bank, ElectronicAccountBank dbBank) {
    dbBank.setManagementName(bank.getManagementName());
    dbBank.setIdCardNo(bank.getIdCardNo());
    dbBank.setBankCardAccountName(bank.getBankCardAccountName());
    dbBank.setBankCardNo(bank.getBankCardNo());
    dbBank.setBankName(bank.getBankName());
    dbBank.setBankId(bank.getBankId());
    dbBank.setBankBranchName(bank.getBankBranchName());
    dbBank.setMobile(bank.getMobile());
    dbBank.setProvinceName(bank.getProvinceName());
    dbBank.setProvinceCode(bank.getProvinceCode());
    dbBank.setCityCode(bank.getCityCode());
    dbBank.setCityName(bank.getCityName());
    dbBank.setDistrictCode(bank.getDistrictCode());
    dbBank.setDistrictName(bank.getDistrictName());
  }

  @Override
  public void sendValidCodeById(String id) {
    Validate.notBlank(id, "ID不能为空");
    ElectronicAccountBank bank = electronicAccountBankRepository.findById(id).orElse(null);
    Validate.notNull(bank, "银行卡信息不存在");
    Validate.isTrue(!ElectronicAccountBankState.UNBOUND.getState().equals(bank.getState()), "当前银行卡已解绑");
    electronicAccountService.updateTiedCardStatusById(bank.getElectronicAccount().getId(), ElectronicAccountTiedCardStatus.WAIT_VERIFY.getStatus());
    // 调用saas平台发送验证码
    ElectronicAccount account = electronicAccountService.findDetailsByTenant();
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(account, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    electronicAccountDtoService.handleTiedCard(electronicAccountDto);
  }

  @Override
  @Transactional
  public void bindById(String id, String validCode) {
    Validate.notBlank(id, "ID不能为空");
    Validate.notBlank(validCode, "验证码不能为空");
    ElectronicAccountBank bank = electronicAccountBankRepository.findById(id).orElse(null);
    Validate.notNull(bank, "银行卡信息不存在");
    ElectronicAccount account = bank.getElectronicAccount();
    Validate.notNull(account, "未找到电子账户信息");
    Validate.isTrue(ElectronicAccountBankState.WAITING_VALID.getState().equals(bank.getState()), "没有待验证的银行卡");
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(account, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
            "accountInfo", "certificate", "banks");
    CpcnResultDto cpcnResultDto = electronicAccountDtoService.bindBank(electronicAccountDto, validCode);
    if (cpcnResultDto.getProcessing() != null && cpcnResultDto.getProcessing()) {
      return;
    }
    electronicAccountService.updateTiedCardStatusById(bank.getElectronicAccount().getId(), ElectronicAccountTiedCardStatus.TIED_CARD.getStatus());
  }

  @Override
  @Transactional
  public void unbindById(String id, String validCode) {
    Validate.notBlank(id, "ID不能为空");
    Validate.notBlank(validCode, "验证码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    tenantSmsService.verifyValidCode(tenantCode, TenantSmsBusinessType.EL_ACCOUNT_UNBIND_BANK, validCode);
    ElectronicAccountBank bank = electronicAccountBankRepository.findById(id).orElse(null);
    Validate.notNull(bank, "银行卡信息不存在");
    ElectronicAccount account = bank.getElectronicAccount();
    Validate.notNull(account, "未找到电子账户信息");
    Validate.isTrue(ElectronicAccountBankState.ENABLE.getState().equals(bank.getState()), "当前银行卡非正常状态");
    // 调用saas平台接口解绑
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(account, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    CpcnResultDto cpcnResultDto = electronicAccountDtoService.unbindBank(electronicAccountDto);
    if(cpcnResultDto.getProcessing() != null && cpcnResultDto.getProcessing()) {
      return;
    }
    bank.setState(ElectronicAccountBankState.UNBOUND.getState());
    electronicAccountBankRepository.saveAndFlush(bank);
    electronicAccountService.updateTiedCardStatusById(bank.getElectronicAccount().getId(), ElectronicAccountTiedCardStatus.UNTIED_CARD.getStatus());
  }

  @Override
  @Transactional
  public void bindByElectronicAccountId(String electronicAccountId) {
    Validate.notBlank(electronicAccountId, "电子账户ID不能为空");
    ElectronicAccountBank bank = electronicAccountBankRepository.findByElectronicAccountIdAndState(electronicAccountId, ElectronicAccountBankState.WAITING_VALID.getState());
    Validate.notNull(bank, "没有待绑定的银行卡");
    bank.setState(ElectronicAccountBankState.ENABLE.getState());
    electronicAccountBankRepository.save(bank);
  }

  @Override
  public void bindByElectronicAccountId(String electronicAccountId, String validCode) {
    Validate.notBlank(electronicAccountId, "电子账户ID不能为空");
    Validate.notBlank(validCode, "验证码不能为空");
    ElectronicAccountBank bank = electronicAccountBankRepository.findByElectronicAccountIdAndState(electronicAccountId, ElectronicAccountBankState.WAITING_VALID.getState());
    Validate.notNull(bank, "没有待绑定的银行卡");
    this.bindById(bank.getId(), validCode);
  }

  @Override
  public ElectronicAccountBank findEnableByElectronicAccountId(String electronicAccountId) {
    if (StringUtils.isBlank(electronicAccountId)) {
      return null;
    }
    return electronicAccountBankRepository.findByElectronicAccountIdAndState(electronicAccountId, ElectronicAccountBankState.ENABLE.getState());
  }

  @Override
  public ElectronicAccountBank findEnableByTenant() {
    String tenantCode = TenantUtils.getTenantCode();
    return electronicAccountBankRepository.findByTenantCodeAndState(tenantCode, ElectronicAccountBankState.ENABLE.getState());
  }

  /**
   * 在保存前做边界校验
   */
  private void saveValidation(ElectronicAccount account, ElectronicAccountBank bank) {
    Validate.notNull(account, "电子账户不能为空");
    Validate.notBlank(account.getId(), "电子账户ID不能为空");
    Validate.notNull(account.getType(), "电子账户类型不能为空");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(bank.getBankCardNo(), "银行卡号不能为空！");
    Validate.notBlank(bank.getBankName(), "开户银行不能为空！");
    Validate.notBlank(bank.getBankId(), "银行卡ID不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(bank.getManagementName() == null || bank.getManagementName().length() <= 32, "经营者姓名填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getIdCardNo() == null || bank.getIdCardNo().length() < 32, "身份证号填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getBankCardAccountName() == null || bank.getBankCardAccountName().length() <= 64, "银行卡账户名称填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(bank.getBankCardNo() == null || bank.getBankCardNo().length() <= 32, "银行卡号填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getBankName() == null || bank.getBankName().length() <= 32, "开户银行填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getBankId() == null || bank.getBankId().length() <= 32, "银行卡ID填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getBankBranchName() == null || bank.getBankBranchName().length() <= 32, "分支行名称填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getMobile() == null || bank.getMobile().length() <= 11, "银行卡预留手机号填入值超过了限定长度(11)，请检查!");
    Validate.isTrue(bank.getProvinceCode() == null || bank.getProvinceCode().length() <= 32, "省编码填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getProvinceName() == null || bank.getProvinceName().length() <= 32, "省名称填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getCityCode() == null || bank.getCityCode().length() <= 32, "市编码填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getCityName() == null || bank.getCityName().length() <= 32, "市名称填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getDistrictCode() == null || bank.getDistrictCode().length() <= 32, "区编码填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(bank.getDistrictName() == null || bank.getDistrictName().length() <= 32, "区名称填入值超过了限定长度(32)，请检查!");

    ElectronicAccountType accountType = ElectronicAccountType.valueOfType(account.getType());
    Validate.notNull(accountType, "不支持的电子账户类型：%s", account.getType());
    if (ElectronicAccountType.INDIVIDUAL.equals(accountType)) {
      Validate.notBlank(bank.getManagementName(), "经营者姓名不能为空");
      Validate.notBlank(bank.getIdCardNo(), "身份证号不能为空");
      Validate.notBlank(bank.getMobile(), "银行卡预留手机号不能为空！");
    } else if (ElectronicAccountType.COMPANY.equals(accountType)) {
      Validate.notBlank(bank.getBankBranchName(), "分支行名称不能为空！");
      Validate.notBlank(bank.getBankCardAccountName(), "银行卡账户名称不能为空");
      Validate.notBlank(bank.getProvinceCode(), "省编码不能为空！");
      Validate.notBlank(bank.getProvinceName(), "省名称不能为空！");
      Validate.notBlank(bank.getCityCode(), "市编码不能为空！");
      Validate.notBlank(bank.getCityName(), "市名称不能为空！");
    }
  }

}
