package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.common.util.DateUtils;
import com.bizunited.empower.business.payment.common.enums.ElAccountBillType;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountState;
import com.bizunited.empower.business.payment.entity.ElectronicAccount;
import com.bizunited.empower.business.payment.entity.ElectronicAccountBillDayStatistics;
import com.bizunited.empower.business.payment.repository.ElectronicAccountBillDayStatisticsRepository;
import com.bizunited.empower.business.payment.service.ElectronicAccountBillDayStatisticsService;
import com.bizunited.empower.business.payment.service.ElectronicAccountBillService;
import com.bizunited.empower.business.payment.service.ElectronicAccountService;
import com.bizunited.platform.common.util.tenant.TenantContextHolder;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.datasource.service.AppDataSourceService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * 电子账户每日统计数据服务
 * @Author: Paul Chan
 * @Date: 2021/4/25 下午2:17
 */
@Service
public class ElectronicAccountBillDayStatisticsServiceImpl implements ElectronicAccountBillDayStatisticsService {
  private static final Logger LOGGER = LoggerFactory.getLogger(ElectronicAccountBillDayStatisticsServiceImpl.class);

  @Autowired
  private ElectronicAccountBillDayStatisticsRepository electronicAccountBillDayStatisticsRepository;

  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private AppDataSourceService appDataSourceService;
  @Autowired
  private ElectronicAccountService electronicAccountService;
  @Autowired
  private ElectronicAccountBillService electronicAccountBillService;

  @Override
  public Page<ElectronicAccountBillDayStatistics> findByConditions(Map<String, Object> conditions, Pageable pageable) {
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    conditions = ObjectUtils.defaultIfNull(conditions, Maps.newHashMap());
    String appCode = TenantUtils.getAppCode();
    conditions.put("tenantCode", appCode);
    return electronicAccountBillDayStatisticsRepository.findByConditions(conditions, pageable);
  }

  @Override
  public void sumYesterdayAmount() {
    LOGGER.info("开始统计昨日电子账户");
    List<String> appCodes = appDataSourceService.findAppCodeByTstatus();
    if(CollectionUtils.isEmpty(appCodes)){
      return;
    }
    ElectronicAccountBillDayStatisticsService statisticsService = applicationContext.getBean(ElectronicAccountBillDayStatisticsService.class);
    for (String appCode : appCodes) {
      try {
        LOGGER.info("开始统计昨日电子账户的经销商：{}", appCode);
        TenantContextHolder.setApp(appCode);
        statisticsService.sumYesterdayAmountByApp();
      } finally {
        TenantContextHolder.clean();
      }
    }
    LOGGER.info("结束统计昨日电子账户");
  }

  @Override
  @Transactional
  public void sumYesterdayAmountByApp() {
    List<ElectronicAccount> accounts = electronicAccountService.findAll();
    if(CollectionUtils.isEmpty(accounts)) {
      return;
    }
    for (ElectronicAccount account : accounts) {
      LOGGER.info("开始统计经销商【{}】的电子账户【{}】的流水信息", account.getTenantCode(), account.getId());
      this.sumYesterdayAmountByAccount(account);
    }
  }

  @Override
  public List<ElectronicAccountBillDayStatistics> findByDays(Integer days) {
    days = ObjectUtils.defaultIfNull(days, 10);
    Pageable pageable = PageRequest.of(0, days);
    Page<ElectronicAccountBillDayStatistics> page = this.findByConditions(null, pageable);
    List<ElectronicAccountBillDayStatistics> statistics = page.getContent();
    statistics = ObjectUtils.defaultIfNull(statistics, Lists.newArrayList());
    statistics = Lists.newArrayList(statistics);
    this.handleStatistics(statistics, days);
    Collections.sort(statistics, Comparator.comparing(ElectronicAccountBillDayStatistics::getDay));
    return statistics;
  }

  /**
   * 处理最近时间的数据，如果不够则增加空数据
   * @param days
   */
  private void handleStatistics(List<ElectronicAccountBillDayStatistics> statistics, Integer days) {
    if(statistics.size() == days) {
      return;
    }
    SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
    Set<String> statisticsDays = statistics.stream().map(s -> dateFormat.format(s.getDay())).collect(Collectors.toSet());
    Calendar calendar = Calendar.getInstance();
    calendar.add(Calendar.DAY_OF_MONTH, -1);
    for (Integer i = 0; i < days; i++) {
      Date day = calendar.getTime();
      String formatDay = dateFormat.format(day);
      if(!statisticsDays.contains(formatDay)) {
        ElectronicAccountBillDayStatistics dayStatistics = new ElectronicAccountBillDayStatistics();
        dayStatistics.setDay(day);
        dayStatistics.setReceiptAmount(BigDecimal.ZERO);
        dayStatistics.setPaymentAmount(BigDecimal.ZERO);
        statistics.add(dayStatistics);
      }
      calendar.add(Calendar.DAY_OF_MONTH, -1);
    }
  }

  /**
   * 统计电子账户昨天的电子账户信息
   * @param account
   */
  private void sumYesterdayAmountByAccount(ElectronicAccount account) {
    if(ElectronicAccountState.APPLYING.getState().equals(account.getState())) {
      return;
    }
    Date startDate = DateUtils.getYesterdayStartDate();
    Date endDate = DateUtils.getYesterdayEndDate();
    long count = electronicAccountBillDayStatisticsRepository.countByAccountIdAndDay(account.getId(), startDate);
    if(count > 0) {
      return;
    }
    BigDecimal yesterdayReceiptAmount = electronicAccountBillService.sumRealAmountByAccountIdAndTypeAndModifyTimeBetween(account.getId(), ElAccountBillType.RECEIPT.getType(), startDate, endDate);
    BigDecimal yesterdayPaymentAmount = electronicAccountBillService.sumRealAmountByAccountIdAndTypeAndModifyTimeBetween(account.getId(), ElAccountBillType.PAYMENT.getType(), startDate, endDate);
    ElectronicAccountBillDayStatistics statistics = new ElectronicAccountBillDayStatistics();
    statistics.setDay(startDate);
    statistics.setReceiptAmount(yesterdayReceiptAmount);
    statistics.setPaymentAmount(yesterdayPaymentAmount);
    statistics.setElectronicAccount(account);
    electronicAccountBillDayStatisticsRepository.save(statistics);
  }

}
