package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.payment.common.enums.ElAccountBillBusinessType;
import com.bizunited.empower.business.payment.common.enums.ElAccountBillChargeRateType;
import com.bizunited.empower.business.payment.common.enums.ElAccountBillState;
import com.bizunited.empower.business.payment.common.enums.ElAccountBillType;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountState;
import com.bizunited.empower.business.payment.dto.TransferNoticeDto;
import com.bizunited.empower.business.payment.entity.ElectronicAccount;
import com.bizunited.empower.business.payment.entity.ElectronicAccountBill;
import com.bizunited.empower.business.payment.repository.ElectronicAccountBillRepository;
import com.bizunited.empower.business.payment.service.ElectronicAccountBillService;
import com.bizunited.empower.business.payment.service.ElectronicAccountService;
import com.bizunited.platform.common.service.CodeGeneratorService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.constant.Constants.EL_ACCOUNT_BILL_BUSINESS_TYPE_PAYMENT;
import static com.bizunited.empower.business.payment.common.constant.Constants.EL_ACCOUNT_BILL_BUSINESS_TYPE_RECEIPT;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.TENANT_EL_ACCOUNT_BILL_BUSINESS_LOCK_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.TENANT_EL_ACCOUNT_BILL_LOCK_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.TENANT_EL_ACCOUNT_BILL_NO_KEY;
import static com.bizunited.empower.business.payment.common.enums.ElAccountBillType.PAYMENT;
import static com.bizunited.empower.business.payment.common.enums.ElAccountBillType.RECEIPT;
import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * 电子账户流水记录服务实现
 *
 * @Author: Paul Chan
 * @Date: 2021/4/20 上午11:48
 */
@Service
public class ElectronicAccountBillServiceImpl implements ElectronicAccountBillService {

  @Autowired
  private ElectronicAccountBillRepository electronicAccountBillRepository;

  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private CodeGeneratorService codeGeneratorService;
  @Autowired
  private ElectronicAccountService electronicAccountService;

  @Override
  @Transactional
  public ElectronicAccountBill create(String accountId, ElAccountBillState state, ElAccountBillBusinessType businessType,
                                      String businessNo, BigDecimal amount, ElAccountBillChargeRateType chargeRateType,
                                      BigDecimal chargeRate, BigDecimal chargeAmount, BigDecimal realAmount, String dealFrom, String remark) {
    this.createValidation(accountId, state, businessType, businessNo, amount, chargeRateType, chargeRate, chargeAmount, realAmount, dealFrom, remark);
    ElectronicAccount account = electronicAccountService.findById(accountId);
    Validate.notNull(account, "未找到电子账户：%s", accountId);
    Validate.isTrue(!ElectronicAccountState.APPLYING.getState().equals(account.getState()), "电子账户未启用");
    Date now = new Date();
    String userAccount = SecurityUtils.getUserAccount();
    ElectronicAccountBill bill = new ElectronicAccountBill();
    bill.setType(this.getTypeByBusinessType(businessType).getType());
    bill.setState(state.getState());
    bill.setBillNo(this.generateBillNo(businessType));
    bill.setBusinessType(businessType.getType());
    bill.setBusinessNo(ObjectUtils.defaultIfNull(businessNo, ""));
    bill.setAmount(amount);
    bill.setChargeRateType(chargeRateType.getType());
    bill.setChargeRate(ObjectUtils.defaultIfNull(chargeRate, BigDecimal.ZERO));
    bill.setChargeAmount(ObjectUtils.defaultIfNull(chargeAmount, BigDecimal.ZERO));
    bill.setRealAmount(realAmount);
    bill.setDealFrom(dealFrom);
    bill.setElectronicAccount(account);
    bill.setCreateTime(now);
    bill.setModifyTime(now);
    bill.setCreateAccount(userAccount);
    bill.setModifyAccount(userAccount);
    bill.setRemark(remark);
    bill.setBalance(account.getBalance());
    return electronicAccountBillRepository.save(bill);
  }

  @Override
  @Transactional
  public List<ElectronicAccountBill> updateStateByBusinessNo(String businessNo, ElAccountBillState state) {
    Validate.notBlank(businessNo, "业务单号不能为空");
    Validate.notNull(state, "更新状态不能为空");
    Validate.isTrue(!state.equals(ElAccountBillState.HANDLING), "不能变更为处理中");
    String tenantCode = TenantUtils.getTenantCode();
    String redisKey = String.format(TENANT_EL_ACCOUNT_BILL_BUSINESS_LOCK_PREFIX, tenantCode, businessNo);
    boolean locked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(locked, "操作繁忙，请稍后再试");
    try {
      List<ElectronicAccountBill> bills = electronicAccountBillRepository.findByTenantCodeAndBusinessNo(TenantUtils.getTenantCode(), businessNo);
      Validate.notEmpty(bills, "未找到业务单号：%s", businessNo);
      for (ElectronicAccountBill bill : bills) {
        this.updateStateByBillNo(bill.getBillNo(), state);
      }
      return bills;
    } finally {
      if (redisMutexService.islock(redisKey)) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  @Override
  @Transactional
  public ElectronicAccountBill updateStateByBillNo(String billNo, ElAccountBillState state) {
    Validate.notBlank(billNo, "流水单号不能为空");
    Validate.notNull(state, "更新状态不能为空");
    Validate.isTrue(!state.equals(ElAccountBillState.HANDLING), "不能变更为处理中");
    String tenantCode = TenantUtils.getTenantCode();
    String redisKey = String.format(TENANT_EL_ACCOUNT_BILL_LOCK_PREFIX, tenantCode, billNo);
    boolean locked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(locked, "操作繁忙，请稍后再试");
    try {
      ElectronicAccountBill bill = electronicAccountBillRepository.findByTenantCodeAndBillNo(tenantCode, billNo);
      Validate.notNull(bill, "未找到流水单：%s", billNo);
      ElectronicAccount account = bill.getElectronicAccount();
      Validate.notNull(account, "未找到流水单的电子账户");
      Validate.isTrue(ElAccountBillState.HANDLING.getState().equals(bill.getState()), "流水单号【%s】不是处理中状态，不能变更状态", billNo);
      bill.setState(state.getState());
      bill.setModifyAccount(SecurityUtils.getUserAccount());
      bill.setModifyTime(new Date());
      electronicAccountBillRepository.save(bill);
      if (state.equals(ElAccountBillState.FAILURE)) {
        electronicAccountService.addBalanceByAccountId(account.getId(), bill.getRealAmount());
      }
      return bill;
    } finally {
      if (redisMutexService.islock(redisKey)) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  @Override
  @Transactional
  public void handleExtractCashMq(TransferNoticeDto transferNoticeDto) {
    Validate.notNull(transferNoticeDto, "通知内容不能为空");
    Validate.notBlank(transferNoticeDto.getTxSN(), "流水号信息不能为空");
    Integer status = transferNoticeDto.getStatus();
    Validate.notNull(status, "提现状态不能为空");
    ElAccountBillState billState;
    if(status == 2) {
      billState = ElAccountBillState.SUCCESS;
    } else if(status == 3) {
      billState = ElAccountBillState.FAILURE;
    } else {
      throw new IllegalStateException(String.format("未知的处理状态:%s", status));
    }
    this.updateStateByBusinessNo(transferNoticeDto.getTxSN(), billState);
  }

  @Override
  public Page<ElectronicAccountBill> findByConditions(Map<String, Object> conditions, Pageable pageable) {
    conditions = ObjectUtils.defaultIfNull(conditions, Maps.newHashMap());
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    conditions.put("tenantCode", TenantUtils.getTenantCode());
    return electronicAccountBillRepository.findByConditions(conditions, pageable);
  }

  @Override
  public BigDecimal sumRealAmountByTypeAndModifyTimeBetween(Integer type, Date startDate, Date endDate) {
    if (!ObjectUtils.allNotNull(type, startDate, endDate)) {
      return BigDecimal.ZERO;
    }
    String tenantCode = TenantUtils.getTenantCode();
    BigDecimal amount = electronicAccountBillRepository.sumRealAmountByTenantCodeAndTypeAndModifyTimeBetween(tenantCode, type, startDate, endDate);
    return ObjectUtils.defaultIfNull(amount, BigDecimal.ZERO);
  }

  @Override
  public BigDecimal sumRealAmountByAccountIdAndTypeAndModifyTimeBetween(String accountId, Integer type, Date startDate, Date endDate) {
    if (!ObjectUtils.allNotNull(type, startDate, endDate) || StringUtils.isBlank(accountId)) {
      return BigDecimal.ZERO;
    }
    BigDecimal amount = electronicAccountBillRepository.sumRealAmountByTenantCodeAndAccountIdAndTypeAndModifyTimeBetween(accountId, type, startDate, endDate);
    return ObjectUtils.defaultIfNull(amount, BigDecimal.ZERO);
  }

  /**
   * 生成流水编号
   *
   * @param businessType
   * @return
   */
  private String generateBillNo(ElAccountBillBusinessType businessType) {
    String appCode = TenantUtils.getTenantCode();
    String date = new SimpleDateFormat("yyyyMMdd").format(new Date());
    String key = String.format(TENANT_EL_ACCOUNT_BILL_NO_KEY, appCode, date);
    String prefix = StringUtils.join(businessType.getPrefix(), date);
    return codeGeneratorService.generate(key, prefix, null, 1, 6, 25, TimeUnit.HOURS);
  }

  /**
   * 根据业务类型获取账单类型
   *
   * @param businessType
   * @return
   */
  private ElAccountBillType getTypeByBusinessType(ElAccountBillBusinessType businessType) {
    if (EL_ACCOUNT_BILL_BUSINESS_TYPE_RECEIPT.contains(businessType)) {
      return RECEIPT;
    }
    if (EL_ACCOUNT_BILL_BUSINESS_TYPE_PAYMENT.contains(businessType)) {
      return PAYMENT;
    }
    throw new IllegalArgumentException("不支持的业务类型");
  }

  /**
   * 校验入参
   *
   * @param accountId
   * @param state
   * @param businessType
   * @param businessNo
   * @param amount
   * @param chargeRate
   * @param realAmount
   * @param dealFrom
   */
  private void createValidation(String accountId, ElAccountBillState state, ElAccountBillBusinessType businessType,
                                String businessNo, BigDecimal amount, ElAccountBillChargeRateType chargeRateType, BigDecimal chargeRate, BigDecimal chargeAmount,
                                BigDecimal realAmount, String dealFrom, String remark) {
    Validate.notBlank(accountId, "电子账户ID不能为空");
    Validate.notNull(state, "流水状态不能为空");
    Validate.notNull(chargeRateType, "手续费率类型不能为空");
    Validate.notNull(businessType, "业务类型不能为空");
    Validate.notNull(amount, "交易金额不能为空");
    Validate.notNull(realAmount, "实际收支金额不能为空");
    Validate.isTrue(amount.compareTo(BigDecimal.ZERO) > 0, "交易金额要大于0");
    Validate.isTrue(realAmount.compareTo(BigDecimal.ZERO) >= 0, "实际收支金额要大于0");
    Validate.isTrue(chargeRate == null || chargeRate.compareTo(BigDecimal.ZERO) >= 0, "手续费率不能小于0");
    Validate.isTrue(chargeAmount == null || chargeAmount.compareTo(BigDecimal.ZERO) >= 0, "手续费不能小于0");
    Validate.isTrue(businessNo == null || businessNo.length() <= 32, "业务单号长度不能超过32");
    Validate.isTrue(dealFrom == null || dealFrom.length() <= 255, "交易来源长度不能超过255");
    Validate.isTrue(remark == null || remark.length() <= 255, "备注长度不能超过255");
  }

}
