package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.payment.common.enums.ElectronicAccountType;
import com.bizunited.empower.business.payment.entity.ElectronicAccount;
import com.bizunited.empower.business.payment.entity.ElectronicAccountCertificate;
import com.bizunited.empower.business.payment.repository.ElectronicAccountCertificateRepository;
import com.bizunited.empower.business.payment.service.ElectronicAccountCertificateService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;

/**
 * ElectronicAccountCertificate业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ElectronicAccountCertificateServiceImpl")
public class ElectronicAccountCertificateServiceImpl implements ElectronicAccountCertificateService {

  @Autowired
  private ElectronicAccountCertificateRepository electronicAccountCertificateRepository;

  @Override
  @Transactional
  public ElectronicAccountCertificate create(ElectronicAccountCertificate electronicAccountCertificate) {
    ElectronicAccountCertificate current = this.createForm(electronicAccountCertificate);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Override
  @Transactional
  public ElectronicAccountCertificate createForm(ElectronicAccountCertificate electronicAccountCertificate) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.createValidation(electronicAccountCertificate);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.electronicAccountCertificateRepository.save(electronicAccountCertificate);

    // 返回最终处理的结果，里面带有详细的关联信息
    return electronicAccountCertificate;
  }

  /**
   * 在创建一个新的ElectronicAccountCertificate模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ElectronicAccountCertificate electronicAccountCertificate) {
    Validate.notNull(electronicAccountCertificate, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(electronicAccountCertificate.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    electronicAccountCertificate.setId(null);
    this.basicsValidation(electronicAccountCertificate);
  }

  @Override
  @Transactional
  public ElectronicAccountCertificate update(ElectronicAccountCertificate certificate) {
    ElectronicAccountCertificate current = this.updateForm(certificate);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Override
  @Transactional
  public ElectronicAccountCertificate updateForm(ElectronicAccountCertificate certificate) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(certificate);
    // ===================基本信息
    String currentId = certificate.getId();
    ElectronicAccountCertificate currentCertificate = this.electronicAccountCertificateRepository.findById(currentId).orElse(null);
    currentCertificate = Validate.notNull(currentCertificate, "未发现指定的原始模型对象信");
    // 开始重新赋值——一般属性
    currentCertificate.setBusinessLicensePath(certificate.getBusinessLicensePath());
    currentCertificate.setBusinessLicenseFilename(certificate.getBusinessLicenseFilename());
    currentCertificate.setBusinessNo(certificate.getBusinessNo());
    currentCertificate.setBusinessShopName(certificate.getBusinessShopName());
    currentCertificate.setBusinessLicenseIssueDate(certificate.getBusinessLicenseIssueDate());
    currentCertificate.setBusinessLicenseExpireDate(certificate.getBusinessLicenseExpireDate());
    currentCertificate.setOrgType(certificate.getOrgType());
    currentCertificate.setAddress(certificate.getAddress());
    currentCertificate.setBusinessScope(certificate.getBusinessScope());
    currentCertificate.setIdCardFrontPath(certificate.getIdCardFrontPath());
    currentCertificate.setIdCardFrontFilename(certificate.getIdCardFrontFilename());
    currentCertificate.setIdCardBackPath(certificate.getIdCardBackPath());
    currentCertificate.setIdCardBackFilename(certificate.getIdCardBackFilename());
    currentCertificate.setIdCardNo(certificate.getIdCardNo());
    currentCertificate.setIdCardIssueDate(certificate.getIdCardIssueDate());
    currentCertificate.setIdCardExpireDate(certificate.getIdCardExpireDate());
    currentCertificate.setSocialCreditCode(certificate.getSocialCreditCode());
    currentCertificate.setRegisteredCapital(certificate.getRegisteredCapital());
    currentCertificate.setLegalPersonName(certificate.getLegalPersonName());
    currentCertificate.setLegalPersonMobile(certificate.getLegalPersonMobile());
    currentCertificate.setLegalPersonEmail(certificate.getLegalPersonEmail());

    this.electronicAccountCertificateRepository.saveAndFlush(currentCertificate);
    return currentCertificate;
  }

  /**
   * 在更新一个已有的ElectronicAccountCertificate模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ElectronicAccountCertificate certificate) {
    Validate.notNull(certificate, "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(!StringUtils.isBlank(certificate.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    this.basicsValidation(certificate);
  }

  /**
   * 基础信息验证
   * @param certificate
   */
  private void basicsValidation(ElectronicAccountCertificate certificate) {
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(certificate.getIdCardNo(), "身份证号不能为空！");
    Validate.notNull(certificate.getIdCardIssueDate(), "身份证发证日期不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(certificate.getBusinessLicensePath() == null || certificate.getBusinessLicensePath().length() < 64, "营业执照图片文件夹填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getBusinessLicenseFilename() == null || certificate.getBusinessLicenseFilename().length() < 64, "营业执照图片文件名填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getBusinessNo() == null || certificate.getBusinessNo().length() < 64, "个体工商注册号码填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getBusinessShopName() == null || certificate.getBusinessShopName().length() < 64, "个体工商字号填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getOrgType() == null || certificate.getOrgType().length() < 11, "形成形式填入值超过了限定长度(11)，请检查!");
    Validate.isTrue(certificate.getAddress() == null || certificate.getAddress().length() < 64, "经营地址填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getBusinessScope() == null || certificate.getBusinessScope().length() < 255, "经营范围填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(certificate.getIdCardFrontPath() == null || certificate.getIdCardFrontPath().length() < 64, "身份证正面文件夹填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getIdCardFrontFilename() == null || certificate.getIdCardFrontFilename().length() < 64, "身份证正面文件名填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getIdCardBackPath() == null || certificate.getIdCardBackPath().length() < 64, "身份证背面文件夹填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getIdCardBackFilename() == null || certificate.getIdCardBackFilename().length() < 64, "身份证背面文件名填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getIdCardNo() == null || certificate.getIdCardNo().length() < 32, "身份证号填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(certificate.getSocialCreditCode() == null || certificate.getSocialCreditCode().length() < 64, "统一社会信用代码填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(certificate.getLegalPersonName() == null || certificate.getLegalPersonName().length() < 32, "法人姓名填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(certificate.getLegalPersonMobile() == null || certificate.getLegalPersonMobile().length() < 32, "法人手机号填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(certificate.getLegalPersonEmail() == null || certificate.getLegalPersonEmail().length() < 32, "法人邮箱填入值超过了限定长度(32)，请检查!");

    ElectronicAccount account = certificate.getElectronicAccount();
    Validate.notNull(account, "电子账户不能为空");
    Validate.notNull(account.getType(), "电子账户类型不能为空");
    ElectronicAccountType accountType = ElectronicAccountType.valueOfType(account.getType());
    Validate.notNull(accountType,  "不支持的电子账户类型：%s", account.getType());
    if (ElectronicAccountType.INDIVIDUAL.equals(accountType)) {
      this.individualValidation(certificate);
    } else if (ElectronicAccountType.COMPANY.equals(accountType)) {
      this.companyValidation(certificate);
    }
  }

  /**
   * 个体工商户信息校验
   * @param certificate
   */
  private void individualValidation(ElectronicAccountCertificate certificate) {
    Validate.notNull(certificate.getBusinessLicensePath(), "营业执照不能为空");
    Validate.notBlank(certificate.getBusinessLicenseFilename(), "营业执照不能为空");
    Validate.notBlank(certificate.getBusinessNo(), "个体工商注册号码不能为空");
    Validate.notBlank(certificate.getBusinessShopName(), "个体工商字号不能为空");
    Validate.notBlank(certificate.getOrgType(), "形成形式不能为空");
    Validate.notBlank(certificate.getAddress(), "经营地址不能为空");
    Validate.notBlank(certificate.getBusinessScope(), "经营范围不能为空");
    Validate.notNull(certificate.getIdCardFrontPath(), "身份证正面不能为空");
    Validate.notBlank(certificate.getIdCardFrontFilename(), "身份证正面不能为空");
    Validate.notNull(certificate.getIdCardBackPath(), "身份证背面不能为空");
    Validate.notBlank(certificate.getIdCardBackFilename(), "身份证背面不能为空");
    Validate.notBlank(certificate.getIdCardNo(), "身份证号不能为空");
    Validate.notNull(certificate.getIdCardIssueDate(), "身份证发证日期不能为空");
  }

  /**
   * 企业信息校验
   * @param certificate
   */
  private void companyValidation(ElectronicAccountCertificate certificate) {
    Validate.notNull(certificate.getBusinessLicensePath(), "营业执照不能为空");
    Validate.notBlank(certificate.getBusinessLicenseFilename(), "营业执照不能为空");
    Validate.notBlank(certificate.getSocialCreditCode(), "统一社会信用代码不能为空");
    Validate.notNull(certificate.getRegisteredCapital(), "注册资本不能为空");
    Validate.isTrue(certificate.getRegisteredCapital().compareTo(BigDecimal.ZERO) > 0, "注册资本不能小于零");
    Validate.notBlank(certificate.getLegalPersonName(), "法人姓名不能为空");
    Validate.notBlank(certificate.getLegalPersonMobile(), "法人手机号不能为空");
    Validate.notBlank(certificate.getLegalPersonEmail(), "法人邮箱不能为空");
    Validate.notBlank(certificate.getIdCardNo(), "身份证号不能为空");
    Validate.notNull(certificate.getIdCardIssueDate(), "身份证发证日期不能为空");
    Validate.notNull(certificate.getBusinessLicenseIssueDate(), "营业执照发证日期不能为空");
  }


} 
