package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.common.service.SignService;
import com.bizunited.empower.business.common.util.ResponseModelUtils;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountType;
import com.bizunited.empower.business.payment.dto.CpcnResultDto;
import com.bizunited.empower.business.payment.dto.ElectronicAccountCertificateDto;
import com.bizunited.empower.business.payment.dto.ElectronicAccountDto;
import com.bizunited.empower.business.payment.dto.SecurePhoneDto;
import com.bizunited.empower.business.payment.dto.SignDto;
import com.bizunited.empower.business.payment.dto.TiedCardVerifyDto;
import com.bizunited.empower.business.payment.entity.ElectronicAccount;
import com.bizunited.empower.business.payment.feign.ElectronicAccountFeignClient;
import com.bizunited.empower.business.payment.feign.ElectronicAccountInfoFeignClient;
import com.bizunited.empower.business.payment.service.ElectronicAccountDtoService;
import com.bizunited.empower.business.payment.service.ElectronicAccountService;
import com.bizunited.empower.business.payment.service.handle.AccountApplyStateHandle;
import com.bizunited.empower.business.tenant.entity.TenantInfo;
import com.bizunited.empower.business.tenant.service.TenantInfoService;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.venus.sdk.service.file.FileHandleService;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Base64Utils;

import javax.transaction.Transactional;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 电子账户dto服务
 * @Author: Paul Chan
 * @Date: 2021/4/14 下午4:20
 */
@Service
public class ElectronicAccountDtoServiceImpl implements ElectronicAccountDtoService {
  private static final Logger LOGGER = LoggerFactory.getLogger(ElectronicAccountDtoServiceImpl.class);

  @Autowired
  private ElectronicAccountFeignClient electronicAccountFeignClient;
  @Autowired
  private ElectronicAccountInfoFeignClient electronicAccountInfoFeignClient;

  @Autowired
  private SignService signService;
  @Autowired
  private FileHandleService venusFileService;
  @Autowired
  private TenantInfoService tenantInfoService;
  @Autowired
  private ElectronicAccountService electronicAccountService;

  @Autowired
  private List<AccountApplyStateHandle> accountApplyStateHandles;

  @Override
  public void create(ElectronicAccountDto electronicAccountDto) {
    Validate.notNull(electronicAccountDto, "创建对象不能为空");
    String tenantCode = electronicAccountDto.getTenantCode();
    TenantInfo tenantInfo = tenantInfoService.findByCode(tenantCode);
    Validate.notNull(tenantInfo, "未找到经销商：%s", tenantCode);
    electronicAccountDto.setAppType(2);
    electronicAccountDto.setClientName(tenantInfo.getTenantName());
    electronicAccountDto.setAppCode(TenantUtils.getAppCode());
    electronicAccountDto.setBusinessId(electronicAccountDto.getId());
    this.handleFileBase64(electronicAccountDto);
    String json = JsonUtils.obj2JsonString(electronicAccountDto);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = electronicAccountInfoFeignClient.create(appId, sign, json);
    ResponseModelUtils.validSuccess(responseModel);
  }

  @Override
  public void update(ElectronicAccountDto electronicAccountDto) {
    Validate.notNull(electronicAccountDto, "更新对象不能为空");
    String tenantCode = electronicAccountDto.getTenantCode();
    TenantInfo tenantInfo = tenantInfoService.findByCode(tenantCode);
    Validate.notNull(tenantInfo, "未找到经销商：%s", tenantCode);
    electronicAccountDto.setClientName(tenantInfo.getTenantName());
    electronicAccountDto.setAppType(2);
    electronicAccountDto.setAppCode(TenantUtils.getAppCode());
    electronicAccountDto.setBusinessId(electronicAccountDto.getId());
    this.handleFileBase64(electronicAccountDto);
    String json = JsonUtils.obj2JsonString(electronicAccountDto);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = electronicAccountInfoFeignClient.update(appId, sign, json);
    ResponseModelUtils.validSuccess(responseModel);
  }

  @Override
  public void updateSecurePhone(ElectronicAccountDto electronicAccountDto) {
    Validate.notNull(electronicAccountDto, "更新对象不能为空");
    SecurePhoneDto dto = new SecurePhoneDto();
    dto.setMobile(electronicAccountDto.getSecurityMobile());
    dto.setBusinessId(electronicAccountDto.getId());
    dto.setTenantCode(electronicAccountDto.getTenantCode());
    String json = JsonUtils.obj2JsonString(dto);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = electronicAccountFeignClient.updateSecurePhone(appId, sign, json);
    ResponseModelUtils.validSuccess(responseModel, "更新失败");
  }

  @Override
  public CpcnResultDto updateForTiedCard(ElectronicAccountDto electronicAccountDto) {
    Validate.notNull(electronicAccountDto, "处理对象不能为空");
    String tenantCode = electronicAccountDto.getTenantCode();
    TenantInfo tenantInfo = tenantInfoService.findByCode(tenantCode);
    Validate.notNull(tenantInfo, "未找到经销商：%s", tenantCode);
    electronicAccountDto.setClientName(tenantInfo.getTenantName());
    electronicAccountDto.setAppType(2);
    electronicAccountDto.setAppCode(TenantUtils.getAppCode());
    electronicAccountDto.setBusinessId(electronicAccountDto.getId());
    String json = JsonUtils.obj2JsonString(electronicAccountDto);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = electronicAccountFeignClient.updateForTiedCard(appId, sign, json);
    ResponseModelUtils.validSuccess(responseModel);
    CpcnResultDto resultDto = ResponseModelUtils.getSuccessData(responseModel, CpcnResultDto.class);
    String msg = resultDto == null || StringUtils.isBlank(resultDto.getMsg()) ? "更新银行卡失败" : resultDto.getMsg();
    Validate.isTrue(resultDto != null && resultDto.getSuccess() != null
        && resultDto.getSuccess(), msg);
    return resultDto;
  }

  @Override
  public CpcnResultDto handleTiedCard(ElectronicAccountDto electronicAccountDto) {
    Validate.notNull(electronicAccountDto, "处理对象不能为空");
    TiedCardVerifyDto dto = new TiedCardVerifyDto();
    dto.setBusinessId(electronicAccountDto.getId());
    dto.setTenantCode(electronicAccountDto.getTenantCode());
    String json = JsonUtils.obj2JsonString(dto);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = electronicAccountFeignClient.handleTiedCard(appId, sign, json);
    LOGGER.info("申请绑卡返回数据：{}", JsonUtils.obj2JsonString(responseModel));
    ResponseModelUtils.validSuccess(responseModel, "申请绑卡失败");
    CpcnResultDto resultDto = ResponseModelUtils.getSuccessData(responseModel, CpcnResultDto.class);
    String msg = resultDto == null || StringUtils.isBlank(resultDto.getMsg()) ? "申请绑卡失败" : resultDto.getMsg();
    Validate.isTrue(resultDto != null && resultDto.getSuccess() != null
        && resultDto.getSuccess(), msg);
    return resultDto;
  }

  @Override
  @Transactional
  public void handleMq(ElectronicAccountDto electronicAccountDto) {
    Validate.notNull(electronicAccountDto, "处理对象不能为空");
    Validate.notBlank(electronicAccountDto.getBusinessId(), "业务ID不能为空");
    Validate.notNull(electronicAccountDto.getApplyState(), "申请状态不能为空");
    Validate.notNull(electronicAccountDto.getTiedCardStatus(), "开卡状态不能为空");
    ElectronicAccount electronicAccount = electronicAccountService.findDetailsById(electronicAccountDto.getBusinessId());
    Validate.notNull(electronicAccount, "未找到需要处理的电子账户信息:%s", electronicAccountDto.getBusinessId());
    List<AccountApplyStateHandle> handles = accountApplyStateHandles.stream().filter(h -> electronicAccountDto.getApplyState().equals(h.getState())).collect(Collectors.toList());
    Validate.notEmpty(handles, "没有对应的处理器");
    AccountApplyStateHandle handle = handles.get(0);
    handle.handle(electronicAccountDto, electronicAccount);
  }

  @Override
  public void sendSignValidCode(ElectronicAccountDto electronicAccountDto, Boolean isUpdate) {
    Validate.notNull(electronicAccountDto, "处理对象不能为空");
    SignDto signDto = new SignDto();
    signDto.setBusinessId(electronicAccountDto.getId());
    signDto.setTenantCode(electronicAccountDto.getTenantCode());
    this.handleUpdateSignData(signDto, electronicAccountDto, isUpdate);
    String json = JsonUtils.obj2JsonString(signDto);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = electronicAccountFeignClient.handleSign(appId, sign, json);
    ResponseModelUtils.validSuccess(responseModel);
  }

  /**
   * 处理签约处理的数据
   * @param signDto
   * @param electronicAccountDto
   * @param isUpdate
   */
  private void handleUpdateSignData(SignDto signDto, ElectronicAccountDto electronicAccountDto, Boolean isUpdate) {
    signDto.setIsUpdate(isUpdate);
    if(isUpdate == null || !isUpdate) {
      return;
    }
    signDto.setName(electronicAccountDto.getSignerName());
    signDto.setIdCardNo(electronicAccountDto.getSignerIdCardNo());
    signDto.setMobile(electronicAccountDto.getSignerMobile());
  }

  @Override
  public CpcnResultDto bindBank(ElectronicAccountDto electronicAccountDto, String validCode) {
    Validate.notNull(electronicAccountDto, "处理对象不能为空");
    Validate.notBlank(validCode, "验证信息不能为空");
    TiedCardVerifyDto dto = new TiedCardVerifyDto();
    dto.setBusinessId(electronicAccountDto.getId());
    if(ElectronicAccountType.INDIVIDUAL.getType().equals(electronicAccountDto.getType())) {
      dto.setSmsCode(validCode);
    } else {
      dto.setAmountReceived(validCode);
    }
    dto.setTenantCode(electronicAccountDto.getTenantCode());
    String json = JsonUtils.obj2JsonString(dto);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = electronicAccountFeignClient.handleTiedCardVerify(appId, sign, json);
    LOGGER.info("绑卡返回数据：{}", JsonUtils.obj2JsonString(responseModel));
    ResponseModelUtils.validSuccess(responseModel, "绑卡失败");
    CpcnResultDto resultDto = ResponseModelUtils.getSuccessData(responseModel, CpcnResultDto.class);
    String msg = resultDto == null || StringUtils.isBlank(resultDto.getMsg()) ? "绑卡失败" : resultDto.getMsg();
    Validate.isTrue(resultDto != null && resultDto.getSuccess() != null
        && resultDto.getSuccess(), msg);
    return resultDto;
  }

  @Override
  public CpcnResultDto unbindBank(ElectronicAccountDto electronicAccountDto) {
    Validate.notNull(electronicAccountDto, "处理对象不能为空");
    TiedCardVerifyDto dto = new TiedCardVerifyDto();
    dto.setBusinessId(electronicAccountDto.getId());
    dto.setTenantCode(electronicAccountDto.getTenantCode());
    String json = JsonUtils.obj2JsonString(dto);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = electronicAccountFeignClient.handleUntieCardVerify(appId, sign, json);
    LOGGER.info("解绑银行卡返回数据：{}", JsonUtils.obj2JsonString(responseModel));
    ResponseModelUtils.validSuccess(responseModel, "解绑失败");
    CpcnResultDto resultDto = ResponseModelUtils.getSuccessData(responseModel, CpcnResultDto.class);
    String msg = resultDto == null || StringUtils.isBlank(resultDto.getMsg()) ? "绑卡失败" : resultDto.getMsg();
    Validate.isTrue(resultDto != null && resultDto.getSuccess() != null
            && resultDto.getSuccess(), msg);
    return resultDto;
  }

  @Override
  public String findSignUrl(ElectronicAccountDto electronicAccountDto, String validCode) {
    Validate.notNull(electronicAccountDto, "处理对象不能为空");
    Validate.notBlank(validCode, "验证信息不能为空");
    SignDto signDto = new SignDto();
    signDto.setBusinessId(electronicAccountDto.getId());
    signDto.setVerifyCode(validCode);
    signDto.setTenantCode(electronicAccountDto.getTenantCode());
    String json = JsonUtils.obj2JsonString(signDto);
    String appId = signService.getCurrentAppId();
    String sign = signService.sign(json);
    ResponseModel responseModel = electronicAccountFeignClient.findVerifyAddress(appId, sign, json);
    return ResponseModelUtils.getSuccessData(responseModel, String.class);
  }

  /**
   * 对文件做base64处理
   * @param electronicAccountDto
   */
  private void handleFileBase64(ElectronicAccountDto electronicAccountDto) {
    ElectronicAccountCertificateDto certificate = electronicAccountDto.getCertificate();
    if(certificate == null) {
      return;
    }
    String businessLicenseFile = this.getFileBase64(certificate.getBusinessLicensePath(), certificate.getBusinessLicenseFilename());
    Validate.notBlank(businessLicenseFile, "营业执照文件未找到");
    String idCardFrontFile = this.getFileBase64(certificate.getIdCardFrontPath(), certificate.getIdCardFrontFilename());
    Validate.notBlank(idCardFrontFile, "身份证国徽面文件未找到");
    String idCardBackFile = this.getFileBase64(certificate.getIdCardBackPath(), certificate.getIdCardBackFilename());
    Validate.notBlank(idCardBackFile, "身份证人像面文件未找到");
    certificate.setBusinessLicenseFile(businessLicenseFile);
    certificate.setIdCardFrontFile(idCardFrontFile);
    certificate.setIdCardBackFile(idCardBackFile);
  }

  /**
   * 读取文件base64
   * @param relativePath
   * @param fileRename
   * @return
   */
  private String getFileBase64(String relativePath, String fileRename) {
    if(StringUtils.isBlank(fileRename) || relativePath == null) {
      return StringUtils.EMPTY;
    }
    byte[] bytes = venusFileService.findContentByFilePathAndFileRename(relativePath, fileRename);
    if(bytes == null  || bytes.length == 0) {
      return StringUtils.EMPTY;
    }
    return Base64Utils.encodeToString(bytes);
  }

}
