package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.payment.common.enums.ElectronicAccountType;
import com.bizunited.empower.business.payment.entity.ElectronicAccount;
import com.bizunited.empower.business.payment.entity.ElectronicAccountInfo;
import com.bizunited.empower.business.payment.repository.ElectronicAccountInfoRepository;
import com.bizunited.empower.business.payment.service.ElectronicAccountInfoService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;

/**
 * ElectronicAccountInfo业务模型的服务层接口实现
 * @author saturn
 */
@Service("ElectronicAccountInfoServiceImpl")
public class ElectronicAccountInfoServiceImpl implements ElectronicAccountInfoService {

  @Autowired
  private ElectronicAccountInfoRepository electronicAccountInfoRepository;

  @Override
  @Transactional
  public ElectronicAccountInfo create(ElectronicAccountInfo electronicAccountInfo) {
    ElectronicAccountInfo current = this.createForm(electronicAccountInfo);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Override
  @Transactional
  public ElectronicAccountInfo createForm(ElectronicAccountInfo electronicAccountInfo) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    this.createValidation(electronicAccountInfo);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.electronicAccountInfoRepository.save(electronicAccountInfo);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return electronicAccountInfo;
  }
  /**
   * 在创建一个新的ElectronicAccountInfo模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ElectronicAccountInfo electronicAccountInfo) {
    Validate.notNull(electronicAccountInfo , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(electronicAccountInfo.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    electronicAccountInfo.setId(null);

    this.basicsValidation(electronicAccountInfo);
  }

  @Override
  @Transactional
  public ElectronicAccountInfo update(ElectronicAccountInfo electronicAccountInfo) {
    ElectronicAccountInfo current = this.updateForm(electronicAccountInfo);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Override
  @Transactional
  public ElectronicAccountInfo updateForm(ElectronicAccountInfo electronicAccountInfo) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(electronicAccountInfo);
    // ===================基本信息
    String currentId = electronicAccountInfo.getId();
    ElectronicAccountInfo currentElectronicAccountInfo = this.electronicAccountInfoRepository.findById(currentId).orElse(null);
    currentElectronicAccountInfo = Validate.notNull(currentElectronicAccountInfo ,"未发现指定的原始模型对象信");
    // 开始重新赋值——一般属性
    currentElectronicAccountInfo.setManagementName(electronicAccountInfo.getManagementName());
    currentElectronicAccountInfo.setCompanyName(electronicAccountInfo.getCompanyName());
    currentElectronicAccountInfo.setCompanyShortName(electronicAccountInfo.getCompanyShortName());
    currentElectronicAccountInfo.setCategory(electronicAccountInfo.getCategory());
    currentElectronicAccountInfo.setEmail(electronicAccountInfo.getEmail());
    currentElectronicAccountInfo.setAddress(electronicAccountInfo.getAddress());
    currentElectronicAccountInfo.setIndustry(electronicAccountInfo.getIndustry());
    currentElectronicAccountInfo.setGeneralIndustry(electronicAccountInfo.getGeneralIndustry());
    currentElectronicAccountInfo.setScale(electronicAccountInfo.getScale());
    currentElectronicAccountInfo.setApprovalNo(electronicAccountInfo.getApprovalNo());
    currentElectronicAccountInfo.setMobile(electronicAccountInfo.getMobile());
    currentElectronicAccountInfo.setProvinceCode(electronicAccountInfo.getProvinceCode());
    currentElectronicAccountInfo.setProvinceName(electronicAccountInfo.getProvinceName());
    currentElectronicAccountInfo.setCityCode(electronicAccountInfo.getCityCode());
    currentElectronicAccountInfo.setCityName(electronicAccountInfo.getCityName());
    currentElectronicAccountInfo.setDistrictCode(electronicAccountInfo.getDistrictCode());
    currentElectronicAccountInfo.setDistrictName(electronicAccountInfo.getDistrictName());
    
    this.electronicAccountInfoRepository.saveAndFlush(currentElectronicAccountInfo);
    return currentElectronicAccountInfo;
  }

  /**
   * 在更新一个已有的ElectronicAccountInfo模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ElectronicAccountInfo electronicAccountInfo) {
    Validate.notNull(electronicAccountInfo , "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(!StringUtils.isBlank(electronicAccountInfo.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    this.basicsValidation(electronicAccountInfo);
  }

  /**
   * 基础信息验证
   * @param accountInfo
   */
  private void basicsValidation(ElectronicAccountInfo accountInfo) {
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(accountInfo.getProvinceCode(), "省编码不能为空！");
    Validate.notBlank(accountInfo.getProvinceName(), "省名称不能为空！");
    Validate.notBlank(accountInfo.getCityCode(), "市编码不能为空！");
    Validate.notBlank(accountInfo.getCityName(), "市名称不能为空！");
    Validate.notBlank(accountInfo.getDistrictCode(), "区编码不能为空！");
    Validate.notBlank(accountInfo.getDistrictName(), "区名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(accountInfo.getManagementName() == null || accountInfo.getManagementName().length() < 32 , "经营者姓名填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getCompanyName() == null || accountInfo.getCompanyName().length() < 64 , "企业名称填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(accountInfo.getCompanyShortName() == null || accountInfo.getCompanyShortName().length() < 32 , "企业简称填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getCategory() == null || accountInfo.getCategory().length() < 32 , "分类类别填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getEmail() == null || accountInfo.getEmail().length() < 64 , "邮箱填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(accountInfo.getAddress() == null || accountInfo.getAddress().length() < 128 , "详细地址填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(accountInfo.getIndustry() == null || accountInfo.getIndustry().length() < 32 , "行业填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getGeneralIndustry() == null || accountInfo.getGeneralIndustry().length() < 32 , "通用行业填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getScale() == null || accountInfo.getScale().length() < 32 , "规模填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getApprovalNo() == null || accountInfo.getApprovalNo().length() < 64 , "开户许可证核准号填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(accountInfo.getMobile() == null || accountInfo.getMobile().length() < 32 , "手机号填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getProvinceCode() == null || accountInfo.getProvinceCode().length() < 32 , "省编码填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getProvinceName() == null || accountInfo.getProvinceName().length() < 32 , "省名称填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getCityCode() == null || accountInfo.getCityCode().length() < 32 , "市编码填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getCityName() == null || accountInfo.getCityName().length() < 32 , "市名称填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getDistrictCode() == null || accountInfo.getDistrictCode().length() < 32 , "区编码填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(accountInfo.getDistrictName() == null || accountInfo.getDistrictName().length() < 32 , "区名称填入值超过了限定长度(32)，请检查!");

    ElectronicAccount account = accountInfo.getElectronicAccount();
    Validate.notNull(account, "电子账户不能为空");
    Validate.notNull(account.getType(), "电子账户类型不能为空");
    ElectronicAccountType accountType = ElectronicAccountType.valueOfType(account.getType());
    Validate.notNull(accountType,  "不支持的电子账户类型：%s", account.getType());
    if (ElectronicAccountType.INDIVIDUAL.equals(accountType)) {
      this.individualValidation(accountInfo);
    } else if (ElectronicAccountType.COMPANY.equals(accountType)) {
      this.companyValidation(accountInfo);
    }
  }

  /**
   * 个体工商户信息校验
   * @param accountInfo
   */
  private void individualValidation(ElectronicAccountInfo accountInfo) {
    Validate.notBlank(accountInfo.getManagementName(), "经营者姓名不能为空");
    Validate.notBlank(accountInfo.getMobile(), "手机号不能为空");
  }

  /**
   * 企业信息校验
   * @param accountInfo
   */
  private void companyValidation(ElectronicAccountInfo accountInfo) {
    Validate.notBlank(accountInfo.getCompanyName(), "企业名称不能为空");
    Validate.notBlank(accountInfo.getCompanyShortName(), "企业简称不能为空");
    Validate.notBlank(accountInfo.getCategory(), "分类类别不能为空");
    Validate.notBlank(accountInfo.getEmail(), "邮箱不能为空");
    Validate.notBlank(accountInfo.getAddress(), "企业地址不能为空");
    Validate.notBlank(accountInfo.getIndustry(), "行业不能为空");
    Validate.notBlank(accountInfo.getGeneralIndustry(), "通用行业不能为空");
    Validate.notBlank(accountInfo.getScale(), "企业规模不能为空");
    Validate.notBlank(accountInfo.getApprovalNo(), "开户许可证核准号不能为空");
  }

}
