package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.payment.common.enums.ElAccountBillBusinessType;
import com.bizunited.empower.business.payment.common.enums.ElAccountBillChargeRateType;
import com.bizunited.empower.business.payment.common.enums.ElAccountBillState;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountApplyState;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountSignerType;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountState;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountTiedCardStatus;
import com.bizunited.empower.business.payment.common.enums.ElectronicAccountType;
import com.bizunited.empower.business.payment.dto.ElectronicAccountDto;
import com.bizunited.empower.business.payment.entity.ElectronicAccount;
import com.bizunited.empower.business.payment.entity.ElectronicAccountBank;
import com.bizunited.empower.business.payment.entity.ElectronicAccountCertificate;
import com.bizunited.empower.business.payment.entity.ElectronicAccountInfo;
import com.bizunited.empower.business.payment.repository.ElectronicAccountRepository;
import com.bizunited.empower.business.payment.service.ElectronicAccountBankService;
import com.bizunited.empower.business.payment.service.ElectronicAccountBillService;
import com.bizunited.empower.business.payment.service.ElectronicAccountCertificateService;
import com.bizunited.empower.business.payment.service.ElectronicAccountDtoService;
import com.bizunited.empower.business.payment.service.ElectronicAccountFailureRecordService;
import com.bizunited.empower.business.payment.service.ElectronicAccountInfoService;
import com.bizunited.empower.business.payment.service.ElectronicAccountRateVoService;
import com.bizunited.empower.business.payment.service.ElectronicAccountService;
import com.bizunited.empower.business.payment.service.ExtractCashDtoService;
import com.bizunited.empower.business.payment.vo.ElectronicAccountRateVo;
import com.bizunited.empower.business.payment.vo.ExtractCashVo;
import com.bizunited.empower.business.tenant.common.enums.TenantSmsBusinessType;
import com.bizunited.empower.business.tenant.entity.TenantTerminalSetting;
import com.bizunited.empower.business.tenant.service.TenantSmsService;
import com.bizunited.empower.business.tenant.service.TenantTerminalSettingService;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.bizunited.empower.business.payment.common.constant.RedisKeys.TENANT_EL_ACCOUNT_LOCK_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.TENANT_SMS_EL_ACCOUNT_MOBILE_CODE_KEY;
import static com.bizunited.empower.business.payment.common.enums.ElectronicAccountApplyState.WAIT_INTO;
import static com.bizunited.empower.business.payment.common.enums.ElectronicAccountApplyState.WAIT_OPEN_ACCOUNT;
import static com.bizunited.empower.business.payment.common.enums.ElectronicAccountApplyState.WAIT_PRE_APPLY;
import static com.bizunited.empower.business.payment.common.enums.ElectronicAccountApplyState.WAIT_SIGN;
import static com.bizunited.empower.business.payment.common.enums.ElectronicAccountApplyState.WAIT_TIED_CARD;
import static com.bizunited.empower.business.payment.common.enums.ElectronicAccountApplyState.WAIT_VERIFY;
import static com.bizunited.empower.business.tenant.common.constant.RedisKeys.TENANT_SMS_CODE_MAP_KEY;
import static com.bizunited.empower.business.tenant.common.enums.TenantSmsBusinessType.EL_ACCOUNT_MOBILE_NEW;
import static com.bizunited.empower.business.tenant.common.enums.TenantSmsBusinessType.EL_ACCOUNT_MODIFY_MOBILE_NEW;
import static com.bizunited.empower.business.tenant.common.enums.TenantSmsBusinessType.EL_ACCOUNT_MODIFY_MOBILE_OLD;

/**
 * ElectronicAccount业务模型的服务层接口实现
 * @author Paul Chan
 */
@Service("ElectronicAccountServiceImpl")
public class ElectronicAccountServiceImpl implements ElectronicAccountService {

  @Autowired
  private ElectronicAccountRepository electronicAccountRepository;

  @Autowired
  private TenantSmsService tenantSmsService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ExtractCashDtoService extractCashDtoService;
  @Autowired
  private ElectronicAccountDtoService electronicAccountDtoService;
  @Autowired
  private TenantTerminalSettingService tenantTerminalSettingService;
  @Autowired
  private ElectronicAccountBillService electronicAccountBillService;
  @Autowired
  private ElectronicAccountInfoService electronicAccountInfoService;
  @Autowired
  private ElectronicAccountBankService electronicAccountBankService;
  @Autowired
  private ElectronicAccountRateVoService electronicAccountRateVoService;
  @Autowired
  private ElectronicAccountCertificateService electronicAccountCertificateService;
  @Autowired
  private ElectronicAccountFailureRecordService electronicAccountFailureRecordService;

  private static Set<Integer> FAILURE_APPLY_STATES = Sets.newHashSet(ElectronicAccountApplyState.PRE_APPLY_FAILURE.getState(),
      ElectronicAccountApplyState.COLLECT_FAILURE.getState(),
      ElectronicAccountApplyState.OPEN_ACCOUNT_FAILURE.getState(),
      ElectronicAccountApplyState.TIED_CARD_FAILURE.getState(),
      ElectronicAccountApplyState.VERIFY_FAILURE.getState(),
      ElectronicAccountApplyState.INTO_FAILURE.getState(),
      ElectronicAccountApplyState.SIGN_FAILURE.getState());

  @Override
  @Transactional
  public ElectronicAccount create(ElectronicAccount electronicAccount) {
    ElectronicAccount current = this.createForm(electronicAccount);
    return current;
  }

  @Override
  @Transactional
  public ElectronicAccount createForm(ElectronicAccount electronicAccount) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    String userAccount = SecurityUtils.getUserAccount();
    electronicAccount.setTenantCode(TenantUtils.getTenantCode());
    electronicAccount.setCreateAccount(userAccount);
    electronicAccount.setCreateTime(now);
    electronicAccount.setModifyAccount(userAccount);
    electronicAccount.setModifyTime(now);
    electronicAccount.setTiedCardStatus(ElectronicAccountTiedCardStatus.UNTIED_CARD.getStatus());
    electronicAccount.setApplyState(WAIT_PRE_APPLY.getState());
    electronicAccount.setState(ElectronicAccountState.APPLYING.getState());
    electronicAccount.setBalance(BigDecimal.ZERO);
    TenantTerminalSetting terminalSetting = tenantTerminalSettingService.findByAppCodeAndAppType(TenantUtils.getAppCode(), 3);
    if(terminalSetting !=  null) {
      electronicAccount.setAppId(terminalSetting.getMiniAppID());
    }

    this.createValidation(electronicAccount);

    // 保存关联信息
    ElectronicAccountInfo accountInfo = electronicAccount.getAccountInfo();
    accountInfo.setElectronicAccount(electronicAccount);
    electronicAccountInfoService.create(accountInfo);
    ElectronicAccountCertificate certificate = electronicAccount.getCertificate();
    certificate.setElectronicAccount(electronicAccount);
    electronicAccountCertificateService.create(certificate);

    this.electronicAccountRepository.save(electronicAccount);

    // 保存银行信息
    electronicAccountBankService.save(electronicAccount, electronicAccount.getBanks());

    // 同步到saas平台
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(electronicAccount, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    electronicAccountDtoService.create(electronicAccountDto);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return electronicAccount;
  }

  /**
   * 在创建一个新的ElectronicAccount模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ElectronicAccount electronicAccount) { 
    Validate.notNull(electronicAccount , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(electronicAccount.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    electronicAccount.setId(null);
    this.basicsValidation(electronicAccount);
    long count = electronicAccountRepository.countByTenantCode(TenantUtils.getTenantCode());
    Validate.isTrue(count == 0, "只能申请一个电子账户");
  }

  @Override
  @Transactional
  public ElectronicAccount update(ElectronicAccount electronicAccount) {
    ElectronicAccount current = this.updateForm(electronicAccount);
    return current;
  }

  @Override
  @Transactional
  public ElectronicAccount updateForm(ElectronicAccount electronicAccount) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(electronicAccount);
    // ===================基本信息
    String currentId = electronicAccount.getId();
    ElectronicAccount currentElectronicAccount = this.electronicAccountRepository.findDetailsById(currentId);
    currentElectronicAccount = Validate.notNull(currentElectronicAccount ,"未发现指定的原始模型对象信");
    this.updateStateValidation(currentElectronicAccount);
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentElectronicAccount.setModifyAccount(SecurityUtils.getUserAccount());
    currentElectronicAccount.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentElectronicAccount.setExtend1(electronicAccount.getExtend1());
    currentElectronicAccount.setExtend2(electronicAccount.getExtend2());
    currentElectronicAccount.setExtend3(electronicAccount.getExtend3());
    currentElectronicAccount.setExtend4(electronicAccount.getExtend4());
    currentElectronicAccount.setExtend5(electronicAccount.getExtend5());
    currentElectronicAccount.setExtend6(electronicAccount.getExtend6());
    currentElectronicAccount.setExtend7(electronicAccount.getExtend7());
    currentElectronicAccount.setExtend8(electronicAccount.getExtend8());
    currentElectronicAccount.setExtend9(electronicAccount.getExtend9());
    currentElectronicAccount.setExtend10(electronicAccount.getExtend10());
    currentElectronicAccount.setExtend11(electronicAccount.getExtend11());
    currentElectronicAccount.setSignerType(electronicAccount.getSignerType());
    currentElectronicAccount.setSignerName(electronicAccount.getSignerName());
    currentElectronicAccount.setSignerIdCardNo(electronicAccount.getSignerIdCardNo());
    currentElectronicAccount.setSignerMobile(electronicAccount.getSignerMobile());
    this.updateApplyStateByFailure(currentElectronicAccount);
    
    this.electronicAccountRepository.saveAndFlush(currentElectronicAccount);

    // 保存关联信息
    ElectronicAccountInfo accountInfo = electronicAccount.getAccountInfo();
    accountInfo.setElectronicAccount(electronicAccount);
    ElectronicAccountInfo updatedAccountInfo = electronicAccountInfoService.update(accountInfo);
    currentElectronicAccount.setAccountInfo(updatedAccountInfo);
    ElectronicAccountCertificate certificate = electronicAccount.getCertificate();
    certificate.setElectronicAccount(electronicAccount);
    ElectronicAccountCertificate updatedCertificate = electronicAccountCertificateService.update(certificate);
    currentElectronicAccount.setCertificate(updatedCertificate);
    // 保存银行信息
    Set<ElectronicAccountBank> savedBanks = electronicAccountBankService.save(currentElectronicAccount, electronicAccount.getBanks());
    currentElectronicAccount.setBanks(savedBanks);

    // 同步到saas平台
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(currentElectronicAccount, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    electronicAccountDtoService.update(electronicAccountDto);

    return currentElectronicAccount;
  }

  /**
   * 更新失败后的申请状态
   * @param account
   */
  private void updateApplyStateByFailure(ElectronicAccount account) {
    ElectronicAccountApplyState applyState = ElectronicAccountApplyState.valueOfState(account.getApplyState());
    ElectronicAccountApplyState modifyApplyState = null;
    switch (applyState) {
      case PRE_APPLY_FAILURE:
      case COLLECT_FAILURE:
        modifyApplyState = WAIT_PRE_APPLY;
        break;
      case OPEN_ACCOUNT_FAILURE:
        modifyApplyState = WAIT_OPEN_ACCOUNT;
        break;
      case TIED_CARD_FAILURE:
      case VERIFY_FAILURE:
        modifyApplyState = WAIT_TIED_CARD;
        break;
      case INTO_FAILURE:
        modifyApplyState = WAIT_INTO;
        break;
      case SIGN_FAILURE:
        modifyApplyState = WAIT_SIGN;
        break;
      default:
        throw new IllegalArgumentException("apply state error");
    }
    if(modifyApplyState != null) {
      account.setApplyState(modifyApplyState.getState());
    }
  }

  @Override
  @Transactional
  public void updateTiedCardStatusById(String electronicAccountId, Integer tiedCardStatus) {
    Validate.notBlank(electronicAccountId, "电子账户ID不能为空");
    Validate.notNull(tiedCardStatus, "绑卡状态不能为空");
    ElectronicAccountTiedCardStatus status = ElectronicAccountTiedCardStatus.valueOfStatus(tiedCardStatus);
    Validate.notNull(status, "不支持的状态：%s", tiedCardStatus);
    ElectronicAccount electronicAccount = this.findById(electronicAccountId);
    Validate.notNull(electronicAccount, "未找到电子账户：%s", electronicAccountId);
    if(tiedCardStatus.equals(electronicAccount.getTiedCardStatus())) {
      return;
    }
    electronicAccount.setTiedCardStatus(tiedCardStatus);
    electronicAccount.setModifyTime(new Date());
    electronicAccount.setModifyAccount(SecurityUtils.getUserAccount());
    if(status.equals(ElectronicAccountTiedCardStatus.TIED_CARD)) {
      if(WAIT_VERIFY.getState().equals(electronicAccount.getApplyState())) {
        electronicAccount.setRepeatedApplyTime(new Date());
        if(ElectronicAccountType.INDIVIDUAL.getType().equals(electronicAccount.getType())) {
          // 如果是绑定银行卡，则更新状态到待开户状态
          electronicAccount.setApplyState(WAIT_OPEN_ACCOUNT.getState());
        } else {
          // 如果是绑定银行卡，则更新状态到待签约状态
          electronicAccount.setApplyState(WAIT_SIGN.getState());
        }
      }
    }
    electronicAccountRepository.save(electronicAccount);
    if(status.equals(ElectronicAccountTiedCardStatus.TIED_CARD)) {
      electronicAccountBankService.bindByElectronicAccountId(electronicAccountId);
    }
  }

  @Override
  @Transactional
  public void updateApplyStateById(String id, Integer applyState, String failureReason) {
    Validate.notBlank(id, "ID不能为空");
    Validate.notNull(applyState, "变更申请状态不能为空");
    ElectronicAccountApplyState applyStateEnum = ElectronicAccountApplyState.valueOfState(applyState);
    Validate.notNull(applyStateEnum, "错误的申请状态：%s", applyState);
    ElectronicAccount account = this.findById(id);
    Validate.notNull(account, "未找到电子账户：%s", id);
    if(FAILURE_APPLY_STATES.contains(applyState)) {
      account.setLatestFailureReason(failureReason);
      electronicAccountFailureRecordService.create(id, failureReason);
    }
    account.setApplyState(applyState);
    account.setModifyTime(new Date());
    electronicAccountRepository.save(account);
  }

  @Override
  @Transactional
  public void updateStateById(String id, Integer state) {
    Validate.notBlank(id, "ID不能为空");
    Validate.notNull(state, "变更状态不能为空");
    ElectronicAccountState stateEnum = ElectronicAccountState.valueOfState(state);
    Validate.notNull(stateEnum, "不支持的状态：%s", state);
    Validate.isTrue(!stateEnum.equals(ElectronicAccountState.APPLYING), "状态不能修改为申请中");
    ElectronicAccount account = this.findById(id);
    Validate.notNull(account, "未找到电子账户：%s", id);
    if(state.equals(account.getState())) {
      return;
    }
    account.setState(state);
    account.setModifyTime(new Date());
    electronicAccountRepository.save(account);
  }

  @Override
  @Transactional
  public void updateTimesById(String id, Date preApplyTime, Date repeatedApplyTime, Date signTime, Date openTime, Date expireTime) {
    Validate.notBlank(id, "ID不能为空");
    ElectronicAccount account = this.findById(id);
    Validate.notNull(account, "未找到电子账户：%s", id);
    if(preApplyTime != null) {
      account.setPreApplyTime(preApplyTime);
    }
    if(repeatedApplyTime != null) {
      account.setRepeatedApplyTime(repeatedApplyTime);
    }
    if(signTime != null) {
      account.setSignTime(signTime);
    }
    if(openTime != null) {
      account.setOpenTime(openTime);
    }
    if(expireTime != null) {
      account.setExpireTime(expireTime);
    }
    account.setModifyTime(new Date());
    electronicAccountRepository.save(account);
  }

  @Override
  @Transactional
  public void updateShowBalance(Boolean showBalance) {
    Validate.notNull(showBalance, "是否显示余额不能为空");
    ElectronicAccount account = this.findByTenant();
    Validate.notNull(account, "未开通电子账户");
    account.setShowBalance(showBalance);
    account.setModifyTime(new Date());
    account.setModifyAccount(SecurityUtils.getUserAccount());
    electronicAccountRepository.save(account);
  }

  @Override
  @Transactional
  public void updateMerchantInfoById(String id, String merchantCode, String merchantName, String billingCycle) {
    Validate.notBlank(id, "ID不能为空");
    Validate.notBlank(merchantCode, "商户编码不能为空");
    Validate.notBlank(merchantName, "商户名称不能为空");
    ElectronicAccount account = this.findById(id);
    Validate.notNull(account, "未找到电子账户：%s", id);
    account.setMerchantCode(merchantCode);
    account.setMerchantName(merchantName);
    account.setBillingCycle(billingCycle);
    account.setModifyTime(new Date());
    electronicAccountRepository.save(account);
  }

  @Override
  @Transactional
  public void updateUserIdById(String id, String userId) {
    Validate.notBlank(id, "ID不能为空");
    Validate.notBlank(userId, "更新账户绑定的唯一标识不能为空");
    ElectronicAccount account = this.findById(id);
    Validate.notNull(account, "未找到电子账户：%s", id);
    account.setUserId(userId);
    account.setModifyTime(new Date());
    electronicAccountRepository.save(account);
  }

  @Override
  @Transactional
  public void receipt(BigDecimal amount, BigDecimal chargeRate, BigDecimal chargeAmount, BigDecimal realAmount, String businessNo, String remark, String dealFrom) {
    Validate.notNull(amount, "交易金额不能为空");
    Validate.isTrue(amount.compareTo(BigDecimal.ZERO) > 0, "交易金额必须大于0");
    Validate.notBlank(businessNo, "业务流水号不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    String redisKey = String.format(TENANT_EL_ACCOUNT_LOCK_PREFIX, tenantCode);
    boolean isLocked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "操作繁忙，请稍后再试");
    try {
      ElectronicAccount account = this.findDetailsByTenant();
      Validate.notNull(account, "当前经销商未开通电子账户");
      BigDecimal balance = account.getBalance().add(realAmount);
      account.setBalance(balance);
      account.setModifyTime(new Date());
      account.setModifyAccount(SecurityUtils.getUserAccount());
      electronicAccountRepository.save(account);
      electronicAccountBillService.create(account.getId(), ElAccountBillState.SUCCESS, ElAccountBillBusinessType.RECEIPT, businessNo,
          amount, ElAccountBillChargeRateType.RATE, chargeRate, chargeAmount, realAmount, dealFrom, "提现");
    } finally {
      if(redisMutexService.islock(redisKey)) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  @Override
  @Transactional
  public void withdraw(BigDecimal amount, String validCode) {
    Validate.notNull(amount, "提现金额不能为空");
    Validate.isTrue(amount.compareTo(BigDecimal.ZERO) > 0, "提现金额必须大于0");
    Validate.notBlank(validCode, "验证码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    tenantSmsService.verifyValidCode(tenantCode, TenantSmsBusinessType.EL_ACCOUNT_WITHDRAW, validCode);
    String redisKey = String.format(TENANT_EL_ACCOUNT_LOCK_PREFIX, tenantCode);
    boolean isLocked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "操作繁忙，请稍后再试");
    try {
      ElectronicAccount account = this.findDetailsByTenant();
      Validate.notNull(account, "当前经销商未开通电子账户");

      Validate.isTrue(ElectronicAccountState.ENABLE.getState().equals(account.getState()), "电子账户未启用");
      Validate.isTrue(ElectronicAccountTiedCardStatus.TIED_CARD.getStatus().equals(account.getTiedCardStatus()), "电子账户未绑定银行卡");
      ElAccountBillChargeRateType rateType = ElAccountBillChargeRateType.RATE;
      BigDecimal chargeRate = BigDecimal.ZERO;
      BigDecimal chargeAmount = BigDecimal.ZERO;
      List<ElectronicAccountRateVo> rates = electronicAccountRateVoService.findByTenant();
      if(!CollectionUtils.isEmpty(rates)) {
        // 计算手续费
        rates = rates.stream().filter(r -> r.getRateItemNum() == 6).collect(Collectors.toList());
        if(!CollectionUtils.isEmpty(rates)) {
          ElectronicAccountRateVo rate = rates.get(0);
          if(rate.getRateType() == ElAccountBillChargeRateType.RATE.getType()) {
            rateType = ElAccountBillChargeRateType.RATE;
            chargeRate = rate.getRate();
            chargeAmount = amount.multiply(chargeRate);
          } else if (rate.getRateType() == ElAccountBillChargeRateType.AMOUNT.getType()) {
            rateType = ElAccountBillChargeRateType.AMOUNT;
            chargeRate = rate.getRate();
            chargeAmount = rate.getRate();
          }
        }
      }
      BigDecimal realAmount = amount.add(chargeAmount);
      Validate.isTrue(account.getBalance().compareTo(realAmount) >= 0, "余额不足！");
      BigDecimal balance = account.getBalance().subtract(realAmount);
      account.setBalance(balance);
      account.setModifyTime(new Date());
      account.setModifyAccount(SecurityUtils.getUserAccount());
      electronicAccountRepository.save(account);
      electronicAccountRepository.flush();
      ExtractCashVo extractCashVo = extractCashDtoService.extractCash(account, amount);
      electronicAccountBillService.create(account.getId(), ElAccountBillState.HANDLING, ElAccountBillBusinessType.WITHDRAW, extractCashVo.getTxSN(),
          amount, rateType, chargeRate, chargeAmount, realAmount, account.getMerchantName(), "提现");
    } finally {
      if(redisMutexService.islock(redisKey)) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  @Override
  @Transactional
  public void addBalanceByAccountId(String id, BigDecimal amount) {
    Validate.notNull(amount, "金额不能为空");
    Validate.isTrue(amount.compareTo(BigDecimal.ZERO) > 0, "金额必须大于0");
    String tenantCode = TenantUtils.getTenantCode();
    String redisKey = String.format(TENANT_EL_ACCOUNT_LOCK_PREFIX, tenantCode);
    boolean isLocked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "操作繁忙，请稍后再试");
    try {
      ElectronicAccount account = findDetailsById(id);
      Validate.notNull(account, "未找到电子账户");
      BigDecimal balance = account.getBalance().add(amount);
      account.setBalance(balance);
      account.setModifyTime(new Date());
      account.setModifyAccount(SecurityUtils.getUserAccount());
      electronicAccountRepository.save(account);
    } finally {
      if(redisMutexService.islock(redisKey)) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  @Override
  public void sendValidCode(String mobile, Integer businessType) {
    Validate.notNull(businessType, "短信业务类型不能为空");
    TenantSmsBusinessType businessTypeEnum = TenantSmsBusinessType.valueOfType(businessType);
    Validate.notNull(businessTypeEnum, "不支持的短信业务类型：%s", businessType);
    ElectronicAccount account = this.findDetailsByTenant();
    Validate.notNull(account, "当前经销商未开通电子账户");
    String sendMobile;
    switch (businessTypeEnum) {
      case EL_ACCOUNT_MODIFY_MOBILE_NEW:
      case EL_ACCOUNT_MOBILE_NEW:
        Validate.notBlank(mobile, "手机号码不能为空");
        sendMobile = mobile;
        break;
      case EL_ACCOUNT_MODIFY_MOBILE_OLD:
      case EL_ACCOUNT_WITHDRAW:
      case EL_ACCOUNT_UNBIND_BANK:
        Validate.notBlank(account.getSecurityMobile(), "电子账户未绑定安全手机号");
        sendMobile = account.getSecurityMobile();
        break;
      case EL_ACCOUNT_SIGN:
        throw new UnsupportedOperationException("不支持该业务类型");
      case EL_ACCOUNT_BIND_BANK_CARD:
        // TODO: 2021/4/19 Paul 调用saas平台接口发送短信
        return;
      default:
        throw new IllegalArgumentException("不支持的短信业务类型");
    }
    String tenantCode = TenantUtils.getTenantCode();
    tenantSmsService.sendValidCode(tenantCode, sendMobile, businessTypeEnum);
  }

  @Override
  public void verifyValidCode(String mobile, String validCode, Integer businessType) {
    Validate.notBlank(mobile, "手机号码不能为空");
    Validate.notBlank(validCode, "验证码不能为空");
    Validate.notNull(businessType, "短信业务类型不能为空");
    TenantSmsBusinessType businessTypeEnum = TenantSmsBusinessType.valueOfType(businessType);
    Validate.notNull(businessTypeEnum, "不支持的短信业务类型：%s", businessType);
    String tenantCode = TenantUtils.getTenantCode();
    tenantSmsService.verifyValidCode(tenantCode, mobile, businessTypeEnum, validCode);
    // 如果验证码验证成功，则将验证码存储在redis中，用于后续操作的验证码验证
    String redisKey = String.format(TENANT_SMS_EL_ACCOUNT_MOBILE_CODE_KEY, tenantCode, businessType);
    // 一个操作的有效时间为5分钟
    redisMutexService.setMCode(redisKey, TENANT_SMS_CODE_MAP_KEY, validCode, 300000);
  }

  @Override
  @Transactional
  public void bindSecurityMobile(String mobile, String validCode) {
    Validate.notBlank(mobile, "手机号码不能为空");
    Validate.notBlank(validCode, "验证码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    tenantSmsService.verifyValidCode(tenantCode, mobile, EL_ACCOUNT_MOBILE_NEW,  validCode);
    ElectronicAccount account = this.findDetailsByTenant();
    Validate.notNull(account, "当前经销商未开通电子账户");
    Validate.isTrue(StringUtils.isBlank(account.getSecurityMobile()), "当前电子账户已绑定安全手机，请走更换流程");
    account.setSecurityMobile(mobile);
    account.setModifyTime(new Date());
    account.setModifyAccount(SecurityUtils.getUserAccount());
    electronicAccountRepository.saveAndFlush(account);
    // 同步到saas平台
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(account, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    electronicAccountDtoService.updateSecurePhone(electronicAccountDto);
  }

  @Override
  @Transactional
  public void rebindSecurityMobile(String mobile, String oldMobileValidCode, String newMobileValidCode) {
    Validate.notBlank(mobile, "手机号码不能为空");
    Validate.notBlank(oldMobileValidCode, "旧手机号验证码不能为空");
    Validate.notBlank(newMobileValidCode, "新手机号验证码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    ElectronicAccount account = this.findDetailsByTenant();
    Validate.notNull(account, "当前经销商未开通电子账户");
    Validate.notBlank(account.getSecurityMobile(), "电子账户未绑定过手机，请先绑定安全手机");
    Validate.isTrue(!mobile.equals(account.getSecurityMobile()), "新旧手机不能一样");
    tenantSmsService.verifyValidCode(tenantCode, mobile, EL_ACCOUNT_MODIFY_MOBILE_NEW,  newMobileValidCode);
    String redisKey = String.format(TENANT_SMS_EL_ACCOUNT_MOBILE_CODE_KEY, tenantCode, EL_ACCOUNT_MODIFY_MOBILE_OLD.getType());
    String redisOldMobileValidCode = redisMutexService.getMCode(redisKey, TENANT_SMS_CODE_MAP_KEY);
    Validate.isTrue(oldMobileValidCode.equals(redisOldMobileValidCode), "当前操作已超时，请重新操作！");
    account.setSecurityMobile(mobile);
    account.setModifyTime(new Date());
    account.setModifyAccount(SecurityUtils.getUserAccount());
    electronicAccountRepository.saveAndFlush(account);
    redisMutexService.removeMCode(redisKey, TENANT_SMS_CODE_MAP_KEY);
    // 同步到saas平台
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(account, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    electronicAccountDtoService.updateSecurePhone(electronicAccountDto);
  }

  @Override
  @Transactional
  public void sendSignValidCode(Boolean isUpdate, String name, String idCardNo, String mobile) {
    ElectronicAccount account = this.findDetailsByTenant();
    Validate.notNull(account, "当前经销商未开通电子账户");
    if(isUpdate != null && isUpdate) {
      Validate.notBlank(name, "签约人姓名不能为空");
      Validate.notBlank(idCardNo, "签约人身份证号不能为空");
      Validate.notBlank(mobile, "签约人手机号不能为空");
      account.setSignerType(ElectronicAccountSignerType.AGENT.getType());
      account.setSignerName(name);
      account.setSignerIdCardNo(idCardNo);
      account.setSignerMobile(mobile);
      account.setModifyAccount(SecurityUtils.getUserAccount());
      account.setModifyTime(new Date());
      electronicAccountRepository.saveAndFlush(account);
    }
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(account, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    electronicAccountDtoService.sendSignValidCode(electronicAccountDto, isUpdate);
  }

  @Override
  public String getSignUrlByTenant(String validCode) {
    ElectronicAccount account = this.findDetailsByTenant();
    Validate.notNull(account, "经销商未申请电子账户");
    Validate.isTrue(WAIT_SIGN.getState().equals(account.getApplyState()), "待签约状态才能签约");
    ElectronicAccountDto electronicAccountDto = nebulaToolkitService.copyObjectByWhiteList(account, ElectronicAccountDto.class, HashSet.class, ArrayList.class,
        "accountInfo", "certificate", "banks");
    return electronicAccountDtoService.findSignUrl(electronicAccountDto, validCode);
  }

  /**
   * 更新前校验当前电子账户状态
   * @param account
   */
  private void updateStateValidation(ElectronicAccount account) {
    Validate.isTrue(ElectronicAccountState.APPLYING.getState().equals(account.getState()), "只有申请中的电子账户可以修改");
    Validate.isTrue(FAILURE_APPLY_STATES.contains(account.getApplyState()), "当前状态不能修改信息");
  }

  /**
   * 在更新一个已有的ElectronicAccount模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ElectronicAccount account) {
    Validate.notNull(account , "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(!StringUtils.isBlank(account.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    this.basicsValidation(account);
  }

  /**
   * 基础边界校验
   * @param account
   */
  private void basicsValidation(ElectronicAccount account) {
    // 基础信息判断，基本属性，需要满足not null
    Validate.notNull(account.getType(), "类型不能为空！");
    Validate.notNull(account.getSignerType(), "签约人类型不能为空！");
    ElectronicAccountType electronicAccountType = ElectronicAccountType.valueOfType(account.getType());
    Validate.notNull(electronicAccountType, "不支持的电子账户类型：%s", account.getType());
    ElectronicAccountSignerType signerType = ElectronicAccountSignerType.valueOfType(account.getSignerType());
    Validate.notNull(signerType, "不支持的签约人类型：%s", account.getSignerType());
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(account.getExtend1() == null || account.getExtend1().length() < 255 , "扩展字段1超过了限定长度(255)，请检查!");
    Validate.isTrue(account.getExtend2() == null || account.getExtend2().length() < 255 , "扩展字段2超过了限定长度(255)，请检查!");
    Validate.isTrue(account.getExtend3() == null || account.getExtend3().length() < 255 , "扩展字段3超过了限定长度(255)，请检查!");
    Validate.isTrue(account.getExtend4() == null || account.getExtend4().length() < 255 , "扩展字段4超过了限定长度(255)，请检查!");
    Validate.isTrue(account.getExtend5() == null || account.getExtend5().length() < 255 , "扩展字段5超过了限定长度(255)，请检查!");
    Validate.isTrue(account.getExtend6() == null || account.getExtend6().length() < 255 , "扩展字段6超过了限定长度(255)，请检查!");
    Validate.isTrue(account.getExtend7() == null || account.getExtend7().length() < 255 , "扩展字段超过了限定长度(255)，请检查!");
    Validate.isTrue(account.getTenantCode() == null || account.getTenantCode().length() < 255 , "租户编号超过了限定长度(255)，请检查!");
    Validate.isTrue(account.getLatestFailureReason() == null || account.getLatestFailureReason().length() < 255 , "失败原因超过了限定长度(255)，请检查!");
    Validate.isTrue(account.getSignerName() == null || account.getSignerName().length() < 32 , "经办人姓名超过了限定长度(32)，请检查!");
    Validate.isTrue(account.getSignerIdCardNo() == null || account.getSignerIdCardNo().length() < 32 , "经办人身份证号超过了限定长度(32)，请检查!");
    Validate.isTrue(account.getSignerMobile() == null || account.getSignerMobile().length() < 32 , "经办人手机号码超过了限定长度(32)，请检查!");

    if(ElectronicAccountSignerType.AGENT.equals(signerType)) {
      Validate.notBlank(account.getSignerName(), "经办人姓名不能为空");
      Validate.notBlank(account.getSignerIdCardNo(), "经办人身份证号不能为空");
      Validate.notBlank(account.getSignerMobile(), "经办人手机号码不能为空");
    }
  }

  @Override
  public ElectronicAccount findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return this.electronicAccountRepository.findDetailsById(id);
  }

  @Override
  public ElectronicAccount findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return electronicAccountRepository.findById(id).orElse(null);
  }

  @Override
  public ElectronicAccount findDetailsByTenant() {
    String tenantCode = TenantUtils.getTenantCode();
    return electronicAccountRepository.findDetailsByTenantCode(tenantCode);
  }

  @Override
  public ElectronicAccount findByTenant() {
    String tenantCode = TenantUtils.getTenantCode();
    return electronicAccountRepository.findByTenantCode(tenantCode);
  }

  @Override
  public ElectronicAccount findByTenantCode(String tenantCode) {
    if(StringUtils.isBlank(tenantCode)) {
      return null;
    }
    return electronicAccountRepository.findByTenantCode(tenantCode);
  }

  @Override
  public List<ElectronicAccount> findAll() {
    return electronicAccountRepository.findAll();
  }

  @Override
  public ElectronicAccount findByMerchantCode(String merchantCode) {
    return electronicAccountRepository.findByMerchantCode(merchantCode);
  }

}
