package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.payment.entity.FundsAccount;
import com.bizunited.empower.business.payment.repository.FundsAccountRepository;
import com.bizunited.empower.business.payment.service.FundsAccountService;
import com.bizunited.empower.business.common.util.SecurityUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.Optional;

/**
 * FundsAccount
 * @author saturn
 */
@Service("FundsAccountServiceImpl")
public class FundsAccountServiceImpl implements FundsAccountService {
  @Autowired
  private FundsAccountRepository fundsAccountRepository;
  @Transactional
  @Override
  public FundsAccount create(FundsAccount fundsAccount) {
    FundsAccount current = this.createForm(fundsAccount);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public FundsAccount createForm(FundsAccount fundsAccount) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    fundsAccount.setCreateAccount(SecurityUtils.getUserAccount());
    fundsAccount.setCreateTime(now);
    fundsAccount.setModifyAccount(SecurityUtils.getUserAccount());
    fundsAccount.setModifyTime(now);

    this.createValidation(fundsAccount);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.fundsAccountRepository.save(fundsAccount);
    // 返回最终处理的结果，里面带有详细的关联信息
    return fundsAccount;
  }
  /**
   * 在创建一个新的ReceivableAccount模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(FundsAccount fundsAccount) {
    Validate.notNull(fundsAccount, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(fundsAccount.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    fundsAccount.setId(null);
    Validate.notBlank(fundsAccount.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(fundsAccount.getName(), "添加信息时，账户名称不能为空！");
    Validate.notBlank(fundsAccount.getBank(), "添加信息时，开户银行不能为空！");
    Validate.notBlank(fundsAccount.getAccount(), "添加信息时，开户账号不能为空！");
    Validate.notNull(fundsAccount.getAccountStatus(), "添加信息时，账户状态 启用:1、禁用:0不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(fundsAccount.getTenantCode() == null || fundsAccount.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(fundsAccount.getName() == null || fundsAccount.getName().length() < 255 , "账户名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(fundsAccount.getBank() == null || fundsAccount.getBank().length() < 255 , "开户银行,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(fundsAccount.getAccount() == null || fundsAccount.getAccount().length() < 255 , "开户账号,在进行添加时填入值超过了限定长度(255)，请检查!");
  }
  @Transactional
  @Override
  public FundsAccount update(FundsAccount fundsAccount) {
    FundsAccount current = this.updateForm(fundsAccount);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public FundsAccount updateForm(FundsAccount fundsAccount) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(fundsAccount);
    // ===================基本信息
    String currentId = fundsAccount.getId();
    Optional<FundsAccount> op_currentReceivableAccount = this.fundsAccountRepository.findById(currentId);
    FundsAccount currentFundsAccount = op_currentReceivableAccount.orElse(null);
    currentFundsAccount = Validate.notNull(currentFundsAccount,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentFundsAccount.setModifyAccount(SecurityUtils.getUserAccount());
    currentFundsAccount.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentFundsAccount.setTenantCode(fundsAccount.getTenantCode());
    currentFundsAccount.setName(fundsAccount.getName());
    currentFundsAccount.setBank(fundsAccount.getBank());
    currentFundsAccount.setAccount(fundsAccount.getAccount());
    currentFundsAccount.setAccountStatus(fundsAccount.getAccountStatus());
    
    this.fundsAccountRepository.saveAndFlush(currentFundsAccount);
    return currentFundsAccount;
  }
  /**
   * 在更新一个已有的ReceivableAccount模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(FundsAccount fundsAccount) {
    Validate.isTrue(!StringUtils.isBlank(fundsAccount.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(fundsAccount.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(fundsAccount.getName(), "修改信息时，账户名称不能为空！");
    Validate.notBlank(fundsAccount.getBank(), "修改信息时，开户银行不能为空！");
    Validate.notBlank(fundsAccount.getAccount(), "修改信息时，开户账号不能为空！");
    Validate.notNull(fundsAccount.getAccountStatus(), "修改信息时，账户状态 启用:1、禁用:0不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(fundsAccount.getTenantCode() == null || fundsAccount.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(fundsAccount.getName() == null || fundsAccount.getName().length() < 255 , "账户名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(fundsAccount.getBank() == null || fundsAccount.getBank().length() < 255 , "开户银行,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(fundsAccount.getAccount() == null || fundsAccount.getAccount().length() < 255 , "开户账号,在进行修改时填入值超过了限定长度(255)，请检查!");
  } 
  @Override
  public FundsAccount findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.fundsAccountRepository.findDetailsById(id);
  }
  @Override
  public FundsAccount findById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<FundsAccount> op = fundsAccountRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    FundsAccount current = this.findById(id);
    if(current != null) { 
      this.fundsAccountRepository.delete(current);
    }
  }
} 
