package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.common.enums.PaymentStatus;
import com.bizunited.empower.business.payment.common.enums.PaymentType;
import com.bizunited.empower.business.payment.common.enums.WalletBillBusinessType;
import com.bizunited.empower.business.payment.entity.DefrayInfo;
import com.bizunited.empower.business.payment.entity.PaymentInfo;
import com.bizunited.empower.business.payment.repository.PaymentInfoRepository;
import com.bizunited.empower.business.payment.service.CustomerCreditService;
import com.bizunited.empower.business.payment.service.CustomerWalletService;
import com.bizunited.empower.business.payment.service.PaymentInfoService;
import com.bizunited.empower.business.payment.service.notifier.PaymentEventListener;
import com.bizunited.empower.business.payment.vo.AssociatedPaymentVo;
import com.bizunited.empower.business.payment.vo.CustomerPaymentVo;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.constant.Constants.PAYMENT_INFO_NO_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.PAYMENT_INFO_LOCK_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.PAYMENT_INFO_NO_KEY_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.PAYMENT_ORDER_LOCK_PREFIX;

/**
 * PaymentInfo业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("PaymentInfoServiceImpl")
public class PaymentInfoServiceImpl implements PaymentInfoService {
  @Autowired
  private PaymentInfoRepository paymentInfoRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired(required = false)
  private List<PaymentEventListener> paymentEventListeners;
  @Autowired
  private CustomerService customerService;
  @Autowired
  private CustomerCreditService customerCreditService;
  @Autowired
  private CustomerWalletService customerWalletService;

  @Override
  public PaymentInfo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.paymentInfoRepository.findDetailsById(id);
  }

  @Override
  public PaymentInfo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<PaymentInfo> op = paymentInfoRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public PaymentInfo findByPaymentCode(String paymentCode) {
    if (StringUtils.isBlank(paymentCode)) {
      return null;
    }
    return this.paymentInfoRepository.findByPaymentCodeAndTenantCode(paymentCode, TenantUtils.getTenantCode());
  }

  @Override
  public PaymentInfo findByAssociatedCode(String associatedCode) {
    if (StringUtils.isBlank(associatedCode)) {
      return null;
    }
    return this.paymentInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public PaymentInfo createByAssociatedCode(String customerCode, String associatedCode, BigDecimal amount, PaymentType paymentType) {
    /*
     * 1、根据订单创建应付账款数据
     *  1.1、初始化应付账款信息
     *  1.2、检查订单是否已经存在应付账款数据
     * 2、创建付款单
     */
    this.createByAssociatedCodeValidation(customerCode, associatedCode, amount, paymentType);

    String lockKey = null;
    if (StringUtils.isNotBlank(associatedCode)) {
      lockKey = String.format(PAYMENT_ORDER_LOCK_PREFIX, TenantUtils.getTenantCode(), associatedCode);
    } else {
      lockKey = String.format(PAYMENT_ORDER_LOCK_PREFIX, TenantUtils.getTenantCode(), customerCode);
    }
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应付账单繁忙，请稍后再试");

    Customer customer = customerService.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
    Validate.notNull(customer, "客户信息异常，请检查");

    PaymentInfo paymentInfo = new PaymentInfo();
    try {
      Date now = new Date();
      paymentInfo.setCreateAccount(SecurityUtils.getUserAccount());
      paymentInfo.setCreateTime(now);
      paymentInfo.setModifyAccount(SecurityUtils.getUserAccount());
      paymentInfo.setModifyTime(now);
      paymentInfo.setAssociatedCode(associatedCode);
      paymentInfo.setTenantCode(TenantUtils.getTenantCode());
      paymentInfo.setPaymentCode(generateNo());
      paymentInfo.setCustomerCode(customerCode);
      paymentInfo.setCustomerName(customer.getCustomerName());
      paymentInfo.setPayType(paymentType.getValue());
      // 待付款金额
      paymentInfo.setWaitPayAmount(amount);
      // 待确认金额
      paymentInfo.setWaitConfirmAmount(BigDecimal.ZERO);
      // 应付款金额
      paymentInfo.setPayAmount(amount);
      // 已付款金额
      paymentInfo.setPayedAmount(BigDecimal.ZERO);
      paymentInfo.setPaymentStatus(PaymentStatus.NO_PAYMENT.getValue());
      paymentInfo.setTstatus(1);
      paymentInfo = this.paymentInfoRepository.save(paymentInfo);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
    return paymentInfo;
  }


  @Override
  public CustomerPaymentVo findAggregationByCustomerCode(String customerCode) {
    Map<String, Object> data = this.paymentInfoRepository.findAggregationByCustomerCode(customerCode, TenantUtils.getTenantCode());
    CustomerPaymentVo vo = new CustomerPaymentVo();
    if (data.isEmpty()) {
      vo.setCustomerCode(customerCode);
      vo.setPaymentNum(0);
      vo.setPaymentAmount(BigDecimal.ZERO);
      vo.setPayedAmount(BigDecimal.ZERO);
      vo.setWaitPayAmount(BigDecimal.ZERO);
      vo.setWaitConfirmAmount(BigDecimal.ZERO);
    } else {
      vo.setCustomerCode((String) data.get("customerCode"));
      vo.setPaymentNum(((BigInteger) data.get("paymentNum")).intValue());
      vo.setPaymentAmount(((BigDecimal) data.get("payAmount")));
      vo.setPayedAmount(((BigDecimal) data.get("payedAmount")));
      vo.setWaitPayAmount(((BigDecimal) data.get("waitPayAmount")));
      vo.setWaitConfirmAmount(((BigDecimal) data.get("waitConfirmAmount")));
    }
    return vo;
  }

  @Override
  @Transactional
  public PaymentInfo updateByAssociatedCode(String associatedCode, BigDecimal amount) {
    /*
     * 更新应付账款(待增加日志功能)
     * 1、参数有效性验证
     * 2、更新待付金额、待确认金额、付款金额、应付金额、已付金额
     */
    this.updateByAssociatedCodeValidation(associatedCode, amount);
    PaymentInfo paymentInfo = this.paymentInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
    Validate.notNull(paymentInfo, "应付账款单不存在，请检查！");
    Validate.isTrue(paymentInfo.getPaymentStatus() == PaymentStatus.NO_PAYMENT.getValue().intValue(), "该应付账款单已部分付款或完成付款，不能进行修改");
    paymentInfo.setWaitPayAmount(amount);
    paymentInfo.setWaitConfirmAmount(BigDecimal.ZERO);
    paymentInfo.setPayAmount(amount);
    paymentInfo.setPayedAmount(BigDecimal.ZERO);
    return this.paymentInfoRepository.save(paymentInfo);
  }

  @Override
  @Transactional
  public PaymentInfo cancelByAssociatedCode(String associatedCode) {
    /*
     * 取消应付账款单（为支付状态下可以取消应付账款）
     * 1、判定应付账款单状态 只有在未进行付款的状态下才能进行取消
     */
    PaymentInfo paymentInfo = this.paymentInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
    Validate.notNull(paymentInfo, "应付账款单不存在，请检查！");
    Validate.isTrue(paymentInfo.getPaymentStatus().intValue() == PaymentStatus.NO_PAYMENT.getValue().intValue(), "应付账款单取消失败，已存在部分付款或完成付款。");
    paymentInfo.setTstatus(0);
    return this.paymentInfoRepository.saveAndFlush(paymentInfo);
  }

  @Override
  public Page<AssociatedPaymentVo> findByConditionsForAssociated(Pageable pageable, InvokeParams conditions) {
    conditions.getInvokeParams().put("tenantCode", TenantUtils.getTenantCode());
    Page<AssociatedPaymentVo> page = this.paymentInfoRepository.queryPageForAssociated(pageable, conditions);
    return page;
  }

  @Override
  public Page<CustomerPaymentVo> findByConditionsForCustomer(Pageable pageable, InvokeParams conditions) {
    conditions.getInvokeParams().put("tenantCode", TenantUtils.getTenantCode());
    Page<Object[]> page = this.paymentInfoRepository.queryPageForCustomer(pageable, conditions);
    List<CustomerPaymentVo> list = Lists.newArrayList();
    if (page != null && page.getTotalElements() > 0L) {
      page.getContent().stream().forEach(data -> {
        CustomerPaymentVo vo = new CustomerPaymentVo();
        vo.setCustomerCode((String) data[0]);
        vo.setCustomerName((String) data[1]);
        vo.setCustomerLevel((String) data[2]);
        vo.setCustomerCategory((String) data[3]);
        vo.setSalesName((String) data[4]);
        vo.setPaymentNum(((BigInteger) data[5]).intValue());
        vo.setPaymentAmount(((BigDecimal) data[6]));
        vo.setPayedAmount(((BigDecimal) data[7]));
        vo.setWaitPayAmount(((BigDecimal) data[8]));
        vo.setWaitConfirmAmount(((BigDecimal) data[9]));
        list.add(vo);
      });
    } else {
      return null;
    }
    return new PageImpl<>(list, pageable, page.getTotalElements());
  }

  /**
   * 订单创建参数验证
   *
   * @param customerCode
   * @param associatedCode
   * @param amount
   * @param paymentType
   */
  private void createByAssociatedCodeValidation(String customerCode, String associatedCode, BigDecimal amount, PaymentType paymentType) {
    Validate.notBlank(customerCode, "客户编号不存在，请检查");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "金额为0或者负，请检查");
    Validate.notNull(paymentType, "应付账款类型异常，请检查");
    if (StringUtils.isNotBlank(associatedCode)) {
      PaymentInfo paymentInfo = this.paymentInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
      Validate.isTrue(paymentInfo == null, "该应付账款已存在");
    }
  }

  /**
   * 订单更新参数验证
   *
   * @param associatedCode
   * @param amount
   */
  private void updateByAssociatedCodeValidation(String associatedCode, BigDecimal amount) {
    Validate.notBlank(associatedCode, "关联单据不存在，请检查");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "订单金额为空或者负数，请检查");
  }

  @Override
  @Transactional
  public PaymentInfo waitPay(String paymentCode, BigDecimal amount) {
    /*
     * 待确认金额付款（线下支付需要有确认环节）
     * 1、检查付款金额是否小于待付款金额
     * 2、检查付款金额+待确认金额是否小于待付款金额
     * 3、增加应付账款的待确认金额
     */
    Validate.notBlank(paymentCode, "应付账款单编号不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "付款单金额不能为负或0");
    PaymentInfo paymentInfo = this.findByPaymentCode(paymentCode);
    Validate.notNull(paymentInfo, "应付账款不能为空");
    Validate.isTrue(amount.compareTo(paymentInfo.getWaitPayAmount()) <= 0, "支付金额大于待支付金额，请检查！");
    Validate.isTrue((amount.add(paymentInfo.getWaitConfirmAmount())).compareTo(paymentInfo.getWaitPayAmount()) <= 0, "支付金额大于待支付金额，请检查！");
    String lockKey = String.format(PAYMENT_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), paymentCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应付账单繁忙，请稍后再试");
    try {
      paymentInfo.setWaitConfirmAmount(paymentInfo.getWaitConfirmAmount().add(amount));
      return this.paymentInfoRepository.saveAndFlush(paymentInfo);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public PaymentInfo cancelWaitPay(String paymentCode, BigDecimal amount) {
    /*
     * 取消待确认金额付款（线下支付需要有确认环节）
     * 1、检查待确认金额是否大于金额
     * 2、检查付款单状态
     */
    Validate.notBlank(paymentCode, "应付账款单编号不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "付款单金额不能为负或0");
    PaymentInfo paymentInfo = this.findByPaymentCode(paymentCode);
    Validate.notNull(paymentInfo, "应付账款不能为空");
    Validate.isTrue((amount.compareTo(paymentInfo.getWaitConfirmAmount())) <= 0, "退款失败，带确认金额小于该次退款金额, 请检查！");
    String lockKey = String.format(PAYMENT_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), paymentCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应付账单繁忙，请稍后再试");
    try {
      paymentInfo.setWaitConfirmAmount(paymentInfo.getWaitConfirmAmount().subtract(amount));
      return this.paymentInfoRepository.saveAndFlush(paymentInfo);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public PaymentInfo confirmPay(String paymentCode, BigDecimal amount, DefrayInfo defrayInfo) {
    /*
     * 确认的支付金额
     * 1、检查待确认金额是否满足
     * 2、减少待确认金额
     * 3、增加已付款金额
     * 4、通知变更事件
     */
    Validate.notBlank(paymentCode, "应付账款单编号不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "付款单金额不能为负或0");
    PaymentInfo paymentInfo = this.findByPaymentCode(paymentCode);
    Validate.notNull(paymentInfo, "应付账款不能为空");
    Validate.isTrue(paymentInfo.getWaitConfirmAmount().compareTo(amount) >= 0, "待确认金额异常，请检查！");
    Validate.isTrue(paymentInfo.getWaitPayAmount().compareTo(amount) >= 0, "确认付款金额超出待确认付款金额！");
    String lockKey = String.format(PAYMENT_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), paymentCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应付账单繁忙，请稍后再试");
    try {
      paymentInfo.setWaitConfirmAmount(paymentInfo.getWaitConfirmAmount().subtract(amount));
      paymentInfo.setWaitPayAmount(paymentInfo.getWaitPayAmount().subtract(amount));
      paymentInfo.setPayedAmount(paymentInfo.getPayedAmount().add(amount));
      // 待付款为0 完成付款信息 通知相关模块
      if (paymentInfo.getWaitPayAmount().compareTo(BigDecimal.ZERO) == 0) {
        paymentInfo.setPaymentStatus(PaymentStatus.ALREADY_PAYMENT.getValue());
        paymentInfo = this.paymentInfoRepository.saveAndFlush(paymentInfo);
        payEvent(paymentInfo, defrayInfo);
        completeEvent(paymentInfo);
      } else {
        if (paymentInfo.getPaymentStatus().intValue() == PaymentStatus.NO_PAYMENT.getValue().intValue()) {
          paymentInfo.setPaymentStatus(PaymentStatus.PART_PAYMENT.getValue());
        }
        paymentInfo = this.paymentInfoRepository.saveAndFlush(paymentInfo);
        payEvent(paymentInfo, defrayInfo);
      }
      return paymentInfo;
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  public PaymentInfo give(String paymentCode, BigDecimal amount, DefrayInfo defrayInfo) {
    /*
     * 直接应付金额（现信用支付使用）
     * 1、检查待确认金额是否满足
     * 2、检查是信用支付的情况下需要确认金额会否足够
     * 3、更新待收款金额
     * 4、更新已收款金额
     * 5、更新用户信用额度
     * 6、通知变更事件
     */
    Validate.notBlank(paymentCode, "应付账款单编号不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "付款单金额不能为负或0");
    PaymentInfo paymentInfo = this.findByPaymentCode(paymentCode);
    Validate.notNull(paymentInfo, "应付账款不能为空");
    Validate.isTrue(paymentInfo.getWaitPayAmount().compareTo(amount) >= 0, "收付单金额异常，请检查！");

    String lockKey = String.format(PAYMENT_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), paymentCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应付账单繁忙，请稍后再试");
    try {
      paymentInfo.setWaitPayAmount(paymentInfo.getWaitPayAmount().subtract(amount));
      paymentInfo.setPayedAmount(paymentInfo.getPayedAmount().add(amount));
      if (FundsChannelType.CREDITPAY.getValue().intValue() == defrayInfo.getFundsChannel().intValue()) {
        this.customerCreditService.recover(defrayInfo.getCustomerCode(), amount, defrayInfo.getDefrayCode(), null);
      }
      if (FundsChannelType.BALANCE.getValue().intValue() == defrayInfo.getFundsChannel().intValue()) {
        this.customerWalletService.recover(defrayInfo.getCustomerCode(), amount, WalletBillBusinessType.REFUND.getType(), defrayInfo.getDefrayCode(), null);
      }

      // 待付款为0 完成付款信息 通知相关模块
      if (paymentInfo.getWaitPayAmount().compareTo(BigDecimal.ZERO) == 0) {
        paymentInfo.setPaymentStatus(PaymentStatus.ALREADY_PAYMENT.getValue());
        paymentInfo = this.paymentInfoRepository.saveAndFlush(paymentInfo);
        payEvent(paymentInfo, defrayInfo);
        completeEvent(paymentInfo);
      } else {
        if (paymentInfo.getPaymentStatus().intValue() == PaymentStatus.NO_PAYMENT.getValue().intValue()) {
          paymentInfo.setPaymentStatus(PaymentStatus.PART_PAYMENT.getValue());
        }
        paymentInfo = this.paymentInfoRepository.saveAndFlush(paymentInfo);
        payEvent(paymentInfo, defrayInfo);
      }
      return paymentInfo;
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  public BigDecimal sumWaitPayAmount() {
    String tenantCode = TenantUtils.getTenantCode();
    return paymentInfoRepository.sumWaitPayAmountByTenantCode(tenantCode);
  }

  /**
   * 生成应付账款号
   *
   * @return
   */
  private String generateNo() {
    Date now = new Date();
    SimpleDateFormat dayFormat = new SimpleDateFormat("yyyyMMdd");
    String redisKey = String.format(PAYMENT_INFO_NO_KEY_PREFIX, TenantUtils.getTenantCode(), dayFormat.format(now));
    // 设置key有效期为一天
    String index = redisMutexService.getAndIncrement(redisKey, 1, 8, 86401, TimeUnit.SECONDS);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(PAYMENT_INFO_NO_PREFIX, format.format(now), index);
  }

  /**
   * 应付账款部分支付状态通知
   *
   * @param paymentInfo 应付账款单
   * @param defrayInfo  付款单
   */
  private void payEvent(PaymentInfo paymentInfo, DefrayInfo defrayInfo) {
    if (CollectionUtils.isNotEmpty(paymentEventListeners)) {
      for (PaymentEventListener eventListener : paymentEventListeners) {
        eventListener.onDefray(paymentInfo, defrayInfo);
      }
    }
  }

  /**
   * 应付账款完成支付状态通知
   *
   * @param paymentInfo 应付账款单
   */
  private void completeEvent(PaymentInfo paymentInfo) {
    if (CollectionUtils.isNotEmpty(paymentEventListeners)) {
      for (PaymentEventListener eventListener : paymentEventListeners) {
        eventListener.onPaymentComplete(paymentInfo);
      }
    }
  }
}
