package com.bizunited.empower.business.payment.service.internal;
import com.bizunited.empower.business.payment.entity.ReceiptAttachment;
import com.bizunited.empower.business.payment.repository.ReceiptAttachmentRepository;
import com.bizunited.empower.business.payment.service.ReceiptAttachmentService;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.Optional;
import java.util.Set;

/**
 * ReceiptFile业务模型的服务层接口实现
 * @author saturn
 */
@Service("ReceiptAttachmentServiceImpl")
public class ReceiptAttachmentServiceImpl implements ReceiptAttachmentService {
  @Autowired
  private ReceiptAttachmentRepository receiptAttachmentRepository;
  @Transactional
  @Override
  public ReceiptAttachment create(ReceiptAttachment receiptAttachment) {
    ReceiptAttachment current = this.createForm(receiptAttachment);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public ReceiptAttachment createForm(ReceiptAttachment receiptAttachment) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    receiptAttachment.setCreateAccount(SecurityUtils.getUserAccount());
    receiptAttachment.setCreateTime(now);
    receiptAttachment.setModifyAccount(SecurityUtils.getUserAccount());
    receiptAttachment.setModifyTime(now);

    this.createValidation(receiptAttachment);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.receiptAttachmentRepository.save(receiptAttachment);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return receiptAttachment;
  }
  /**
   * 在创建一个新的ReceiptFile模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ReceiptAttachment receiptAttachment) {
    Validate.notNull(receiptAttachment, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(receiptAttachment.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    receiptAttachment.setId(null);
    Validate.notBlank(receiptAttachment.getRelativePath(), "添加信息时，相对路径不能为空！");
    Validate.notBlank(receiptAttachment.getFileName(), "添加信息时，原始文件名不能为空！");
    Validate.notBlank(receiptAttachment.getReFileName(), "添加信息时，重命名后的文件名不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(receiptAttachment.getRelativePath() == null || receiptAttachment.getRelativePath().length() < 255 , "相对路径,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(receiptAttachment.getFileName() == null || receiptAttachment.getFileName().length() < 128 , "原始文件名,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(receiptAttachment.getReFileName() == null || receiptAttachment.getReFileName().length() < 128 , "重命名后的文件名,在进行添加时填入值超过了限定长度(128)，请检查!");
  }
  @Transactional
  @Override
  public ReceiptAttachment update(ReceiptAttachment receiptAttachment) {
    ReceiptAttachment current = this.updateForm(receiptAttachment);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public ReceiptAttachment updateForm(ReceiptAttachment receiptAttachment) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(receiptAttachment);
    // ===================基本信息
    String currentId = receiptAttachment.getId();
    Optional<ReceiptAttachment> op_currentReceiptFile = this.receiptAttachmentRepository.findById(currentId);
    ReceiptAttachment currentReceiptAttachment = op_currentReceiptFile.orElse(null);
    Validate.notNull(currentReceiptAttachment,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentReceiptAttachment.setModifyAccount(SecurityUtils.getUserAccount());
    currentReceiptAttachment.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentReceiptAttachment.setRelativePath(receiptAttachment.getRelativePath());
    currentReceiptAttachment.setFileName(receiptAttachment.getFileName());
    currentReceiptAttachment.setReFileName(receiptAttachment.getReFileName());
    currentReceiptAttachment.setReceiptInfo(receiptAttachment.getReceiptInfo());
    
    this.receiptAttachmentRepository.saveAndFlush(currentReceiptAttachment);
    return currentReceiptAttachment;
  }
  /**
   * 在更新一个已有的ReceiptFile模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ReceiptAttachment receiptAttachment) {
    Validate.isTrue(!StringUtils.isBlank(receiptAttachment.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(receiptAttachment.getRelativePath(), "修改信息时，相对路径不能为空！");
    Validate.notBlank(receiptAttachment.getFileName(), "修改信息时，原始文件名不能为空！");
    Validate.notBlank(receiptAttachment.getReFileName(), "修改信息时，重命名后的文件名不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(receiptAttachment.getRelativePath() == null || receiptAttachment.getRelativePath().length() < 255 , "相对路径,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(receiptAttachment.getFileName() == null || receiptAttachment.getFileName().length() < 128 , "原始文件名,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(receiptAttachment.getReFileName() == null || receiptAttachment.getReFileName().length() < 128 , "重命名后的文件名,在进行修改时填入值超过了限定长度(128)，请检查!");
    
    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  } 
  @Override
  public Set<ReceiptAttachment> findDetailsByReceiptInfo(String receiptInfo) {
    if(StringUtils.isBlank(receiptInfo)) { 
      return Sets.newHashSet();
    }
    return this.receiptAttachmentRepository.findDetailsByReceiptInfo(receiptInfo);
  }
  @Override
  public ReceiptAttachment findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.receiptAttachmentRepository.findDetailsById(id);
  }
  @Override
  public ReceiptAttachment findById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<ReceiptAttachment> op = receiptAttachmentRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    ReceiptAttachment current = this.findById(id);
    if(current != null) { 
      this.receiptAttachmentRepository.delete(current);
    }
  }
} 
