package com.bizunited.empower.business.payment.service.internal;

import com.alibaba.fastjson.JSON;
import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.common.enums.ReceiptStatus;
import com.bizunited.empower.business.payment.common.enums.ReceivableType;
import com.bizunited.empower.business.payment.common.enums.WalletBillBusinessType;
import com.bizunited.empower.business.payment.dto.ReceiptDto;
import com.bizunited.empower.business.payment.dto.TransferNoticeDto;
import com.bizunited.empower.business.payment.entity.CustomerBank;
import com.bizunited.empower.business.payment.entity.ReceiptInfo;
import com.bizunited.empower.business.payment.entity.ReceivableInfo;
import com.bizunited.empower.business.payment.repository.ReceiptInfoRepository;
import com.bizunited.empower.business.payment.service.CustomerBankService;
import com.bizunited.empower.business.payment.service.CustomerWalletService;
import com.bizunited.empower.business.payment.service.ReceiptCancelStrategy;
import com.bizunited.empower.business.payment.service.ReceiptInfoService;
import com.bizunited.empower.business.payment.service.ReceiptStrategy;
import com.bizunited.empower.business.payment.service.ReceivableInfoService;
import com.bizunited.empower.business.payment.vo.PayVo;
import com.bizunited.empower.business.payment.vo.ReceiptInfoVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.constant.Constants.RECEIPT_INFO_NO_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.RECEIPT_INFO_LOCK_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.RECEIPT_INFO_NO_KEY_PREFIX;

/**
 * ReceiptInfo业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ReceiptInfoServiceImpl")
public class ReceiptInfoServiceImpl implements ReceiptInfoService {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ReceiptInfoServiceImpl.class);

  @Autowired
  private ReceiptInfoRepository receiptInfoRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private ReceivableInfoService receivableInfoService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private CustomerService customerService;
  @Autowired
  private CustomerBankService customerBankService;
  @Autowired
  private CustomerWalletService customerWalletService;
  @Autowired(required = false)
  private List<ReceiptStrategy> receiptStrategies;
  @Autowired(required = false)
  private List<ReceiptCancelStrategy> receiptCancelStrategies;

  @Override
  @Transactional
  public ReceiptInfo create(ReceiptInfo receiptInfo) {
    return this.create(receiptInfo, null);
  }

  @Transactional
  @Override
  public ReceiptInfo create(ReceiptInfo receiptInfo, InvokeParams params) {
    ReceiptInfo current = this.createForm(receiptInfo, params);
    return current;
  }

  @Override
  @Transactional
  public ReceiptInfo createWithoutConfirm(ReceiptInfo receiptInfo) {
    ReceiptInfo current = this.createForm(receiptInfo, null);
    return this.confirm(current.getReceiptCode());
  }

  @Override
  @Transactional
  public ReceiptInfo createForm(ReceiptInfo receiptInfo) {
    return this.createForm(receiptInfo, null);
  }

  @Transactional
  @Override
  public ReceiptInfo createForm(ReceiptInfo receiptInfo, InvokeParams params) {
    Date now = new Date();
    receiptInfo.setCreateAccount(SecurityUtils.getUserAccount());
    receiptInfo.setCreateTime(now);
    receiptInfo.setModifyAccount(SecurityUtils.getUserAccount());
    receiptInfo.setModifyTime(now);

    // 设置编号、租户、用户等信息
    receiptInfo.setReceiptCode(generateNo());
    receiptInfo.setTenantCode(TenantUtils.getTenantCode());

    this.createValidation(receiptInfo);

    FundsChannelType fundsChannelType = FundsChannelType.valueOfType(receiptInfo.getFundsChannel());
    ReceiptStrategy currentStrategy = getReceiptStrategy(fundsChannelType);
    currentStrategy.handler(receiptInfo, params);
    return receiptInfo;
  }

  /**
   * 在创建一个新的ReceiptInfo模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ReceiptInfo receiptInfo) {
    Validate.notNull(receiptInfo, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(receiptInfo.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    receiptInfo.setId(null);
    Validate.notNull(receiptInfo.getReceiptAmount(), "添加信息时，金额不能为空");
    Validate.notBlank(receiptInfo.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(receiptInfo.getReceiptCode(), "添加信息时，收款单流水编号不能为空！");
    Validate.notNull(receiptInfo.getReceiptAmount(), "添加信息时，收款金额不能为空！");
    Validate.notNull(receiptInfo.getFundsChannel(), "添加信息时，收款资金渠道不能为空!");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(receiptInfo.getReceiptAmount().compareTo(BigDecimal.ZERO) >= 0, "添加信息时，金额不能为负!");
    Validate.isTrue(receiptInfo.getTenantCode() == null || receiptInfo.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(receiptInfo.getReceiptCode() == null || receiptInfo.getReceiptCode().length() < 64, "收款单流水编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.notNull(receiptInfo.getReceivableInfo(), "添加信息时，应收账款信息不能为空!");
    ReceivableInfo receivableInfo = this.receivableInfoService.findByReceivableCode(receiptInfo.getReceivableInfo().getReceivableCode());
    Validate.notNull(receivableInfo, "应收账款信息不能为空");
    receiptInfo.setReceivableInfo(receivableInfo);
    receiptInfo.setCustomerCode(receivableInfo.getCustomerCode());
    ReceiptInfo currentReceiptInfo = this.findByReceiptCode(receiptInfo.getReceiptCode());
    Validate.isTrue(currentReceiptInfo == null, "收款单流水编号已存在,请检查");
  }

  @Override
  public Set<ReceiptInfo> findDetailsByReceivableInfo(String receivableInfo) {
    if (StringUtils.isBlank(receivableInfo)) {
      return Sets.newHashSet();
    }
    return this.receiptInfoRepository.findDetailsByReceivableInfo(receivableInfo);
  }

  @Override
  public ReceiptInfo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.receiptInfoRepository.findDetailsById(id);
  }

  @Override
  public ReceiptInfo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Optional<ReceiptInfo> op = receiptInfoRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public ReceiptInfo findByReceiptCode(String receiptCode) {
    if (StringUtils.isBlank(receiptCode)) {
      return null;
    }
    return this.receiptInfoRepository.findByReceiptCode(receiptCode, TenantUtils.getTenantCode());
  }

  @Override
  public ReceiptInfo findDetailsByReceiptCode(String receiptCode) {
    if (StringUtils.isBlank(receiptCode)) {
      return null;
    }
    return this.receiptInfoRepository.findDetailsByReceiptCode(receiptCode, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public ReceiptInfoVo redoByReceiptCode(String receiptCode, Integer fundsChannel, InvokeParams params) {
    FundsChannelType fundsChannelType = FundsChannelType.valueOfType(fundsChannel);
    ReceiptInfo receiptInfo = this.receiptInfoRepository.findByReceiptCode(receiptCode, TenantUtils.getTenantCode());
    Validate.notNull(receiptInfo, "收款单号错误，请检查！");
    if (fundsChannelType == null) {
      fundsChannelType = FundsChannelType.valueOfType(receiptInfo.getFundsChannel());
    }
    receiptInfo.setFundsChannel(fundsChannelType.getValue());
    ReceiptStrategy currentStrategy = getReceiptStrategy(fundsChannelType);
    currentStrategy.handler(receiptInfo, params);
    ReceiptInfoVo receiptInfoVo = this.nebulaToolkitService.copyObjectByWhiteList(receiptInfo, ReceiptInfoVo.class, HashSet.class, ArrayList.class, "receivableInfo", "receiptAttachments");
    // 第三方支付返回调用扩展参数
    PayVo payVo = (PayVo) params.getInvokeParam("payVo");
    if (payVo != null) {
      receiptInfoVo.setExtend(JSON.toJSONString(payVo));
      receiptInfo.setTxSN(payVo.getTxSN());
      this.receiptInfoRepository.saveAndFlush(receiptInfo);
    }
    return receiptInfoVo;
  }

  @Override
  @Transactional
  public ReceiptInfo confirm(String receiptCode) {
    Validate.notBlank(receiptCode, "业务收款单号不能为空");
    ReceiptInfo receiptInfo = this.receiptInfoRepository.findByReceiptCode(receiptCode, TenantUtils.getTenantCode());
    Validate.notNull(receiptInfo, "收款单数据错误，请联系供应商");
    // 已经是终态的收款单直接返回
    if(receiptInfo.getReceiptStatus().intValue() == ReceiptStatus.CONFIRMED.getValue().intValue() || receiptInfo.getReceiptStatus().intValue() == ReceiptStatus.FAILED.getValue().intValue()){
      return receiptInfo;
    }
    Validate.isTrue(receiptInfo.getReceiptStatus().intValue() == ReceiptStatus.WAITFOR.getValue().intValue(), "收款单状态不是待确认，请检查");
    receiptInfo.setReceiptStatus(ReceiptStatus.CONFIRMED.getValue());
    receiptInfo.setReceiptTime(new Date());
    this.receiptInfoRepository.saveAndFlush(receiptInfo);
    // 确认收款单后 进行应收账款的金额扣除工作
    this.receivableInfoService.confirmReceive(receiptInfo.getReceivableInfo().getReceivableCode(), receiptInfo.getReceiptAmount(), receiptInfo);
    return receiptInfo;
  }

  @Override
  @Transactional
  public ReceiptInfoVo createByAssociatedCode(String associatedCode, BigDecimal amount, FundsChannelType fundsChannelType, InvokeParams params) {
    if (params == null) {
      params = new InvokeParams();
    }
    Validate.notBlank(associatedCode, "关联单据号为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "收款金额不能为负或者0");
    String key = String.format(RECEIPT_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), associatedCode);
    try {
      redisMutexService.lock(key);
      // 信用支付检查当前租户是否开启赊销
      ReceiptInfo receiptInfo = new ReceiptInfo();
      receiptInfo.setReceiptAmount(amount);
      receiptInfo.setFundsChannel(fundsChannelType.getValue());
      receiptInfo.setReceiptTime(new Date());
      ReceivableInfo receivableInfo = this.receivableInfoService.findByAssociatedCode(associatedCode);
      this.checkPayAmount(receivableInfo, amount);
      receiptInfo.setReceivableInfo(receivableInfo);
      receiptInfo = this.create(receiptInfo, params);
      ReceiptInfoVo receiptInfoVo = this.nebulaToolkitService.copyObjectByWhiteList(receiptInfo, ReceiptInfoVo.class, HashSet.class, ArrayList.class, "receivableInfo", "receiptAttachments");
      // 第三方支付返回调用扩展参数
      PayVo payVo = (PayVo) params.getInvokeParam("payVo");
      if (payVo != null) {
        receiptInfoVo.setExtend(JSON.toJSONString(payVo));
        receiptInfo.setTxSN(payVo.getTxSN());
        this.receiptInfoRepository.saveAndFlush(receiptInfo);
      }
      return receiptInfoVo;
    } finally {
      redisMutexService.unlock(key);
    }
  }

  @Override
  @Transactional
  public ReceiptInfo cancelByReceiptCode(String receiptCode) {
    Validate.notBlank(receiptCode, "业务收款单号不能为空");
    ReceiptInfo receiptInfo = this.receiptInfoRepository.findByReceiptCode(receiptCode, TenantUtils.getTenantCode());
    Validate.notNull(receiptInfo, "收款单数据错误，请联系供应商");
    Validate.isTrue(receiptInfo.getReceiptStatus().intValue() == ReceiptStatus.WAITFOR.getValue().intValue(), "收款单状态不是待确认，请检查");
    FundsChannelType fundsChannelType = FundsChannelType.valueOfType(receiptInfo.getFundsChannel());
    ReceiptCancelStrategy currentStrategy = getReceiptCancelStrategy(fundsChannelType);
    // 后续扩展参数使用
    InvokeParams params = new InvokeParams();
    currentStrategy.handler(receiptInfo, params);
    return receiptInfo;
  }

  @Override
  public ReceiptInfo failedByReceiptCode(String receiptCode) {
    Validate.notBlank(receiptCode, "业务收款单号不能为空");
    ReceiptInfo receiptInfo = this.receiptInfoRepository.findByReceiptCode(receiptCode, TenantUtils.getTenantCode());
    Validate.notNull(receiptInfo, "收款单数据错误，请联系供应商");
    Validate.isTrue(receiptInfo.getReceiptStatus().intValue() == ReceiptStatus.WAITFOR.getValue().intValue(), "收款单状态不是待确认，请检查");
    receiptInfo.setReceiptStatus(ReceiptStatus.FAILED.getValue());
    receiptInfo = this.receiptInfoRepository.saveAndFlush(receiptInfo);
    this.receivableInfoService.cancelWaitReceive(receiptInfo.getReceivableInfo().getReceivableCode(), receiptInfo.getReceiptAmount());
    return receiptInfo;
  }

  @Override
  public Page<ReceiptInfoVo> findByConditions(Pageable pageable, InvokeParams conditions) {
    conditions.getInvokeParams().put("tenantCode", TenantUtils.getTenantCode());
    Page<Object> page = this.receiptInfoRepository.queryPage(pageable, conditions);
    List<ReceiptInfoVo> list = Lists.newArrayList();
    if (page != null && page.getTotalElements() > 0L) {
      page.getContent().stream().forEach(data -> {
        ReceiptInfo receiptInfo = this.receiptInfoRepository.findDetailsById((String) data);
        ReceiptInfoVo vo = nebulaToolkitService.copyObjectByWhiteList(receiptInfo, ReceiptInfoVo.class, HashSet.class, ArrayList.class, "receivableInfo");
        Customer customer = customerService.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), vo.getCustomerCode());
        Validate.notNull(customer, "客户信息异常，请检查！");
        vo.setCustomerName(customer.getCustomerName());
        list.add(vo);
      });
    } else {
      return null;
    }
    return new PageImpl<>(list, pageable, page.getTotalElements());
  }

  @Override
  @Transactional(rollbackOn = Exception.class)
  public void handlerTransfer(TransferNoticeDto transferNoticeDto) {
    /*
     * 1、查询用户所有的应收账款（根据时间）
     * 2、统计转账的金额进行应收账款的收款单的创建
     *  2.1、转账金额小于应收账款，部分付款
     *  2.2、转账金额等于应收账款，完成付款
     *  2.3、
     * 3、如果所有应收账款都完成后，剩余的金额转入用户个人余额账户
     */
    if (2 != transferNoticeDto.getStatus()) {
      LOGGER.error("线下转账通知状态异常,流水号:{}, 状态为:{}", transferNoticeDto.getTxSN(), transferNoticeDto.getTxSN());
      return;
    }
    BigDecimal balance = transferNoticeDto.getAmount();
    String tenantCode = transferNoticeDto.getTenantCode();
    String txSN = transferNoticeDto.getTxSN();
    String cardNum = transferNoticeDto.getCardNum();
    CustomerBank customerBank = customerBankService.findByCardNumAndTenantCode(cardNum, tenantCode);
    String customerCode = customerBank.getCustomerCode();
    List<ReceivableInfo> receivableInfos = receivableInfoService.findByCustomerAndTenantCode(customerCode, tenantCode);
    if (CollectionUtils.isEmpty(receivableInfos)) {
      return;
    }
    for (ReceivableInfo receivable : receivableInfos) {
      BigDecimal waitReceiveAmount = receivable.getWaitReceiveAmount();
      BigDecimal waitConfirmAmount = receivable.getWaitConfirmAmount();
      waitReceiveAmount = waitReceiveAmount.subtract(waitConfirmAmount);
      if (balance.compareTo(waitReceiveAmount) >= 0) {
        ReceiptInfo receiptInfo = new ReceiptInfo();
        receiptInfo.setReceivableInfo(receivable);
        receiptInfo.setReceiptAmount(waitReceiveAmount);
        receiptInfo.setFundsChannel(FundsChannelType.TRANSFER.getValue());
        receiptInfo.setReceiptTime(new Date());
        receiptInfo.setTenantCode(tenantCode);
        this.createWithoutConfirm(receiptInfo);
        balance = balance.subtract(waitReceiveAmount);
      } else {
        ReceiptInfo receiptInfo = new ReceiptInfo();
        receiptInfo.setReceivableInfo(receivable);
        receiptInfo.setReceiptAmount(balance);
        receiptInfo.setFundsChannel(FundsChannelType.TRANSFER.getValue());
        receiptInfo.setReceiptTime(new Date());
        receiptInfo.setTenantCode(tenantCode);
        this.createWithoutConfirm(receiptInfo);
        balance = BigDecimal.ZERO;
        break;
      }
    }
    // 完成所有订单后还有剩余
    if (balance.compareTo(BigDecimal.ZERO) > 0) {
      customerWalletService.recover(customerCode, balance, WalletBillBusinessType.TRANSFER_BALANCE.getType(), txSN, null);
    }
  }

  @Override
  public BigDecimal findTotalByConditions(Pageable pageable, InvokeParams conditions) {
    BigDecimal receivedAmountTotal = new BigDecimal(BigInteger.ZERO);
    pageable = PageRequest.of(0, Integer.MAX_VALUE);
    Page<ReceiptInfoVo> byConditions = this.findByConditions(pageable, conditions);
    if (null != byConditions && !CollectionUtils.isEmpty(byConditions.getContent())) {
      List<ReceiptInfoVo> content = byConditions.getContent();
      receivedAmountTotal = content.stream().map(ReceiptInfoVo::getReceiptAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
    }
    return receivedAmountTotal;
  }

  @Override
  public List<ReceiptInfo> findWaitForByCustomerCodeAndFundsChannels(String customerCode, List<Integer> fundsChannels) {
    Validate.isTrue(!CollectionUtils.isEmpty(fundsChannels), "资金类型为空，请检查！");
    return this.receiptInfoRepository.findWaitForByCustomerCodeAndFundsChannelsAndTenantCode(customerCode, fundsChannels, ReceiptStatus.WAITFOR.getValue(), TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public ReceiptInfoVo createByCustomerCodeAndFundsChannels(String customerCode, BigDecimal amount, Integer fundsChannel, InvokeParams params) {
    if (params == null) {
      params = new InvokeParams();
    }
    Validate.notBlank(customerCode, "客户编号为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "收款金额不能为负或者0");
    String key = String.format(RECEIPT_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), customerCode);
    try {
      redisMutexService.lock(key);
      // 信用支付检查当前租户是否开启赊销
      ReceiptInfo receiptInfo = new ReceiptInfo();
      receiptInfo.setReceiptAmount(amount);
      receiptInfo.setFundsChannel(fundsChannel);
      receiptInfo.setReceiptTime(new Date());
      ReceivableInfo receivableInfo = this.receivableInfoService.createByCustomerCode(customerCode, amount, ReceivableType.OTHER);
      this.checkPayAmount(receivableInfo, amount);
      receiptInfo.setReceivableInfo(receivableInfo);
      receiptInfo = this.create(receiptInfo, params);
      ReceiptInfoVo receiptInfoVo = this.nebulaToolkitService.copyObjectByWhiteList(receiptInfo, ReceiptInfoVo.class, HashSet.class, ArrayList.class, "receivableInfo", "receiptAttachments");
      // 第三方支付返回调用扩展参数
      PayVo payVo = (PayVo) params.getInvokeParam("payVo");
      if (payVo != null) {
        receiptInfoVo.setExtend(JSON.toJSONString(payVo));
        receiptInfo.setTxSN(payVo.getTxSN());
        this.receiptInfoRepository.saveAndFlush(receiptInfo);
      }
      return receiptInfoVo;
    } finally {
      redisMutexService.unlock(key);
    }
  }

  @Override
  public List<ReceiptInfo> findByReceipts(List<String> receiptCodes) {
    return receiptInfoRepository.findByReceiptCodes(receiptCodes);
  }

  /**
   * 生成业务编号
   *
   * @return
   */
  private String generateNo() {
    Date now = new Date();
    SimpleDateFormat dayFormat = new SimpleDateFormat("yyyyMMdd");
    String redisKey = String.format(RECEIPT_INFO_NO_KEY_PREFIX, TenantUtils.getTenantCode(), dayFormat.format(now));
    // 设置key有效期为一天
    String index = redisMutexService.getAndIncrement(redisKey, 1, 8, 86401, TimeUnit.SECONDS);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(RECEIPT_INFO_NO_PREFIX, format.format(now), index);
  }

  @Override
  public BigDecimal sumTotalReceiptAmountByAccountAndReceiptTime(String account, Date receiptTime) {
    if (StringUtils.isAnyBlank(account, TenantUtils.getTenantCode()) || receiptTime == null) {
      return BigDecimal.ZERO;
    }
    return receiptInfoRepository.sumTotalReceiptAmountByAccountAndReceiptTimeAndTenantCode(account, receiptTime, TenantUtils.getTenantCode());
  }

  @Override
  public BigInteger countTotalReceiptNumByAccountAndReceiptTime(String account, Date receiptTime) {
    if (StringUtils.isAnyBlank(account, TenantUtils.getTenantCode()) || receiptTime == null) {
      return BigInteger.ZERO;
    }
    return BigInteger.valueOf(receiptInfoRepository.countTotalReceiptNumByAccountAndReceiptTimeAndTenantCode(account, receiptTime, TenantUtils.getTenantCode()));
  }

  @Override
  @Transactional
  public ReceiptInfoVo create(ReceiptDto receiptDto) {
    Validate.notNull(receiptDto, "收款单参数为空，请检查！");
    Validate.notBlank(receiptDto.getAssociatedCode(), "关联单据号为空");
    Validate.notNull(receiptDto.getFundsChannel(), "收款单资金渠道为空，请检查！");
    Validate.isTrue(receiptDto.getReceiptAmount() != null && receiptDto.getReceiptAmount().compareTo(BigDecimal.ZERO) > 0, "收款金额不能为负或者0");
    String key = String.format(RECEIPT_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), receiptDto.getAssociatedCode());
    try {
      redisMutexService.lock(key);
      InvokeParams params = new InvokeParams();
      ReceiptInfo receiptInfo = this.nebulaToolkitService.copyObjectByWhiteList(receiptDto, ReceiptInfo.class, HashSet.class, ArrayList.class, "receiptAttachments");
      ReceivableInfo receivableInfo = this.receivableInfoService.findByAssociatedCode(receiptDto.getAssociatedCode());
      receiptInfo.setReceivableInfo(receivableInfo);
      receiptInfo = this.create(receiptInfo, params);
      ReceiptInfoVo receiptInfoVo = this.nebulaToolkitService.copyObjectByWhiteList(receiptInfo, ReceiptInfoVo.class, HashSet.class, ArrayList.class, "receivableInfo", "receiptAttachments");
      // 第三方支付返回调用扩展参数
      PayVo payVo = (PayVo) params.getInvokeParam("payVo");
      if (payVo != null) {
        receiptInfoVo.setExtend(JSON.toJSONString(payVo));
        receiptInfo.setTxSN(payVo.getTxSN());
        this.receiptInfoRepository.saveAndFlush(receiptInfo);
      }
      return receiptInfoVo;
    } finally {
      redisMutexService.unlock(key);
    }
  }

  /**
   * 获取当前执行策略
   *
   * @param type
   * @return
   */
  private ReceiptStrategy getReceiptStrategy(FundsChannelType type) {
    Optional<ReceiptStrategy> receiptStrategyOp = receiptStrategies.stream().filter(item -> type.getValue() == item.getFundsChannel().intValue()).findFirst();
    ReceiptStrategy receiptStrategy = receiptStrategyOp.orElse(null);
    Validate.notNull(receiptStrategy, String.format("类型【%d】没有对应的处理策略，请检查！", type.getValue()));
    return receiptStrategy;
  }

  /**
   * 获取当前取消收款单执行策略
   *
   * @param type
   * @return
   */
  private ReceiptCancelStrategy getReceiptCancelStrategy(FundsChannelType type) {
    Optional<ReceiptCancelStrategy> receiptCancelStrategyOp = receiptCancelStrategies.stream().filter(item -> type.getValue() == item.getFundsChannel().intValue()).findFirst();
    ReceiptCancelStrategy receiptCancelStrategy = receiptCancelStrategyOp.orElse(null);
    Validate.notNull(receiptCancelStrategy, String.format("类型【%d】没有对应的处理策略，请检查！", type.getValue()));
    return receiptCancelStrategy;
  }

  /**
   * 检查支付金额是否超出应付账单剩余金额
   */
  private void checkPayAmount(ReceivableInfo receivableInfo,BigDecimal amount){
    Validate.isTrue(amount!=null && amount.compareTo(BigDecimal.ZERO) > 0,"金额不能小于等于0");
    BigDecimal waitReceiveAmount = receivableInfo.getWaitReceiveAmount();
    Validate.isTrue(waitReceiveAmount.compareTo(amount) >= 0,"支付金额不能超出待收款金额");
    BigDecimal waitConfirmAmount = receivableInfo.getWaitConfirmAmount();
    Validate.isTrue(waitReceiveAmount.subtract(waitConfirmAmount).compareTo(amount) >=0,"支付金额不能超出待收款金额（请重试）");
  }
}
