package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.common.enums.PaymentType;
import com.bizunited.empower.business.payment.common.enums.ReceivableStatus;
import com.bizunited.empower.business.payment.common.enums.ReceivableType;
import com.bizunited.empower.business.payment.common.enums.WalletBillBusinessType;
import com.bizunited.empower.business.payment.entity.ReceiptInfo;
import com.bizunited.empower.business.payment.entity.ReceivableInfo;
import com.bizunited.empower.business.payment.repository.ReceivableInfoRepository;
import com.bizunited.empower.business.payment.service.CustomerCreditService;
import com.bizunited.empower.business.payment.service.CustomerWalletService;
import com.bizunited.empower.business.payment.service.DefrayInfoService;
import com.bizunited.empower.business.payment.service.PaymentInfoService;
import com.bizunited.empower.business.payment.service.ReceivableInfoService;
import com.bizunited.empower.business.payment.service.notifier.ReceivableEventListener;
import com.bizunited.empower.business.payment.vo.AssociatedReceivableVo;
import com.bizunited.empower.business.payment.vo.CustomerReceivableVo;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.constant.Constants.RECEIVABLE_INFO_NO_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.RECEIVABLE_INFO_LOCK_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.RECEIVABLE_INFO_NO_KEY_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.RECEIVABLE_ORDER_LOCK_PREFIX;

/**
 * ReceivableInfo业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ReceivableInfoServiceImpl")
public class ReceivableInfoServiceImpl implements ReceivableInfoService {
  @Autowired
  private ReceivableInfoRepository receivableInfoRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired(required = false)
  private List<ReceivableEventListener> receivableEventListeners;
  @Autowired
  private CustomerService customerService;
  @Autowired
  private CustomerCreditService customerCreditService;
  @Autowired
  private CustomerWalletService customerWalletService;
  @Autowired
  private DefrayInfoService defrayInfoService;
  @Autowired
  private PaymentInfoService paymentInfoService;

  @Override
  public ReceivableInfo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.receivableInfoRepository.findDetailsById(id);
  }

  @Override
  public ReceivableInfo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ReceivableInfo> op = receivableInfoRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public ReceivableInfo findByReceivableCode(String receivableCode) {
    if (StringUtils.isBlank(receivableCode)) {
      return null;
    }
    return this.receivableInfoRepository.findByReceivableCodeAndTenantCode(receivableCode, TenantUtils.getTenantCode());
  }

  @Override
  public ReceivableInfo findByAssociatedCode(String associatedCode) {
    if (StringUtils.isBlank(associatedCode)) {
      return null;
    }
    return this.receivableInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
  }

  @Override
  public List<ReceivableInfo> findByAssociatedCodes(Set<String> associatedCodes) {
    if (CollectionUtils.isEmpty(associatedCodes)) {
      return Lists.newArrayList();
    }
    return this.receivableInfoRepository.findByAssociatedCodesAndTenantCode(associatedCodes, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public ReceivableInfo createByAssociatedCode(String customerCode, String associatedCode, BigDecimal amount, ReceivableType receivableType) {
    /*
     * 1、根据订单创建应收账款数据
     *  1.1、初始化应收账款信息
     *  1.2、检查订单是否已经存在应收账款数据
     * 2、创建收款单
     */
    this.createByAssociatedCodeValidation(customerCode, associatedCode, amount, receivableType);

    String lockKey = null;
    if (StringUtils.isNotBlank(associatedCode)) {
      lockKey = String.format(RECEIVABLE_ORDER_LOCK_PREFIX, TenantUtils.getTenantCode(), associatedCode);
    } else {
      lockKey = String.format(RECEIVABLE_ORDER_LOCK_PREFIX, TenantUtils.getTenantCode(), customerCode);
    }
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应收账单繁忙，请稍后再试");
    Customer customer = customerService.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
    Validate.notNull(customer, "客户信息异常，请检查");
    ReceivableInfo receivableInfo = new ReceivableInfo();
    try {
      Date now = new Date();
      receivableInfo.setCreateAccount(SecurityUtils.getUserAccount());
      receivableInfo.setCreateTime(now);
      receivableInfo.setModifyAccount(SecurityUtils.getUserAccount());
      receivableInfo.setModifyTime(now);
      receivableInfo.setAssociatedCode(associatedCode);
      receivableInfo.setTenantCode(TenantUtils.getTenantCode());
      receivableInfo.setReceivableCode(generateNo());
      receivableInfo.setCustomerCode(customerCode);
      receivableInfo.setCustomerName(customer.getCustomerName());
      receivableInfo.setReceivableType(receivableType.getValue());
      // 待收款金额
      receivableInfo.setWaitReceiveAmount(amount);
      // 待确认金额
      receivableInfo.setWaitConfirmAmount(BigDecimal.ZERO);
      // 应收款金额
      receivableInfo.setReceivableAmount(amount);
      // 已收款金额
      receivableInfo.setReceivedAmount(BigDecimal.ZERO);
      receivableInfo.setReceivableStatus(ReceivableStatus.NO_RECEIVE.getValue());
      receivableInfo.setTstatus(1);
      receivableInfo = this.receivableInfoRepository.save(receivableInfo);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
    return receivableInfo;
  }


  @Override
  public ReceivableInfo createByCustomerCode(String customerCode, BigDecimal amount, ReceivableType receivableType) {
    /*
     * 1、根据订单创建应收账款数据
     *  1.1、初始化应收账款信息
     *  1.2、检查订单是否已经存在应收账款数据
     * 2、创建收款单
     */
    this.createByCustomerCodeValidation(customerCode, amount, receivableType);

    String lockKey = String.format(RECEIVABLE_ORDER_LOCK_PREFIX, TenantUtils.getTenantCode(), customerCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应收账单繁忙，请稍后再试");
    Customer customer = customerService.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
    Validate.notNull(customer, "客户信息异常，请检查");
    ReceivableInfo receivableInfo = new ReceivableInfo();
    try {
      Date now = new Date();
      receivableInfo.setCreateAccount(SecurityUtils.getUserAccount());
      receivableInfo.setCreateTime(now);
      receivableInfo.setModifyAccount(SecurityUtils.getUserAccount());
      receivableInfo.setModifyTime(now);
      receivableInfo.setTenantCode(TenantUtils.getTenantCode());
      receivableInfo.setReceivableCode(generateNo());
      receivableInfo.setCustomerCode(customerCode);
      receivableInfo.setCustomerName(customer.getCustomerName());
      receivableInfo.setReceivableType(receivableType.getValue());
      // 待收款金额
      receivableInfo.setWaitReceiveAmount(amount);
      // 待确认金额
      receivableInfo.setWaitConfirmAmount(BigDecimal.ZERO);
      // 应收款金额
      receivableInfo.setReceivableAmount(amount);
      // 已收款金额
      receivableInfo.setReceivedAmount(BigDecimal.ZERO);
      receivableInfo.setReceivableStatus(ReceivableStatus.NO_RECEIVE.getValue());
      receivableInfo.setTstatus(1);
      receivableInfo = this.receivableInfoRepository.save(receivableInfo);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
    return receivableInfo;
  }

  @Override
  @Transactional
  public ReceivableInfo updateByAssociatedCode(String associatedCode, BigDecimal amount) {
    /*
     * 更新应收账款(待增加日志功能)
     * 1、参数有效性验证
     * 2、更新待收金额、待确认金额、收款金额、应收金额、已收金额
     */
    this.updateByAssociatedCodeValidation(associatedCode, amount);
    ReceivableInfo receivableInfo = this.receivableInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
    Validate.notNull(receivableInfo, "应收账款单不存在，请检查！");
    Validate.isTrue(receivableInfo.getReceivableStatus().intValue() == ReceivableStatus.NO_RECEIVE.getValue().intValue(), "该应收账款单已部分付款或完成付款，不能进行修改");
    receivableInfo.setWaitReceiveAmount(amount);
    receivableInfo.setWaitConfirmAmount(BigDecimal.ZERO);
    receivableInfo.setReceivableAmount(amount);
    receivableInfo.setReceivedAmount(BigDecimal.ZERO);
    return this.receivableInfoRepository.save(receivableInfo);
  }

  @Override
  @Transactional
  public ReceivableInfo cancelByAssociatedCode(String associatedCode) {
    /*
     * 取消应收账款单
     * 1、判定应收账款单状态
     * 2、如果没有进行支付则直接进行应付账单的取消
     * 3、已经收款或者部分收款的订单，创建应付账单冲账，应付账单管理当前订单并自动创建付款单至余额支付，金额为用户已经付款的金额
     */
    ReceivableInfo receivableInfo = this.receivableInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
    Validate.notNull(receivableInfo, "应收账款单不存在，请检查！");

    Validate.isTrue(receivableInfo.getWaitConfirmAmount().compareTo(BigDecimal.ZERO) == 0, "该应收账单【%s】存在待确认收款单，请收款后在进行相关操作。", receivableInfo.getReceivableCode());

    if (receivableInfo.getReceivableStatus().intValue() != ReceivableStatus.NO_RECEIVE.getValue().intValue()) {
      // 已经部分收款获取全部收款的情况，需要创建付款单信息
      paymentInfoService.createByAssociatedCode(receivableInfo.getCustomerCode(), receivableInfo.getAssociatedCode(), receivableInfo.getReceivedAmount(), PaymentType.OTHER);
      defrayInfoService.createByAssociatedCode(receivableInfo.getAssociatedCode(), receivableInfo.getReceivedAmount(), FundsChannelType.BALANCE);
      // 冲账订单
      receivableInfo.setReceivableStatus(5);
    } else {
      receivableInfo.setTstatus(0);
    }
    return this.receivableInfoRepository.saveAndFlush(receivableInfo);
  }

  @Override
  public Page<AssociatedReceivableVo> findByConditionsForAssociated(Pageable pageable, InvokeParams conditions) {
    conditions.getInvokeParams().put("tenantCode", TenantUtils.getTenantCode());
    Page<AssociatedReceivableVo> page = this.receivableInfoRepository.queryPageForAssociated(pageable, conditions);
    return page;
  }

  @Override
  public Page<CustomerReceivableVo> findByConditionsForCustomer(Pageable pageable, InvokeParams conditions) {
    conditions.getInvokeParams().put("tenantCode", TenantUtils.getTenantCode());
    Page<Object[]> page = this.receivableInfoRepository.queryPageForCustomer(pageable, conditions);
    List<CustomerReceivableVo> list = Lists.newArrayList();
    if (page != null && page.getTotalElements() > 0L) {
      page.getContent().stream().forEach(data -> {
        CustomerReceivableVo vo = new CustomerReceivableVo();
        vo.setCustomerCode((String) data[0]);
        vo.setCustomerName((String) data[1]);
        vo.setCustomerLevel((String) data[2]);
        vo.setCustomerCategory((String) data[3]);
        vo.setSalesName((String) data[4]);
        vo.setReceivableNum(((BigInteger) data[5]).longValue());
        vo.setReceivableAmount(((BigDecimal) data[6]));
        vo.setReceivedAmount(((BigDecimal) data[7]));
        vo.setWaitReceiveAmount(((BigDecimal) data[8]));
        vo.setWaitConfirmAmount(((BigDecimal) data[9]));
        vo.setId(((String) data[10]));
        list.add(vo);
      });
    } else {
      return null;
    }
    return new PageImpl<>(list, pageable, page.getTotalElements());
  }

  @Override
  public CustomerReceivableVo findAggregationByCustomerCode(String customerCode) {
    Map<String, Object> data = this.receivableInfoRepository.findAggregationByCustomerCode(customerCode, TenantUtils.getTenantCode());
    CustomerReceivableVo vo = new CustomerReceivableVo();
    if (data.isEmpty()) {
      vo.setCustomerCode(customerCode);
      vo.setReceivableNum(0L);
      vo.setReceivableAmount(BigDecimal.ZERO);
      vo.setReceivedAmount(BigDecimal.ZERO);
      vo.setWaitReceiveAmount(BigDecimal.ZERO);
      vo.setWaitConfirmAmount(BigDecimal.ZERO);
    } else {
      vo.setCustomerCode((String) data.get("customerCode"));
      vo.setReceivableNum(((BigInteger) data.get("receivableNum")).longValue());
      vo.setReceivableAmount(((BigDecimal) data.get("receivableAmount")));
      vo.setReceivedAmount(((BigDecimal) data.get("receivedAmount")));
      vo.setWaitReceiveAmount(((BigDecimal) data.get("waitReceiveAmount")));
      vo.setWaitConfirmAmount(((BigDecimal) data.get("waitConfirmAmount")));
    }
    return vo;
  }

  /**
   * 订单创建参数验证
   *
   * @param customerCode
   * @param associatedCode
   * @param amount
   * @param receivableType
   */
  private void createByAssociatedCodeValidation(String customerCode, String associatedCode, BigDecimal amount, ReceivableType receivableType) {
    Validate.notBlank(customerCode, "客户编号不存在，请检查");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "订单金额为空或者负数，请检查");
    Validate.notNull(receivableType, "应收账款类型异常，请检查");
    if (StringUtils.isNotBlank(associatedCode)) {
      ReceivableInfo receivableInfo = this.receivableInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
      Validate.isTrue(receivableInfo == null, "该订单的是应收账款已存在");
    }
  }

  /**
   * 客户创建参数验证
   *
   * @param customerCode
   * @param amount
   * @param receivableType
   */
  private void createByCustomerCodeValidation(String customerCode, BigDecimal amount, ReceivableType receivableType) {
    Validate.notBlank(customerCode, "客户编号不存在，请检查");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "订单金额为空或者负数，请检查");
    Validate.notNull(receivableType, "应收账款类型异常，请检查");
  }

  /**
   * 订单更新参数验证
   *
   * @param associatedCode
   * @param amount
   */
  private void updateByAssociatedCodeValidation(String associatedCode, BigDecimal amount) {
    Validate.notBlank(associatedCode, "关联单据不存在，请检查");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "订单金额为空或者负数，请检查");
  }

  @Override
  @Transactional
  public ReceivableInfo waitReceive(String receivableCode, BigDecimal amount) {
    /*
     * 待确认金额收款（线下支付需要有确认环节）
     * 1、检查收款金额是否小于待收款金额
     * 2、检查收款金额+待确认金额是否小于待收款金额
     * 3、增加应收账款的待确认金额
     */
    Validate.notBlank(receivableCode, "应收账款单编号不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "收款单金额不能为负或0");
    ReceivableInfo receivableInfo = this.findByReceivableCode(receivableCode);
    Validate.notNull(receivableInfo, "应收账款不能为空");
    Validate.isTrue(amount.compareTo(receivableInfo.getWaitReceiveAmount()) <= 0, "收款单金额超出待收金额");
    Validate.isTrue((amount.add(receivableInfo.getWaitConfirmAmount())).compareTo(receivableInfo.getWaitReceiveAmount()) <= 0, "收款单金额超出待收金额");
    String lockKey = String.format(RECEIVABLE_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), receivableCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应收账单繁忙，请稍后再试");
    try {
      receivableInfo.setWaitConfirmAmount(receivableInfo.getWaitConfirmAmount().add(amount));
      return this.receivableInfoRepository.saveAndFlush(receivableInfo);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public ReceivableInfo cancelWaitReceive(String receivableCode, BigDecimal amount) {
    /*
     * 取消待确认金额收款（线下支付需要有确认环节）
     * 1、检查待确认金额是否大于金额
     * 2、检查收款单状态
     */
    Validate.notBlank(receivableCode, "应收账款单编号不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "收款单金额不能为负或0");
    ReceivableInfo receivableInfo = this.findByReceivableCode(receivableCode);
    Validate.notNull(receivableInfo, "应收账款不能为空");
    Validate.isTrue((amount.compareTo(receivableInfo.getWaitConfirmAmount())) <= 0, "收款单金额超出待收金额");
    String lockKey = String.format(RECEIVABLE_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), receivableCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应收账单繁忙，请稍后再试");
    try {
      receivableInfo.setWaitConfirmAmount(receivableInfo.getWaitConfirmAmount().subtract(amount));
      return this.receivableInfoRepository.saveAndFlush(receivableInfo);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public ReceivableInfo confirmReceive(String receivableCode, BigDecimal amount, ReceiptInfo receiptInfo) {
    /*
     * 确认的支付金额
     * 1、检查待确认金额是否满足
     * 2、减少待确认金额
     * 3、增加已收款金额
     * 4、通知变更事件
     */
    Validate.notBlank(receivableCode, "应收账款单编号不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "收款单金额不能为负或0");
    ReceivableInfo receivableInfo = this.findByReceivableCode(receivableCode);
    Validate.notNull(receivableInfo, "应收账款不能为空");
    Validate.isTrue(receivableInfo.getWaitConfirmAmount().compareTo(amount) >= 0, "待确认金额异常，请检查");
    Validate.isTrue(receivableInfo.getWaitReceiveAmount().compareTo(amount) >= 0, "待确认金额异常，请检查");
    String lockKey = String.format(RECEIVABLE_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), receivableCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应收账单繁忙，请稍后再试");
    try {
      receivableInfo.setWaitConfirmAmount(receivableInfo.getWaitConfirmAmount().subtract(amount));
      receivableInfo.setWaitReceiveAmount(receivableInfo.getWaitReceiveAmount().subtract(amount));
      receivableInfo.setReceivedAmount(receivableInfo.getReceivedAmount().add(amount));
      // 待收款为0 完成收款信息 通知相关模块
      if (receivableInfo.getWaitReceiveAmount().compareTo(BigDecimal.ZERO) == 0) {
        receivableInfo.setReceivableStatus(ReceivableStatus.ALREADY_RECEIVE.getValue());
        receivableInfo = this.receivableInfoRepository.saveAndFlush(receivableInfo);
        receiptEvent(receivableInfo, receiptInfo);
        completeEvent(receivableInfo);
      } else {
        if (receivableInfo.getReceivableStatus().intValue() == ReceivableStatus.NO_RECEIVE.getValue().intValue()) {
          receivableInfo.setReceivableStatus(ReceivableStatus.PART_RECEIVE.getValue());
        }
        receivableInfo = this.receivableInfoRepository.saveAndFlush(receivableInfo);
        receiptEvent(receivableInfo, receiptInfo);
      }
      return receivableInfo;
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  @Transactional
  public ReceivableInfo receive(String receivableCode, BigDecimal amount, ReceiptInfo receiptInfo) {
    /*
     * 直接支付金额（现信用支付,余额支付使用）
     * 1、检查待确认金额是否满足
     * 2、检查是信用支付的情况下需要确认金额会否足够
     * 3、更新待收款金额
     * 4、更新已收款金额
     * 5、更新用户信用额度
     * 6、通知变更事件
     */
    Validate.notBlank(receivableCode, "应收账款单编号不能为空");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "收款单金额不能为负或0");
    ReceivableInfo receivableInfo = this.findByReceivableCode(receivableCode);
    Validate.notNull(receivableInfo, "应收账款不能为空");
    Validate.isTrue(receivableInfo.getWaitReceiveAmount().compareTo(amount) >= 0, "收款单金额异常，请检查！");
    if (FundsChannelType.CREDITPAY.getValue().intValue() == receiptInfo.getFundsChannel().intValue()) {
      Boolean validReceiptFlg = customerCreditService.validReceipt(receiptInfo.getCustomerCode(), amount);
      Validate.isTrue(validReceiptFlg, "用户信用额度不足，请联系经销商");
    }
    if (FundsChannelType.BALANCE.getValue().intValue() == receiptInfo.getFundsChannel().intValue()) {
      Boolean validReceiptFlg = customerWalletService.validReceipt(receiptInfo.getCustomerCode(), amount);
      Validate.isTrue(validReceiptFlg, "用户余额额度不足，请联系经销商");
    }
    String lockKey = String.format(RECEIVABLE_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), receivableCode);
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建应收账单繁忙，请稍后再试");
    try {
      receivableInfo.setWaitReceiveAmount(receivableInfo.getWaitReceiveAmount().subtract(amount));
      receivableInfo.setReceivedAmount(receivableInfo.getReceivedAmount().add(amount));
      if (FundsChannelType.CREDITPAY.getValue().intValue() == receiptInfo.getFundsChannel().intValue()) {
        this.customerCreditService.receipt(receiptInfo.getCustomerCode(), amount, receiptInfo.getReceiptCode(), null);
      }
      if (FundsChannelType.BALANCE.getValue().intValue() == receiptInfo.getFundsChannel().intValue()) {
        this.customerWalletService.receipt(receiptInfo.getCustomerCode(), amount, WalletBillBusinessType.PAYMENT.getType(), receiptInfo.getReceiptCode(), null);
      }

      // 待收款为0 完成收款信息 通知相关模块
      if (receivableInfo.getWaitReceiveAmount().compareTo(BigDecimal.ZERO) == 0) {
        receivableInfo.setReceivableStatus(ReceivableStatus.ALREADY_RECEIVE.getValue());
        receivableInfo = this.receivableInfoRepository.saveAndFlush(receivableInfo);
        receiptEvent(receivableInfo, receiptInfo);
        completeEvent(receivableInfo);
      } else {
        if (receivableInfo.getReceivableStatus().intValue() == ReceivableStatus.NO_RECEIVE.getValue().intValue()) {
          receivableInfo.setReceivableStatus(ReceivableStatus.PART_RECEIVE.getValue());
        }
        receivableInfo = this.receivableInfoRepository.saveAndFlush(receivableInfo);
        receiptEvent(receivableInfo, receiptInfo);
      }
      return receivableInfo;
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
  }

  @Override
  public BigDecimal sumWaitReceiveAmount() {
    String tenantCode = TenantUtils.getTenantCode();
    return receivableInfoRepository.sumWaitReceiveAmountByTenantCode(tenantCode);
  }

  @Override
  public List<ReceivableInfo> findByCustomerAndTenantCode(String customerCode, String tenantCode) {
    return receivableInfoRepository.findByCustomerAndTenantCode(customerCode, tenantCode);
  }

  /**
   * 生成应收账款号
   *
   * @return
   */
  private String generateNo() {
    Date now = new Date();
    SimpleDateFormat dayFormat = new SimpleDateFormat("yyyyMMdd");
    String redisKey = String.format(RECEIVABLE_INFO_NO_KEY_PREFIX, TenantUtils.getTenantCode(), dayFormat.format(now));
    // 设置key有效期为一天
    String index = redisMutexService.getAndIncrement(redisKey, 1, 8, 86401, TimeUnit.SECONDS);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(RECEIVABLE_INFO_NO_PREFIX, format.format(now), index);
  }

  /**
   * 应收账款部分支付状态通知
   *
   * @param receivableInfo
   */
  private void receiptEvent(ReceivableInfo receivableInfo, ReceiptInfo receiptInfo) {
    if (CollectionUtils.isNotEmpty(receivableEventListeners)) {
      for (ReceivableEventListener receivableEventListener : receivableEventListeners) {
        receivableEventListener.onReceipt(receivableInfo, receiptInfo);
      }
    }
  }

  /**
   * 应收账款完成支付状态通知
   *
   * @param receivableInfo
   */
  private void completeEvent(ReceivableInfo receivableInfo) {
    if (CollectionUtils.isNotEmpty(receivableEventListeners)) {
      for (ReceivableEventListener receivableEventListener : receivableEventListeners) {
        receivableEventListener.onReceivableComplete(receivableInfo);
      }
    }
  }
}
