package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.payment.common.enums.PaymentStatus;
import com.bizunited.empower.business.payment.entity.SupplierPaymentInfo;
import com.bizunited.empower.business.payment.repository.SupplierPaymentInfoRepository;
import com.bizunited.empower.business.payment.service.SupplierPaymentInfoService;
import com.bizunited.empower.business.payment.service.notifier.SupplierPaymentEventListener;
import com.bizunited.empower.business.payment.vo.SupplierPaymentInfoVo;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.constant.Constants.SUPPLIER_PAYMENT_INFO_NO_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.SUPPLIER_PAYMENT_INFO_NO_KEY_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.SUPPLIER_PAYMENT_ORDER_LOCK_PREFIX;

/**
 * SupplierPaymentInfo业务模型的服务层接口实现
 *
 * @author Keller
 */
@Service("SupplierPaymentInfoServiceImpl")
public class SupplierPaymentInfoServiceImpl implements SupplierPaymentInfoService {

  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private SupplierPaymentInfoRepository supplierPaymentInfoRepository;
  @Autowired(required = false)
  private List<SupplierPaymentEventListener> supplierPaymentEventListeners;

  @Override
  public SupplierPaymentInfo findDetailsById(String id) {
    return supplierPaymentInfoRepository.findDetailsById(id);
  }

  @Override
  public SupplierPaymentInfo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<SupplierPaymentInfo> op = supplierPaymentInfoRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public SupplierPaymentInfo findByPaymentCode(String paymentCode) {
    return supplierPaymentInfoRepository.findByPaymentCodeAndTenantCode(paymentCode, TenantUtils.getTenantCode());
  }

  @Override
  public SupplierPaymentInfo findByAssociatedCode(String associatedCode) {
    return supplierPaymentInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
  }

  @Override
  public Page<SupplierPaymentInfoVo> findByConditionsForAssociated(Pageable pageable, InvokeParams conditions) {
    return supplierPaymentInfoRepository.queryPageForAssociated(pageable, conditions);
  }

  @Override
  @Transactional
  public SupplierPaymentInfo createByAssociatedCode(String supplierCode, String supplierName, String associatedCode, BigDecimal amount) {
    this.createByAssociatedCodeValidation(supplierCode, supplierName, associatedCode, amount);
    String lockKey = null;
    if (StringUtils.isNotBlank(associatedCode)) {
      lockKey = String.format(SUPPLIER_PAYMENT_ORDER_LOCK_PREFIX, TenantUtils.getTenantCode(), associatedCode);
    } else {
      lockKey = String.format(SUPPLIER_PAYMENT_ORDER_LOCK_PREFIX, TenantUtils.getTenantCode(), supplierCode);
    }
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "采购创建应付账单繁忙，请稍后再试");
    SupplierPaymentInfo supplierPaymentInfo = new SupplierPaymentInfo();
    try {
      Date now = new Date();
      supplierPaymentInfo.setCreateAccount(SecurityUtils.getUserAccount());
      supplierPaymentInfo.setCreateTime(now);
      supplierPaymentInfo.setModifyAccount(SecurityUtils.getUserAccount());
      supplierPaymentInfo.setModifyTime(now);
      supplierPaymentInfo.setAssociatedCode(associatedCode);
      supplierPaymentInfo.setTenantCode(TenantUtils.getTenantCode());
      supplierPaymentInfo.setPaymentCode(generateNo());
      supplierPaymentInfo.setSupplierCode(supplierCode);
      supplierPaymentInfo.setSupplierName(supplierName);
      // 待付款金额
      supplierPaymentInfo.setWaitPayAmount(amount);
      // 应付款金额
      supplierPaymentInfo.setPayAmount(amount);
      // 已付款金额
      supplierPaymentInfo.setPayedAmount(BigDecimal.ZERO);
      supplierPaymentInfo.setPaymentStatus(PaymentStatus.NO_PAYMENT.getValue());
      supplierPaymentInfo.setTstatus(1);
      supplierPaymentInfo = this.supplierPaymentInfoRepository.save(supplierPaymentInfo);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
    return supplierPaymentInfo;
  }

  /**
   * 创建应付账单验证
   *
   * @param supplierCode
   * @param supplierName
   * @param associatedCode
   * @param amount
   */
  private void createByAssociatedCodeValidation(String supplierCode, String supplierName, String associatedCode, BigDecimal amount) {
    Validate.notBlank(supplierCode, "供应商编号不能为空，请检查");
    Validate.notBlank(supplierName, "供应商名称不能为空，请检查");
    Validate.notNull(amount, "金额不能为空，请检查");
    if (StringUtils.isNotBlank(associatedCode)) {
      SupplierPaymentInfo supplierPaymentInfo = supplierPaymentInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
      Validate.isTrue(supplierPaymentInfo == null, "该关联单号的供应商应付账款已存在");
    }
  }

  @Override
  @Transactional
  public SupplierPaymentInfo confirmPay(String payCode, Date payTime) {
    Validate.notBlank(payCode, "采购应付账单编号为空，请检查");
    Validate.notNull(payTime, "支付时间不能为空");
    SupplierPaymentInfo supplierPaymentInfo = supplierPaymentInfoRepository.findByPaymentCodeAndTenantCode(payCode, TenantUtils.getTenantCode());
    Validate.notNull(supplierPaymentInfo, "采购应付账数据为空，请检查");
    supplierPaymentInfo.setPayedAmount(supplierPaymentInfo.getWaitPayAmount());
    supplierPaymentInfo.setWaitPayAmount(BigDecimal.ZERO);
    supplierPaymentInfo.setPaymentStatus(2);
    supplierPaymentInfo.setPayTime(payTime);
    supplierPaymentInfo = supplierPaymentInfoRepository.save(supplierPaymentInfo);
    completeEvent(supplierPaymentInfo);
    return supplierPaymentInfo;
  }

  @Override
  @Transactional
  public void cancelByAssociatedCode(String associatedCode) {
    /*
     * 取消应付账款单（为支付状态下可以取消应付账款）
     * 1、判定应付账款单状态 只有在未进行付款的状态下才能进行取消
     */
    SupplierPaymentInfo supplierPaymentInfo = supplierPaymentInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
    Validate.notNull(supplierPaymentInfo, "采购应付账款单不存在，请检查！");
    Validate.isTrue(supplierPaymentInfo.getPaymentStatus().intValue() == PaymentStatus.NO_PAYMENT.getValue().intValue(), "采购应付账款单取消失败，已完成付款。");
    supplierPaymentInfo.setTstatus(0);
    supplierPaymentInfo = supplierPaymentInfoRepository.saveAndFlush(supplierPaymentInfo);
    cancelEvent(supplierPaymentInfo);
  }

  @Override
  public void cancelByPaymentCode(String paymentCode) {
    /*
     * 取消应付账款单（为支付状态下可以取消应付账款）
     * 1、判定应付账款单状态 只有在未进行付款的状态下才能进行取消
     */
    SupplierPaymentInfo supplierPaymentInfo = supplierPaymentInfoRepository.findByPaymentCodeAndTenantCode(paymentCode, TenantUtils.getTenantCode());
    Validate.notNull(supplierPaymentInfo, "采购应付账款单不存在，请检查！");
    Validate.isTrue(supplierPaymentInfo.getPaymentStatus().intValue() == PaymentStatus.NO_PAYMENT.getValue().intValue(), "采购应付账款单取消失败，已完成付款。");
    supplierPaymentInfo.setTstatus(0);
    supplierPaymentInfo = supplierPaymentInfoRepository.saveAndFlush(supplierPaymentInfo);
    cancelEvent(supplierPaymentInfo);
  }

  /**
   * 生成采购应收账款号
   *
   * @return
   */
  private String generateNo() {
    Date now = new Date();
    SimpleDateFormat dayFormat = new SimpleDateFormat("yyyyMMdd");
    String redisKey = String.format(SUPPLIER_PAYMENT_INFO_NO_KEY_PREFIX, TenantUtils.getTenantCode(), dayFormat.format(now));
    // 设置key有效期为一天
    String index = redisMutexService.getAndIncrement(redisKey, 1, 8, 86401, TimeUnit.SECONDS);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(SUPPLIER_PAYMENT_INFO_NO_PREFIX, format.format(now), index);
  }

  /**
   * 应付账款完成支付状态通知
   *
   * @param supplierPaymentInfo
   */
  private void completeEvent(SupplierPaymentInfo supplierPaymentInfo) {
    if (CollectionUtils.isNotEmpty(supplierPaymentEventListeners)) {
      for (SupplierPaymentEventListener supplierPaymentEventListener : supplierPaymentEventListeners) {
        supplierPaymentEventListener.onPaymentComplete(supplierPaymentInfo);
      }
    }
  }

  /**
   * 应付账款取消支付状态通知
   *
   * @param supplierPaymentInfo
   */
  private void cancelEvent(SupplierPaymentInfo supplierPaymentInfo) {
    if (CollectionUtils.isNotEmpty(supplierPaymentEventListeners)) {
      for (SupplierPaymentEventListener supplierPaymentEventListener : supplierPaymentEventListeners) {
        supplierPaymentEventListener.onPaymentCancelled(supplierPaymentInfo);
      }
    }
  }
}
