package com.bizunited.empower.business.payment.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.payment.common.enums.PaymentStatus;
import com.bizunited.empower.business.payment.common.enums.ReceivableStatus;
import com.bizunited.empower.business.payment.entity.SupplierReceivableInfo;
import com.bizunited.empower.business.payment.repository.SupplierReceivableInfoRepository;
import com.bizunited.empower.business.payment.service.SupplierReceivableInfoService;
import com.bizunited.empower.business.payment.service.notifier.SupplierReceivableEventListener;
import com.bizunited.empower.business.payment.vo.SupplierReceivableInfoVo;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.payment.common.constant.Constants.SUPPLIER_RECEIVABLE_INFO_NO_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.SUPPLIER_RECEIVABLE_INFO_NO_KEY_PREFIX;
import static com.bizunited.empower.business.payment.common.constant.RedisKeys.SUPPLIER_RECEIVABLE_ORDER_LOCK_PREFIX;

/**
 * SupplierReceivableInfo业务模型的服务层接口实现
 *
 * @author Keller
 */
@Service("SupplierReceivableInfoServiceImpl")
public class SupplierReceivableInfoServiceImpl implements SupplierReceivableInfoService {

  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private SupplierReceivableInfoRepository supplierReceivableInfoRepository;
  @Autowired(required = false)
  private List<SupplierReceivableEventListener> supplierReceivableEventListeners;

  @Override
  public SupplierReceivableInfo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.supplierReceivableInfoRepository.findDetailsById(id);
  }

  @Override
  public SupplierReceivableInfo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Optional<SupplierReceivableInfo> op = supplierReceivableInfoRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public SupplierReceivableInfo findByReceivableCode(String receivableCode) {
    if (StringUtils.isBlank(receivableCode)) {
      return null;
    }
    return this.supplierReceivableInfoRepository.findByReceivableCodeAndTenantCode(receivableCode, TenantUtils.getTenantCode());
  }

  @Override
  public SupplierReceivableInfo findByAssociatedCode(String associatedCode) {
    if (StringUtils.isBlank(associatedCode)) {
      return null;
    }
    return this.supplierReceivableInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
  }

  @Override
  public List<SupplierReceivableInfo> findByAssociatedCodes(Set<String> associatedCodes) {
    return this.supplierReceivableInfoRepository.findByAssociatedCodesAndTenantCode(associatedCodes, TenantUtils.getTenantCode());
  }

  @Override
  public Page<SupplierReceivableInfoVo> findByConditionsForAssociated(Pageable pageable, InvokeParams conditions) {
    return this.supplierReceivableInfoRepository.queryPageForAssociated(pageable, conditions);
  }

  @Override
  @Transactional
  public SupplierReceivableInfo confirmReceive(String receivableCode, Date receivableTime) {
    Validate.notBlank(receivableCode, "采购应收账款单号为空，请检查");
    Validate.notNull(receivableTime, "收款时间不能为空，请检查");
    SupplierReceivableInfo receivableInfo = this.supplierReceivableInfoRepository.findByReceivableCodeAndTenantCode(receivableCode, TenantUtils.getTenantCode());
    Validate.notNull(receivableInfo, "采购应付账单为空，请检查");
    receivableInfo.setReceivedAmount(receivableInfo.getWaitReceiveAmount());
    receivableInfo.setWaitReceiveAmount(BigDecimal.ZERO);
    receivableInfo.setReceivableTime(receivableTime);
    receivableInfo.setReceivableStatus(2);
    receivableInfo = this.supplierReceivableInfoRepository.saveAndFlush(receivableInfo);
    completeEvent(receivableInfo);
    return receivableInfo;
  }

  @Override
  @Transactional
  public SupplierReceivableInfo createByAssociatedCode(String supplierCode, String supplierName, String associatedCode, BigDecimal amount) {
    this.createByAssociatedCodeValidation(supplierCode, supplierName, associatedCode, amount);

    String lockKey = null;
    if (StringUtils.isNotBlank(associatedCode)) {
      lockKey = String.format(SUPPLIER_RECEIVABLE_ORDER_LOCK_PREFIX, TenantUtils.getTenantCode(), associatedCode);
    } else {
      lockKey = String.format(SUPPLIER_RECEIVABLE_ORDER_LOCK_PREFIX, TenantUtils.getTenantCode(), supplierCode);
    }
    boolean isLocked = redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "创建采购应收账单繁忙，请稍后再试");
    SupplierReceivableInfo supplierReceivableInfo = new SupplierReceivableInfo();
    try {
      Date now = new Date();
      supplierReceivableInfo.setCreateAccount(SecurityUtils.getUserAccount());
      supplierReceivableInfo.setCreateTime(now);
      supplierReceivableInfo.setModifyAccount(SecurityUtils.getUserAccount());
      supplierReceivableInfo.setModifyTime(now);
      supplierReceivableInfo.setAssociatedCode(associatedCode);
      supplierReceivableInfo.setTenantCode(TenantUtils.getTenantCode());
      supplierReceivableInfo.setReceivableCode(generateNo());
      supplierReceivableInfo.setSupplierCode(supplierCode);
      supplierReceivableInfo.setSupplierName(supplierName);
      // 待收款金额
      supplierReceivableInfo.setWaitReceiveAmount(amount);
      // 应收款金额
      supplierReceivableInfo.setReceivableAmount(amount);
      // 已收款金额
      supplierReceivableInfo.setReceivedAmount(BigDecimal.ZERO);
      supplierReceivableInfo.setReceivableStatus(ReceivableStatus.NO_RECEIVE.getValue());
      supplierReceivableInfo.setTstatus(1);
      supplierReceivableInfo = this.supplierReceivableInfoRepository.save(supplierReceivableInfo);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }
    return supplierReceivableInfo;
  }

  @Override
  @Transactional
  public void cancelByAssociatedCode(String associatedCode) {
    /*
     * 取消应付账款单（为支付状态下可以取消应付账款）
     * 1、判定应付账款单状态 只有在未进行付款的状态下才能进行取消
     */
    SupplierReceivableInfo supplierReceivableInfo = supplierReceivableInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
    Validate.notNull(supplierReceivableInfo, "采购应付账款单不存在，请检查！");
    Validate.isTrue(supplierReceivableInfo.getReceivableStatus().intValue() == PaymentStatus.NO_PAYMENT.getValue().intValue(), "采购应收账款单取消失败，已完成付款。");
    supplierReceivableInfo.setTstatus(0);
    supplierReceivableInfoRepository.saveAndFlush(supplierReceivableInfo);
    cancelEvent(supplierReceivableInfo);
  }

  @Override
  @Transactional
  public void cancelByReceivableCode(String receivableCode) {
    /*
     * 取消应付账款单（为支付状态下可以取消应付账款）
     * 1、判定应付账款单状态 只有在未进行付款的状态下才能进行取消
     */
    SupplierReceivableInfo supplierReceivableInfo = supplierReceivableInfoRepository.findByReceivableCodeAndTenantCode(receivableCode, TenantUtils.getTenantCode());
    Validate.notNull(supplierReceivableInfo, "采购应付账款单不存在，请检查！");
    Validate.isTrue(supplierReceivableInfo.getReceivableStatus().intValue() == PaymentStatus.NO_PAYMENT.getValue().intValue(), "采购应收账款单取消失败，已完成付款。");
    supplierReceivableInfo.setTstatus(0);
    supplierReceivableInfoRepository.saveAndFlush(supplierReceivableInfo);
    cancelEvent(supplierReceivableInfo);
  }

  /**
   * 创建验证
   *
   * @param supplierCode
   * @param supplierName
   * @param associatedCode
   * @param amount
   */
  private void createByAssociatedCodeValidation(String supplierCode, String supplierName, String associatedCode, BigDecimal amount) {
    Validate.notBlank(supplierCode, "供应商编号不存在，请检查");
    Validate.notBlank(supplierName, "供应商名称不存在，请检查");
    Validate.isTrue(amount != null && amount.compareTo(BigDecimal.ZERO) > 0, "采购应收账款金额为空或者负数，请检查");
    if (StringUtils.isNotBlank(associatedCode)) {
      SupplierReceivableInfo supplierReceivableInfo = this.supplierReceivableInfoRepository.findByAssociatedCodeAndTenantCode(associatedCode, TenantUtils.getTenantCode());
      Validate.isTrue(supplierReceivableInfo == null, "该关联单号的供应商应收账款已存在");
    }
  }

  /**
   * 生成采购应收账款号
   *
   * @return
   */
  private String generateNo() {
    Date now = new Date();
    SimpleDateFormat dayFormat = new SimpleDateFormat("yyyyMMdd");
    String redisKey = String.format(SUPPLIER_RECEIVABLE_INFO_NO_KEY_PREFIX, TenantUtils.getTenantCode(), dayFormat.format(now));
    // 设置key有效期为一天
    String index = redisMutexService.getAndIncrement(redisKey, 1, 8, 86401, TimeUnit.SECONDS);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(SUPPLIER_RECEIVABLE_INFO_NO_PREFIX, format.format(now), index);
  }

  /**
   * 应收账款完成支付状态通知
   *
   * @param supplierReceivableInfo
   */
  private void completeEvent(SupplierReceivableInfo supplierReceivableInfo) {
    if (CollectionUtils.isNotEmpty(supplierReceivableEventListeners)) {
      for (SupplierReceivableEventListener supplierReceivableEventListener : supplierReceivableEventListeners) {
        supplierReceivableEventListener.onReceivableComplete(supplierReceivableInfo);
      }
    }
  }

  /**
   * 应收账款取消支付状态通知
   *
   * @param supplierReceivableInfo
   */
  private void cancelEvent(SupplierReceivableInfo supplierReceivableInfo) {
    if (CollectionUtils.isNotEmpty(supplierReceivableEventListeners)) {
      for (SupplierReceivableEventListener supplierReceivableEventListener : supplierReceivableEventListeners) {
        supplierReceivableEventListener.onReceivableCancelled(supplierReceivableInfo);
      }
    }
  }
}
