package com.bizunited.empower.business.payment.service.listener;

import com.bizunited.empower.business.payment.configuration.RabbitmqConfig;
import com.bizunited.empower.business.payment.dto.ElectronicAccountDto;
import com.bizunited.empower.business.payment.service.ElectronicAccountDtoService;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.common.util.tenant.TenantContextHolder;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.annotation.RabbitHandler;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;

/**
 * 电子账户MQ监听器
 * @Author: Paul Chan
 * @Date: 2021/4/14 下午3:43
 */
@Component
@ConditionalOnBean(RabbitmqConfig.class)
@RabbitListener(queues = "${operate.bucketCode:}_electronic_account_queue")
public class ElectronicAccountListener {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ElectronicAccountListener.class);

  @Value("${operate.bucketCode:}")
  private String bucketCode;

  @Autowired
  private ElectronicAccountDtoService electronicAccountDtoService;

  @RabbitHandler
  public void process(byte[] contents) {
    if(ArrayUtils.isEmpty(contents)) {
      return;
    }
    String content = new String(contents, StandardCharsets.UTF_8);
    LOGGER.info("电子账户mq收到:{}", content);
    ElectronicAccountDto electronicAccountDto;
    try {
      electronicAccountDto = JsonUtils.json2Obj(content, ElectronicAccountDto.class);
      if(!bucketCode.equals(electronicAccountDto.getBucketCode())) {
        // 不是当前资源桶的不处理
        return;
      }
      if(electronicAccountDto.getAppType() == null || electronicAccountDto.getAppType() != 2) {
        // 不是经销商平台的不处理
        return;
      }
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      throw new RuntimeException("json解析错误", e);
    }
    Validate.notBlank(electronicAccountDto.getAppCode(), "厂商编码不能为空");
    Validate.notBlank(electronicAccountDto.getTenantCode(), "经销商编码不能为空");
    try {
      TenantContextHolder.setApp(electronicAccountDto.getAppCode());
      TenantContextHolder.setTenant(electronicAccountDto.getTenantCode());
      electronicAccountDtoService.handleMq(electronicAccountDto);
    } finally {
      TenantContextHolder.clean();
    }
  }

}
