package com.bizunited.empower.business.payment.service.listener;

import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.payment.configuration.RabbitmqConfig;
import com.bizunited.empower.business.payment.dto.PayNoticeDto;
import com.bizunited.empower.business.payment.entity.ReceiptInfo;
import com.bizunited.empower.business.payment.service.ElectronicAccountService;
import com.bizunited.empower.business.payment.service.ReceiptInfoService;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.common.util.tenant.TenantContextHolder;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.annotation.RabbitHandler;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;

/**
 * 支付事件变动消息监听
 *
 * @author Keller
 */
@Component
@ConditionalOnBean(RabbitmqConfig.class)
@RabbitListener(queues = "${operate.bucketCode:}_payment.queue")
public class PaymentEventListener {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(PaymentEventListener.class);
  @Autowired
  private CustomerService customerService;
  @Autowired
  private ReceiptInfoService receiptInfoService;
  @Autowired
  private ElectronicAccountService electronicAccountService;
  /**
   * 成功状态
   */
  public static final int SUCCESS = 2;
  /**
   * 失败状态
   */
  public static final int FAILURE = 3;

  @Value("${operate.bucketCode:}")
  private String bucketCode;

  @RabbitHandler
  public void process(byte[] contents) {
    if (ArrayUtils.isEmpty(contents)) {
      return;
    }
    String content = new String(contents, StandardCharsets.UTF_8);
    LOGGER.debug("支付通知收到内容:{}", content);
    PayNoticeDto payNoticeDto;
    try {
      payNoticeDto = JsonUtils.json2Obj(content, PayNoticeDto.class);
      if (!bucketCode.equals(payNoticeDto.getBucketCode())) {
        return;
      }
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      throw new RuntimeException("json解析错误", e);
    }
    Validate.notBlank(payNoticeDto.getAppCode(), "appCode不能为空");
    Validate.notBlank(payNoticeDto.getTenantCode(), "tenantCode不能为空");
    try {
      TenantContextHolder.setApp(payNoticeDto.getAppCode());
      TenantContextHolder.setTenant(payNoticeDto.getTenantCode());
      String[] args = payNoticeDto.getOrderNo().split("-");
      Validate.isTrue( args.length > 0, "业务编号格式错误【%s】，请检查", payNoticeDto.getOrderNo());
      String orderNo = args[1];
      if (SUCCESS == payNoticeDto.getStatus()) {
        receiptInfoService.confirm(orderNo);
        ReceiptInfo receiptInfo = receiptInfoService.findByReceiptCode(orderNo);
        Validate.notNull(receiptInfo, "未找到收款单：%s", orderNo);
        Customer customer = customerService.findByTenantCodeAndCustomerCode(payNoticeDto.getTenantCode(), receiptInfo.getCustomerCode());
        Validate.notNull(customer, "未找到客户:%s", receiptInfo.getCustomerCode());
        electronicAccountService.receipt(payNoticeDto.getAmount(), payNoticeDto.getFeeRate(), payNoticeDto.getFee(), payNoticeDto.getRealAmount(),
            orderNo, payNoticeDto.getRemark(), customer.getCustomerName());
      } else if (FAILURE == payNoticeDto.getStatus()) {
        // 产品沟通后确定收款单增加失败状态,据说是为了统计后续的失败率 2021-05-05
        receiptInfoService.failedByReceiptCode(orderNo);
      }
    } finally {
      TenantContextHolder.clean();
    }
  }
}
