package com.bizunited.empower.business.payment.service.listener;

import com.bizunited.empower.business.payment.configuration.RabbitmqConfig;
import com.bizunited.empower.business.payment.dto.TransferNoticeDto;
import com.bizunited.empower.business.payment.service.ElectronicAccountBillService;
import com.bizunited.empower.business.payment.service.ReceiptInfoService;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.common.util.tenant.TenantContextHolder;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.annotation.RabbitHandler;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;

/**
 * 转账事件变动消息监听
 *
 */
@Component
@ConditionalOnBean(RabbitmqConfig.class)
@RabbitListener(queues = "${operate.bucketCode:}_transfer.queue")
public class TransferEventListener {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(TransferEventListener.class);

  @Value("${operate.bucketCode:}")
  private String bucketCode;

  @Autowired
  private ElectronicAccountBillService electronicAccountBillService;
  @Autowired
  private ReceiptInfoService receiptInfoService;
  /**
   * 提现类型
   */
  public final static int EXTRACT_TYPE = 1;
  /**
   * 线下充值类型
   */
  public final static int RECHARGE_TYPE = 2;

  @RabbitHandler
  public void process(byte[] contents) {
    if(ArrayUtils.isEmpty(contents)) {
      return;
    }
    String content = new String(contents, StandardCharsets.UTF_8);
    LOGGER.info("电子账户提现mq收到:{}", content);
    TransferNoticeDto transferNoticeDto;
    try {
      transferNoticeDto = JsonUtils.json2Obj(content, TransferNoticeDto.class);
      if(!bucketCode.equals(transferNoticeDto.getBucketCode())) {
        // 不是当前资源桶的不处理
        return;
      }
      /*if(electronicAccountDto.getAppType() == null || electronicAccountDto.getAppType() != 2) {
        // 不是经销商平台的不处理
        return;
      }*/
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      throw new RuntimeException("json解析错误", e);
    }
    Validate.notBlank(transferNoticeDto.getAppCode(), "appCode不能为空");
    Validate.notBlank(transferNoticeDto.getTenantCode(), "tenantCode不能为空");
    try {
      TenantContextHolder.setApp(transferNoticeDto.getAppCode());
      TenantContextHolder.setTenant(transferNoticeDto.getTenantCode());
      if(EXTRACT_TYPE == transferNoticeDto.getBusinessType()) {
        // 提现
        electronicAccountBillService.handleExtractCashMq(transferNoticeDto);
      }else if(RECHARGE_TYPE == transferNoticeDto.getBusinessType()){
        // 线下充值
        receiptInfoService.handlerTransfer(transferNoticeDto);
      }
    } finally {
      TenantContextHolder.clean();
    }
  }

}
