package com.bizunited.empower.business.payment.service.strategy;

import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.common.enums.ReceiptStatus;
import com.bizunited.empower.business.payment.entity.CustomerCredit;
import com.bizunited.empower.business.payment.entity.ReceiptInfo;
import com.bizunited.empower.business.payment.repository.ReceiptInfoRepository;
import com.bizunited.empower.business.payment.service.CustomerCreditService;
import com.bizunited.empower.business.payment.service.ReceiptStrategy;
import com.bizunited.empower.business.payment.service.ReceivableInfoService;
import com.bizunited.empower.business.tenant.utils.DealerTenantUtils;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;

/**
 * 信用应收单
 *
 * @author Keller
 */
@Service
public class CreditPayStrategy implements ReceiptStrategy {
  @Autowired
  private ReceiptInfoRepository receiptInfoRepository;
  @Autowired
  private ReceivableInfoService receivableInfoService;
  @Autowired
  private CustomerCreditService customerCreditService;

  @Override
  public Integer getFundsChannel() {
    return FundsChannelType.CREDITPAY.getValue();
  }

  @Override
  @Transactional
  public InvokeParams handler(ReceiptInfo receiptInfo, InvokeParams params) {
    /*
     * 信用支付
     * 1、检查信用支付状态
     * 2、直接扣除待付款金额
     */
    Validate.isTrue(DealerTenantUtils.getCreditBillStatus() != null && DealerTenantUtils.getCreditBillStatus(), "无法使用信用支付，当前供应商不支持信用支付功能！");
    CustomerCredit customerCredit = this.customerCreditService.findByCustomerCode(receiptInfo.getCustomerCode());
    Validate.notNull(customerCredit, "您目前还未开通信用支付功能，请联系您的供应商开启。");
    Validate.isTrue(customerCredit.getTstatus() == 1, "您目前的信用支付功能已被禁用，请联系您的供应商开启。");

    receiptInfo.setReceiptStatus(ReceiptStatus.CONFIRMED.getValue());
    receiptInfo = this.receiptInfoRepository.saveAndFlush(receiptInfo);
    this.receivableInfoService.receive(receiptInfo.getReceivableInfo().getReceivableCode(), receiptInfo.getReceiptAmount(), receiptInfo);
    return params;
  }
}
