package com.bizunited.empower.business.payment.service.strategy;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.empower.business.common.service.SignService;
import com.bizunited.empower.business.common.util.ResponseModelUtils;
import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.entity.CustomerWechat;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.common.enums.ReceiptStatus;
import com.bizunited.empower.business.payment.dto.PayDto;
import com.bizunited.empower.business.payment.entity.ElectronicAccount;
import com.bizunited.empower.business.payment.entity.ReceiptInfo;
import com.bizunited.empower.business.payment.entity.ReceivableInfo;
import com.bizunited.empower.business.payment.feign.PaymentFeignClient;
import com.bizunited.empower.business.payment.repository.ReceiptInfoRepository;
import com.bizunited.empower.business.payment.service.ElectronicAccountService;
import com.bizunited.empower.business.payment.service.ReceiptInfoService;
import com.bizunited.empower.business.payment.service.ReceiptStrategy;
import com.bizunited.empower.business.payment.service.ReceivableInfoService;
import com.bizunited.empower.business.payment.vo.PayVo;
import com.bizunited.empower.business.tenant.entity.TenantTerminalSetting;
import com.bizunited.empower.business.tenant.service.TenantTerminalSettingService;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.InetAddress;
import java.net.UnknownHostException;

import static com.bizunited.empower.business.payment.common.constant.RedisKeys.RECEIPT_INFO_LOCK_PREFIX;

/**
 * 小程序支付
 *
 * @author Keller
 */
@Service
public class MiniPayStrategy implements ReceiptStrategy {
  @Value("${spring.profiles.active}")
  private String env;
  /**
   * 应用类型
   */
  public static final Integer APP_TYPE = 3;
  /**
   * 生产环境值
   */
  public static final String ENV_PROD = "prod";
  @Autowired
  private ReceiptInfoRepository receiptInfoRepository;
  @Autowired
  private ReceivableInfoService receivableInfoService;
  @Autowired
  private ReceiptInfoService receiptInfoService;
  @Autowired
  private PaymentFeignClient paymentFeignClient;
  @Autowired
  private SignService signService;
  @Autowired
  private CustomerService customerService;
  @Autowired
  private TenantTerminalSettingService tenantTerminalSettingService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private ElectronicAccountService electronicAccountService;

  /**
   * 支付方式
   */
  public static final String PAYMENT_WAY = "80";
  /**
   * 跳转前来源 40=小程序
   */
  public static final String REDIRECT_SOURCE = "40";
  /**
   * 支付方式 51=小程序支付
   */
  public static final String PAY_WAY = "51";
  /**
   * 支付类型 31=微信
   */
  public static final String PAY_TYPE = "31";
  /**
   * 订单流水状态正式标书
   */
  public static final Integer PAYMENT_SUCCESS_STATUS = 2;

  @Override
  public Integer getFundsChannel() {
    return FundsChannelType.WECHAYPAY.getValue();
  }

  @Override
  @Transactional
  public InvokeParams handler(ReceiptInfo receiptInfo, InvokeParams params) {
    /*
     * 小程序支付
     * 1、生成支付单据但是为待确认状态
     * 2、生成流水号
     * 3、调用远程支付
     * 4、返回小程序支付
     */
    String key = String.format(RECEIPT_INFO_LOCK_PREFIX, TenantUtils.getTenantCode(), receiptInfo.getReceiptCode());
    try {
      redisMutexService.lock(key);
      ReceivableInfo receivableInfo = receivableInfoService.findByReceivableCode(receiptInfo.getReceivableInfo().getReceivableCode());
      Validate.notNull(receivableInfo, "应付信息异常，请检查！");
      Validate.notNull(receivableInfo.getAssociatedCode(), "订单信息异常，请检查！");

      TenantTerminalSetting tenantTerminalSetting = tenantTerminalSettingService.findByAppCodeAndAppType(TenantUtils.getAppCode(), APP_TYPE);
      Validate.notNull(tenantTerminalSetting, "小程序配置信息异常，请检查！");

      Customer customer = customerService.findDetailsByTenantCodeAndPhone(TenantUtils.getTenantCode(), SecurityUtils.getUserAccount());
      Validate.notNull(customer, "用户信息异常，请检查！");
      CustomerWechat customerWechat = customer.getCustomerWechat();
      Validate.notNull(customerWechat, "用户小程序信息异常，请检查！");
      Validate.notBlank(customerWechat.getTerminalOpenid(), "用户小程序openid信息异常，请检查！");

      PayDto dto = new PayDto();
      // 支付金额转化为 单位：分
      dto.setAmount(receiptInfo.getReceiptAmount().multiply(new BigDecimal("100")).setScale(0, RoundingMode.UP).toString());
      dto.setTenantCode(TenantUtils.getTenantCode());

      // 测试环境中金要求写死的值
      if (ENV_PROD.equals(env)) {
        if (StringUtils.isNotBlank(receivableInfo.getAssociatedCode())) {
          dto.setGoodsName(receivableInfo.getAssociatedCode());
        } else {
          dto.setGoodsName(receivableInfo.getReceivableCode());
        }
      } else {
        dto.setGoodsName("10");
      }
      dto.setSubOpenID(customerWechat.getTerminalOpenid());
      dto.setSubAppID(tenantTerminalSetting.getMiniAppID());
      dto.setOrderNo(StringUtils.join(TenantUtils.getTenantCode(), "-", receiptInfo.getReceiptCode()));
      dto.setPayerUserName(customer.getCustomerName());
      dto.setPayerUserId(customer.getId().replaceAll("-", ""));
      dto.setPaymentWay(PAYMENT_WAY);
      dto.setClientIP(getIp());
      dto.setRedirectSource(REDIRECT_SOURCE);
      dto.setPayWay(PAY_WAY);
      dto.setPayType(PAY_TYPE);

      String payInfo = JSONObject.toJSONString(dto);
      String sign = signService.sign(payInfo);
      ResponseModel result = paymentFeignClient.pay(signService.getCurrentAppId(), sign, payInfo);
      boolean isSuccess = ResponseModelUtils.isSuccess(result);
      if (!isSuccess) {
        this.receiptInfoService.findByReceiptCode(receiptInfo.getReceiptCode());
        return params;
      }
      PayVo payVo = ResponseModelUtils.getSuccessData(result, PayVo.class);
      Validate.notNull(payVo, "支付接口调用失败，请重试！");
      // 该收款单号已经完成支付
      if (PAYMENT_SUCCESS_STATUS.equals(payVo.getStatus())) {
        receiptInfoService.confirm(receiptInfo.getReceiptCode());
        params.putInvokeParam("payVo", payVo);
        return params;
      }
      //判断是否重新发起的支付请求
      boolean redo = false;
      if (StringUtils.isNotBlank(receiptInfo.getId())) {
        redo = true;
      }
      // 微信支付参数返回至备注信息中
      params.putInvokeParam("payVo", payVo);
      receiptInfo.setReceiptStatus(ReceiptStatus.WAITFOR.getValue());
      if (StringUtils.isNotBlank(payVo.getMerchantCode())) {
        // 如果是电子账户操作
        ElectronicAccount electronicAccount = electronicAccountService.findByMerchantCode(payVo.getMerchantCode());
        receiptInfo.setElectronicAccount(electronicAccount);
      }
      receiptInfo = this.receiptInfoRepository.saveAndFlush(receiptInfo);
      if(!redo){
        this.receivableInfoService.waitReceive(receiptInfo.getReceivableInfo().getReceivableCode(), receiptInfo.getReceiptAmount());
      }
      return params;
    } finally {
      redisMutexService.unlock(key);
    }
  }

  private String getIp() {
    ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
    HttpServletRequest request = servletRequestAttributes.getRequest();
    String ipAddress = null;
    ipAddress = request.getHeader("x-forwarded-for");
    if (ipAddress == null || ipAddress.length() == 0 || "unknown".equalsIgnoreCase(ipAddress)) {
      ipAddress = request.getHeader("Proxy-Client-IP");
    }
    if (ipAddress == null || ipAddress.length() == 0 || "unknown".equalsIgnoreCase(ipAddress)) {
      ipAddress = request.getHeader("WL-Proxy-Client-IP");
    }
    if (ipAddress == null || ipAddress.length() == 0 || "unknown".equalsIgnoreCase(ipAddress)) {
      ipAddress = request.getRemoteAddr();
      if (ipAddress.equals("127.0.0.1")) {
        // 根据网卡取本机配置的IP
        InetAddress inet = null;
        try {
          inet = InetAddress.getLocalHost();
        } catch (UnknownHostException e) {
          throw new RuntimeException("获取客户端ip地址异常", e);
        }
        ipAddress = inet.getHostAddress();
      }
    }
    // 对于通过多个代理的情况，第一个IP为客户端真实IP,多个IP按照','分割
    if (ipAddress != null && ipAddress.length() > 15) {
      if (ipAddress.indexOf(",") > 0) {
        ipAddress = ipAddress.substring(0, ipAddress.indexOf(","));
      }
    }
    return ipAddress;
  }
}
