package com.bizunited.empower.business.product.controller;

import com.bizunited.empower.business.product.dto.ProductSpecificationOrderDto;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.service.ProductSpecificationVoService;
import com.bizunited.empower.business.product.vo.ProductSpecificationVo;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.script.context.InvokeParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/v1/productSpecificationVo")
public class ProductSpecificationVoController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ProductSpecificationVoController.class);

  @Autowired
  private ProductSpecificationVoService productSpecificationVoService;

  /**
   * 根据商品规格、单位 查询转换率
   */
  @ApiOperation(value = "根据商品规格、单位 查询转换率")
  @GetMapping(value="/findConvertRate")
  public ResponseModel findConvertRate(@RequestParam("specificationCode") @ApiParam(name="specificationCode" , value="规格编码")String specificationCode,
                                   @RequestParam("unitCode") @ApiParam(name="unitCode" , value="单位编码")String unitCode){
    try {
      BigDecimal current = this.productSpecificationVoService.findConvertRate(specificationCode, unitCode);
      return this.buildHttpResult(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据商品规格查询：商品，品牌，分类，具体规格
   */
  @ApiOperation(value = "根据商品规格集合查询：商品，品牌，分类，具体规格")
  @GetMapping(value="/findBySpecificationCodeList")
  public ResponseModel findBySpecificationCodeList(@RequestParam("specificationCodes") @ApiParam(name="specificationCodes" , value="规格编码")List<String> specificationCodes){
    try {
      List<ProductSpecificationVo> current = this.productSpecificationVoService.findBySpecificationCodeList(specificationCodes);
      return this.buildHttpResult(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据商品规格集合 客户编号 查询：商品，品牌，分类，具体规格 以及客户关联仓库的库存
   */
  @ApiOperation(value = "根据商品规格集合 客户编号 查询：商品，品牌，分类，具体规格 以及客户关联仓库的库存")
  @GetMapping(value="/findBySpecificationCodesAndCustomerCode")
  public ResponseModel findBySpecificationCodesAndCustomerCode(@RequestParam("specificationCodes") @ApiParam(name="specificationCodes" , value="规格编码")List<String> specificationCodes,
                                                               @RequestParam("customerCode") @ApiParam(name="customerCode" , value="规格编码") String customerCode){
    try {
      List<ProductSpecificationVo> current = this.productSpecificationVoService.findBySpecificationCodesAndCustomerCode(specificationCodes,customerCode);
      return this.buildHttpResult(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据商品 品牌集合 查询：商品规格，基本单位
   */
  @ApiOperation(value = "根据商品 品牌集合 查询：商品规格，基本单位")
  @GetMapping(value="/findSpecAndBasicUnitByBrandCodeList")
  public ResponseModel findSpecAndBasicUnitByBrandCodeList(@RequestParam("brandCodeList") @ApiParam(name="brandCodeList" , value="商品品牌")List<String> brandCodeList){
    try {
      List<Pair<String , String>> current = this.productSpecificationVoService.findSpecAndBasicUnitByBrandCodeList(brandCodeList);
      return this.buildHttpResultW(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据商品 分类集合 查询：商品规格，基本单位
   */
  @ApiOperation(value = "根据商品 分类集合 查询：商品规格，基本单位")
  @GetMapping(value="/findSpecAndBasicUnitByCategoryCodeList")
  public ResponseModel findSpecAndBasicUnitByCategoryCodeList(@RequestParam("categoryCodeList") @ApiParam(name="categoryCodeList" , value="商品分类")List<String> categoryCodeList){
    try {
      List<Pair<String , String>> current = this.productSpecificationVoService.findSpecAndBasicUnitByCategoryCodeList(categoryCodeList);
      return this.buildHttpResultW(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }



  /**
   * 校验这些订单是否满足起订量，限定量
   *
   * @param productSpecificationOrderDtoList
   * @return
   */
  @ApiOperation(value = "校验这些订单是否满足起订量，限定量")
  @PatchMapping(value = "/validateOrderQuantity")
  public ResponseModel validateOrderQuantity(@RequestBody List<ProductSpecificationOrderDto> productSpecificationOrderDtoList) {
    try {
      this.productSpecificationVoService.validateOrderQuantity(productSpecificationOrderDtoList);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据商品规格查询：商品，品牌，分类，具体规格
   * */
  @ApiOperation(value = "根据商品规格查询：商品，品牌，分类，具体规格")
  @GetMapping(value="/findBySpecificationCode")
  public ResponseModel findBySpecificationCode(@RequestParam("specificationCode") @ApiParam(name="specificationCode" , value="规格编码")String specificationCode){
    try {
      ProductSpecificationVo current = this.productSpecificationVoService.findBySpecificationCode(specificationCode);
      return this.buildHttpResult(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查询这个租户下的所有规格编码
   * */
  @ApiOperation(value = "查询这个租户下的所有规格编码")
  @GetMapping(value="/findAllSpecificationCode")
  public ResponseModel findAllSpecificationCode(){
    try {
      List<String> current = this.productSpecificationVoService.findAllSpecificationCode();
      return this.buildHttpResult(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }


  /**
   * 多条件分页查询
   *
   * @return
   */
  @ApiOperation(value = "多条件分页查询", notes = "分页参数为page和size，page从0开始，size默认50;" +
      "可传的参数：productCode：商品编码/商品名称（模糊），brandCode：品牌编码（精准），categoryCode：分类编码（精准），" +
      "categoryFlatCode：分类快速编号（它和它的子集），brandFlatCode：品牌快速编号（它和它的子集），available : true  当查询条件为true时，查询所有库存大于0的，其他情况查询所有的" +
      "warehouseCode: 仓库编码  查询仓库下 实际库存大于0的数据")
  @GetMapping("findByConditions")
  public ResponseModel findByConditions(@PageableDefault(50) Pageable pageable,
                                        HttpServletRequest request) {
    try {
      InvokeParams conditions = this.getInvokeParams(request);
      Page<ProductSpecificationVo> result = this.productSpecificationVoService.findByConditions(pageable,conditions);
      return this.buildHttpResult(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 从请求对象
   * @param request
   * @return
   */
  private InvokeParams getInvokeParams(HttpServletRequest request) {
    Map<String, Object> params = new HashMap<>();
    Enumeration<String> names = request.getParameterNames();
    if (names != null) {
      while (names.hasMoreElements()) {
        String name = names.nextElement();
        String[] values = request.getParameterValues(name);
        if (values != null && values.length == 1) {
          params.put(name, values[0]);
        } else if (values != null && values.length > 1) {
          params.put(name, values);
        }
      }
    }
    InvokeParams conditions = new InvokeParams();
    conditions.putInvokeParams(params);
    return conditions;
  }
  /**
   * 通过条形码查询规格信息
   * @param barCode
   * @return
   */
  @ApiOperation(value = "通过条形码查询商品规格信息")
  @GetMapping("/findByBarCode")
  public ResponseModel findByBarCode(@RequestParam("barCode")@ApiParam("条形码")String barCode,@ApiParam("客户编码")@RequestParam("customerCode") String customerCode){
    try {
      ProductSpecificationVo result = this.productSpecificationVoService.findByBarCode(barCode,customerCode);
      return this.buildHttpResultW(result, new String[]{"product","product.productBrand","productUnitSpecificationAndPrices","productPricings","productBarCodeInfos","productUnitSpecificationAndPrices.productUnit"});
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

}
