package com.bizunited.empower.business.product.controller;

import com.bizunited.empower.business.product.optimize.vo.ProductFlatVo;
import com.bizunited.empower.business.product.optimize.service.ProductFlatService;
import com.bizunited.empower.business.product.service.ProductVoService;
import com.bizunited.empower.business.product.vo.ProductVo;
import com.bizunited.empower.business.product.vo.SaasProductVo;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.script.context.InvokeParams;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Product业务模型的MVC Controller层实现，基于HTTP Restful风格
 *
 * @author saturn
 */
@RestController
@RequestMapping("/v1/productVo")
public class ProductVoController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ProductVoController.class);

  @Autowired
  private ProductVoService productVoService;

  @Autowired
  private ProductFlatService productFlatService;

  /**
   * 多条件分页查询
   *
   * @return
   */
  @ApiOperation(value = "多条件分页查询", notes = "分页参数为page和size，page从0开始，size默认50;" +
      "可传的参数：shelfStatus：1 上架 2 下架，productCode：商品编码/商品名称（模糊），brandCode：品牌编码（精准），categoryCode：分类编码（精准），" +
      "sellingPriceLeft/sellingPriceRight：销售价格（范围），tagCode：标签编码（精准），categoryFlatCode：分类快速编号（它和它的子集），brandFlatCode：品牌快速编号（它和它的子集）" +
      "warehouseCode :所属仓库" +
      "productCodeList：商品编码以逗号间隔的拼接串")
  @GetMapping("findByConditions")
  public ResponseModel findByConditions(@PageableDefault(50) Pageable pageable,
                                        HttpServletRequest request) {
    try {
      InvokeParams conditions = this.getInvokeParams(request);
      Page<ProductVo> result = this.productVoService.findByConditions(pageable,conditions);
      return this.buildHttpResult(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 多条件分页查询
   *
   * @return
   */
  @ApiOperation(value = "多条件分页查询", notes = "分页参数为page和size，page从0开始，size默认50;" +
      "可传的参数：productImageStatus：1 待完善 2 已完善，specificationImageStatus : 1 待配图  2 完成配图 productCode：商品编码/商品名称（模糊），brandCode：品牌编码（精准），categoryCode：分类编码（精准），" +
      "categoryFlatCode：分类快速编号（它和它的子集），brandFlatCode：品牌快速编号（它和它的子集）")
  @GetMapping("findImageResourceByConditions")
  public ResponseModel findImageResourceByConditions(@PageableDefault(50) Pageable pageable,
                                        HttpServletRequest request) {
    try {
      InvokeParams conditions = this.getInvokeParams(request);
      Page<ProductVo> result = this.productVoService.findImageResourceByConditions(pageable,conditions);
      return this.buildHttpResult(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  /**
   * 从请求对象
   * @param request
   * @return
   */
  private InvokeParams getInvokeParams(HttpServletRequest request) {
    Map<String, Object> params = new HashMap<>();
    Enumeration<String> names = request.getParameterNames();
    if (names != null) {
      while (names.hasMoreElements()) {
        String name = names.nextElement();
        String[] values = request.getParameterValues(name);
        if (values != null && values.length == 1) {
          params.put(name, values[0]);
        } else if (values != null && values.length > 1) {
          params.put(name, values);
        }
      }
    }
    InvokeParams conditions = new InvokeParams();
    conditions.putInvokeParams(params);
    return conditions;
  }


  /**
   * 按照Product实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   * @param id 主键
   */
  @ApiOperation(value = "按照Product实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value="/findDetailsById" , method={RequestMethod.GET})
  public ResponseModel findDetailsById(@RequestParam("id") @ApiParam("主键") String id) {
    try {
      ProductVo result = this.productVoService.findDetailsById(id);
      return this.buildHttpResult(result);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照Product实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   * 根据客户编码 返回客户价格
   * @param id 主键
   * @param customerCode 客户编码
   */
  @ApiOperation(value = "按照Product实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value="/findDetailsByIdAndCustomerCode" , method={RequestMethod.GET})
  public ResponseModel findDetailsByIdAndCustomerCode(@RequestParam("id") @ApiParam("主键") String id,@RequestParam("customerCode")@ApiParam("客户编码") String customerCode) {
    try {
      ProductVo result = this.productVoService.findDetailsByIdAndCustomerCode(id,customerCode);
      return this.buildHttpResult(result);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照Product编号以及客户编号进行明细查询，查询的明细包括当前业务表单所有的关联属性。
   * 根据客户编码 返回客户价格
   * @param code 商品编号
   * @param customerCode 客户编码
   */
  @ApiOperation(value = "按照Product编号以及客户编号进行明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value="/findDetailsByProductCodeAndCustomerCode" , method={RequestMethod.GET})
  public ResponseModel findDetailsByProductCodeAndCustomerCode(@RequestParam("code") @ApiParam("商品编号") String code,@RequestParam("customerCode")@ApiParam("客户编码") String customerCode) {
    try {
      ProductVo result = this.productVoService.findDetailsByProductCodeAndCustomerCode(code, customerCode);
      return this.buildHttpResult(result);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照Product实体中的（code）进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   */
  @ApiOperation(value = "按照Product实体中的code进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value="/findDetailsByProductCode" , method={RequestMethod.GET})
  public ResponseModel findDetailsByProductCode(@RequestParam("code") @ApiParam("商品编码") String code) {
    try {
      ProductVo result = this.productVoService.findDetailsByProductCode(code);
      return this.buildHttpResult(result);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }


  /**
   * 根据商品集合查询：单位
   */
  @ApiOperation(value = "根据商品集合查询：单位")
  @GetMapping(value="/findUnitByProductCodeList")
  public ResponseModel findUnitByProductCodeList(@RequestParam("productCodes") @ApiParam(name="productCodes" , value="规格编码") List<String> productCodes){
    try {
      List<ProductFlatVo> current = this.productFlatService.findByProductCodeList(productCodes, TenantUtils.getTenantCode());
      return this.buildHttpResult(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过条形码查询规格信息
   * @param barCode
   * @return
   */
  @ApiOperation(value = "通过条形码查询Saas系统商品库规格信息")
  @GetMapping("/findSaasProductByBarCode")
  public ResponseModel findSaasProductByBarCode(@RequestParam("barCode")@ApiParam("条形码")String barCode){

    try {
      SaasProductVo result = this.productVoService.findSaasProductByBarCode(barCode);
      return this.buildHttpResultW(result, new String[]{"productCategory","productBrand","productFiles"});
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过条形码查询商品信息
   * @param barCode
   * @return
   */
  @ApiOperation(value = "通过条形码查询商品信息")
  @GetMapping("/findByBarCode")
  public ResponseModel findByBarCode(@RequestParam("barCode") @ApiParam("条形码") String barCode){
    try {
      ProductVo result = this.productVoService.findByBarCode(barCode);
      return this.buildHttpResult(result);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }

  }

  /**
   * 通过规格编码查询商品信息
   * @param specificationCode
   * @return
   */
  @ApiOperation(value = "通过规格编码查询商品信息")
  @GetMapping("/findBySpecificationCodes")
  public ResponseModel findBySpecificationCodes(@RequestParam("specificationCodes") @ApiParam("规格编码") List<String> specificationCodes){
    try {
      List<ProductVo> result = this.productVoService.findBySpecificationCodes(specificationCodes);
      return this.buildHttpResult(result);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }

  }



}
