package com.bizunited.empower.business.product.optimize.service.internal;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.ReadLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.WriteLock;

import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductBarCodeInfo;
import com.bizunited.empower.business.product.entity.ProductFile;
import com.bizunited.empower.business.product.entity.ProductMultipleSpecification;
import com.bizunited.empower.business.product.entity.ProductPricing;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.entity.ProductTag;
import com.bizunited.empower.business.product.entity.ProductUnitAndPrice;
import com.bizunited.empower.business.product.entity.ProductUnitSpecificationAndPrice;
import com.bizunited.empower.business.product.optimize.vo.ProductFlatVo;
import com.bizunited.empower.business.product.optimize.vo.ProductPricingFlatVo;
import com.bizunited.empower.business.product.optimize.vo.ProductSpecificationFlatVo;
import com.bizunited.empower.business.product.optimize.vo.ProductUnitAndPriceFlatVo;
import com.bizunited.empower.business.product.optimize.vo.ProductUnitSpecificationAndPriceFlatVo;
import com.bizunited.empower.business.product.optimize.service.ProductFlatService;
import com.bizunited.empower.business.product.service.ProductService;
import com.bizunited.empower.business.product.service.ProductSpecificationService;
import com.bizunited.platform.common.service.NebulaToolkitService;

import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.redisson.Redisson;
import org.redisson.api.RTopic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * ProductFlatServiceImpl 操作商品信息与缓存交互的service实现类
 *
 * @Author: hefan
 * @Date: 2021/6/22 11:30
 */
@Service("ProductFlatServiceImpl")
public class ProductFlatServiceImpl implements ProductFlatService {

  @Autowired
  private Redisson redisson;
  @Autowired
  private ProductService productService;
  @Autowired
  private ProductSpecificationService productSpecificationService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 易变的属性标记当前全菜单树缓存是否正在进行刷新，以及是那个线程在进行刷新
   */
  private volatile Thread flashingThread = null;
  private AtomicReference<Thread> atomicReference = new AtomicReference<Thread>(null);

  /**
   * 保证在缓存刷新过程中，需要读取缓存的线程进行阻塞(按照租户进行区分)</br>
   * K0：经销商信息</br>
   * V0：可以使用的读写分离锁</p>
   */
  private static Map<String, ReentrantReadWriteLock> productInfoLockMapping = Maps.newConcurrentMap();

  /**
   * 这是按照经销商为单位进行的当前有效的部分商品缓存信息
   * K0：经销商信息</br>
   * K1：商品编号</br>
   * V0：商品详细信息（带所有关联信息）
   */
  private static Map<String, Map<String, ProductFlatVo>> productInfoCacheMapping = Maps.newConcurrentMap();

  /**
   * 商品规格code索引，维护商品缓存的时候，同时维护索引
   * K0：经销商code
   * K:商品规格code
   * V:商品code
   */
  private static Map<String, Map<String, String>> specificationCodeIndex = Maps.newConcurrentMap();

  private static final Logger LOGGER = LoggerFactory.getLogger(ProductFlatServiceImpl.class);

  @Override
  public void notifyCacheRefresh(String tenantCode, String productCode) {
    if (StringUtils.isBlank(tenantCode)) {
      return;
    }
    Pair<String, String> pair = new ImmutablePair<>(tenantCode, productCode);
    RTopic topic = redisson.getTopic(ProductFlatService.PRODUCT_BASIC_INFO_NOTIFY);
    topic.publish(pair);
  }

  @Override
  public void clearCache(Pair<String, String> pair) {
    String tenantCode = pair.getLeft();
    WriteLock writeLock = this.getWriteLockByTenantCode(tenantCode);
    try {
      writeLock.lock();
      if (productInfoCacheMapping != null) {
        //按商品粒度去缓存
        Map<String, ProductFlatVo> productCodeProductFlatDtoMap = productInfoCacheMapping.get(tenantCode);
        if (productCodeProductFlatDtoMap != null) {
          String productCode = pair.getRight();
          ProductFlatVo remove = productCodeProductFlatDtoMap.remove(productCode);
          this.removeIndex(tenantCode, remove);
        }
      }
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
    } finally {
      writeLock.unlock();
    }
  }

  @Override
  public List<ProductFlatVo> findByProductCodeList(List<String> codeList, String tenantCode) {
    /**
     * 1.入参判定
     * 2.获取读锁实例
     * 3.获取读锁
     * 4.获取缓存中的数据
     *  4.1.经销商还没缓存数据，按需加入商品信息到缓存
     *  4.2.缓存中，系统中，真没有数据
     * 5.遍历入参商品编号，获取商品信息。
     * 6.缓存中只获取到部分数据
     *  6.1 用没拿到信息的code去刷新一遍缓存
     * 返回获取到的商品结果
     *
     */
    // 入参判定
    if (!this.validateParam(codeList, tenantCode)) {
      return null;
    }
    // 获取读锁实例
    ReadLock readLock = this.getReadLockByTenantCode(tenantCode);
    try {
      // 获取读锁
      readLock.lockInterruptibly();
      // 获取缓存中的数据
      Map<String, ProductFlatVo> productCodeProductFlatDtoMap = productInfoCacheMapping.get(tenantCode);
      // 为null，该经销商一条信息都还没缓存
      if (productCodeProductFlatDtoMap == null) {
        // 按需加入商品信息缓存
        productCodeProductFlatDtoMap = this.doRefreshCache(tenantCode, codeList, true);
      }
      // 调用doRefreshCache还为空，那就是真没数据
      if (productCodeProductFlatDtoMap.isEmpty()) {
        return null;
      }
      // 遍历入参商品编号，获取商品信息; 注意：productCodeProductFlatDtoMap是不一定有全量信息的
      List<String> noCachedCodeList = Lists.newLinkedList();
      List<ProductFlatVo> resultList = Lists.newArrayList();
      this.getResultFromCache(codeList, productCodeProductFlatDtoMap, true, noCachedCodeList, resultList);
      // 缓存里拿不到的信息，再去刷新一遍缓存
      if (!noCachedCodeList.isEmpty()) {
        // 按需加入商品信息缓存
        productCodeProductFlatDtoMap = this.doRefreshCache(tenantCode, noCachedCodeList, true);
        this.getResultFromCache(noCachedCodeList, productCodeProductFlatDtoMap, false, null, resultList);
      }
      return resultList;
    } catch (InterruptedException e) {
      LOGGER.error(e.getMessage(), e);
      Thread.currentThread().interrupt();
      return null;
    } finally {
      readLock.unlock();
    }
  }

  @Override
  public ProductFlatVo findByProductCode(String productCode) {
    if (StringUtils.isBlank(productCode)) {
      return null;
    }
    List<String> codeList = Lists.newArrayList(productCode);
    List<ProductFlatVo> list = this.findByProductCodeList(codeList, TenantUtils.getTenantCode());
    return CollectionUtils.isEmpty(list) ? null : list.get(0);
  }

  @Override
  public Optional<ProductFlatVo> findBySpecificationCode(String specificationCode) {
    if (StringUtils.isBlank(specificationCode)) {
      return null;
    }
    List<String> codeList = Lists.newArrayList(specificationCode);
    List<ProductFlatVo> list = this.findBySpecificationCodeList(codeList);
    return CollectionUtils.isEmpty(list) ? Optional.empty() : Optional.ofNullable(list.get(0));
  }

  @Override
  public List<ProductFlatVo> findBySpecificationCodeList(List<String> specificationCodeList) {
    if (CollectionUtils.isEmpty(specificationCodeList)) {
      return null;
    }
    // 缓存中有的商品code
    Set<String> productCodeSet = Sets.newHashSet();
    // 缓存中没有的规格code
    Set<String> specCodeSet = Sets.newHashSet();
    // 通过规格索引，查看内存中是否已缓存商品信息
    Map<String, String> specCodeIndex = specificationCodeIndex.get(TenantUtils.getTenantCode());
    if (specCodeIndex == null) {
      specCodeSet.addAll(specificationCodeList);
    } else {
      for (String code : specificationCodeList) {
        String productCode = specCodeIndex.get(code);
        if (StringUtils.isNotBlank(productCode)) {
          productCodeSet.add(productCode);
        } else {
          specCodeSet.add(code);
        }
      }
    }
    // 未缓存该商品信息，走数据库获取商品code
    Set<String> productCodeSetDB = Sets.newHashSet();
    if (!CollectionUtils.isEmpty(specCodeSet)) {
      productCodeSetDB = productSpecificationService.findProductCodeBySpecificationCodes(specCodeSet);
    }
    // 获取到商品code
    List<String> codeList = (List<String>) org.apache.commons.collections.CollectionUtils.union(productCodeSet, productCodeSetDB);
    return this.findByProductCodeList(codeList, TenantUtils.getTenantCode());
  }

  /**
   * 从缓存中获取商品结果数据
   *
   * @param codeList                     要获取的商品信息的code集合
   * @param productCodeProductFlatDtoMap 缓存中的数据
   * @param collectNoCached              true：要收集缓存中不存在的商品code
   * @param noCachedCodeList             不存在于缓存中的商品code集合
   * @param resultList                   查询结果收集到的商品信息
   */
  private void getResultFromCache(List<String> codeList, Map<String, ProductFlatVo> productCodeProductFlatDtoMap, boolean collectNoCached, List<String> noCachedCodeList, List<ProductFlatVo> resultList) {
    for (String productCode : codeList) {
      ProductFlatVo productFlatVo = productCodeProductFlatDtoMap.get(productCode);
      if (productFlatVo == null && collectNoCached) {
        noCachedCodeList.add(productCode);
      } else {
        resultList.add(productFlatVo);
      }
    }
  }

  /**
   * 校验入参，商品codeList不能为空，还得能获取到当前租户
   *
   * @param codeList
   * @return
   */
  private boolean validateParam(List<String> codeList, String tenantCode) {
    boolean validate = true;
    if (CollectionUtils.isEmpty(codeList)) {
      return !validate;
    }
    if (StringUtils.isBlank(tenantCode)) {
      return !validate;
    }
    return validate;
  }

  /**
   * 按需加入商品信息缓存
   *
   * @param tenantCode
   * @param codeList
   * @param hasReadLock
   * @return 经销商的所有商品缓存信息
   */
  private Map<String, ProductFlatVo> doRefreshCache(String tenantCode, List<String> codeList, boolean hasReadLock) {
    /**
     *  校验入参
     *  获取读写锁实例对象
     *  外部有读锁，释放读锁
     *  用CAS思路来处理重复刷新
     *  查询数据库
     *  获取写锁，开始执行刷新逻辑
     *  制造数据（entity转dto），存入缓存
     *  返回缓存的所有数据
     */
    // 校验入参
    if (!this.validateParam(codeList, tenantCode)) {
      return Maps.newConcurrentMap();
    }
    WriteLock writeLock = this.getWriteLockByTenantCode(tenantCode);
    ReadLock readLock = this.getReadLockByTenantCode(tenantCode);
    try {
      if (hasReadLock) {
        readLock.unlock();
      }
      // 用CAS思路来处理重复刷新
      Thread currentThread = Thread.currentThread();
      boolean isFlashingThread = false;
      if (isFlashingThread = this.atomicReference.compareAndSet(null, currentThread)) {
        this.flashingThread = currentThread;
      } else {
        Thread.yield();
      }
      // 如果没有抢占到加载任务，则一直自旋，直到刷新过程结束
      if (!isFlashingThread) {
        while (flashingThread != null) {
          Thread.yield();
        }
        return productInfoCacheMapping.get(tenantCode);
      }
      // 查询数据库
      List<Product> productList = productService.findByTenantCodeAndProductCodeIn(codeList);
      // 查询结果是空的，但原来缓存不一定是空的(不动缓存内容)
      if (CollectionUtils.isEmpty(productList)) {
        return Maps.newConcurrentMap();
      }
      // 获取写锁，开始执行刷新逻辑
      writeLock.lockInterruptibly();
      LOGGER.info("product info cache refreshing ..... ");
      // 制造数据，存入缓存
      Map<String, ProductFlatVo> cacheMap = productInfoCacheMapping.get(tenantCode);
      cacheMap = Objects.isNull(cacheMap) ? Maps.newConcurrentMap() : cacheMap;
      // entity 转 dto
      for (Product product : productList) {
        ProductFlatVo dto = this.productEntityToDto(product);
        cacheMap.put(product.getProductCode(), dto);
        // 创建索引
        this.createIndex(tenantCode, dto);
      }
      productInfoCacheMapping.put(tenantCode, cacheMap);
      return cacheMap;
    } catch (InterruptedException e) {
      LOGGER.error(e.getMessage(), e);
      return Maps.newConcurrentMap();
    } finally {
      this.atomicReference.compareAndSet(Thread.currentThread(), null);
      this.flashingThread = null;
      if (writeLock.isHeldByCurrentThread()) {
        writeLock.unlock();
      }
      if (hasReadLock) {
        readLock.lock();
      }
    }
  }

  /**
   * 创建索引
   *
   * @param dto
   */
  private void createIndex(String tenantCode, ProductFlatVo dto) {
    Map<String, String> specCodeIndex = specificationCodeIndex.get(tenantCode);
    if (specCodeIndex == null) {
      specCodeIndex = Maps.newConcurrentMap();
    }
    // 创建 商品规格code的索引
    Map<String, ProductSpecificationFlatVo> productSpecificationMap = dto.getProductSpecificationMap();
    for (Map.Entry<String, ProductSpecificationFlatVo> entry : productSpecificationMap.entrySet()) {
      specCodeIndex.put(entry.getKey(), dto.getProductCode());
    }
    specificationCodeIndex.put(tenantCode, specCodeIndex);
  }

  /**
   * 当主数据从内存中移除后，移除对应的索引
   *
   * @param tenantCode
   * @param dto
   */
  private void removeIndex(String tenantCode, ProductFlatVo dto) {
    Map<String, String> specCodeIndex = specificationCodeIndex.get(tenantCode);
    if (specCodeIndex == null) {
      return;
    }
    // 移除 商品规格code的索引
    Map<String, ProductSpecificationFlatVo> productSpecificationMap = dto.getProductSpecificationMap();
    for (Map.Entry<String, ProductSpecificationFlatVo> entry : productSpecificationMap.entrySet()) {
      specCodeIndex.remove(entry.getKey());
    }
  }

  /**
   * 商品缓存的基本信息的数据结构转换，只在这里进行
   * product entity 转 dto
   * TODO flatDto 需要改造
   *
   * @param product
   * @return
   */
  private ProductFlatVo productEntityToDto(Product product) {
    ProductFlatVo dto = new ProductFlatVo();
    Set<ProductSpecificationFlatVo> specDtoSet = Sets.newHashSet();
    dto.setProductCode(product.getProductCode());
    dto.setProductName(product.getProductName());
    dto.setCategoryCode(product.getProductCategory().getCode());
    dto.setCategoryName(product.getProductCategory().getName());
    dto.setCategoryParentCode(product.getProductCategory().getParentCode());
    dto.setCategoryFlatCode(product.getProductCategory().getFlatCode());
    dto.setBrandName(product.getProductBrand().getBrandName());
    dto.setBrandCode(product.getProductBrand().getBrandCode());
    dto.setBrandLogoRelativePath(product.getProductBrand().getLogoRelativePath());
    dto.setBrandLogoFileName(product.getProductBrand().getLogoFileName());
    dto.setShelfStatus(product.getShelfStatus());
    dto.setDefaultWarehouseCode(product.getDefaultWarehouseCode());
    dto.setWarehouseName(product.getWarehouseName());
    dto.setSearchKeyword(product.getSearchKeyword());
    dto.setRemark(product.getRemark());
    Set<ProductFile> productFiles = product.getProductFiles();
    productFiles = (Set<ProductFile>) nebulaToolkitService.copyCollectionByWhiteList(productFiles, ProductFile.class, ProductFile.class, HashSet.class, ArrayList.class);
    dto.setProductFiles(productFiles);
    Set<ProductMultipleSpecification> productMultipleSpecifications = product.getProductMultipleSpecifications();
    productMultipleSpecifications = (Set<ProductMultipleSpecification>) nebulaToolkitService.copyCollectionByWhiteList(productMultipleSpecifications, ProductMultipleSpecification.class, ProductMultipleSpecification.class, HashSet.class, ArrayList.class);
    dto.setProductMultipleSpecifications(productMultipleSpecifications);
    Set<ProductTag> tags = product.getTags();
    tags = (Set<ProductTag>) nebulaToolkitService.copyCollectionByWhiteList(tags, ProductTag.class, ProductTag.class, HashSet.class, ArrayList.class);
    dto.setTags(tags);
    Set<ProductUnitAndPrice> productUnitAndPrices = product.getProductUnitAndPrices();
    productUnitAndPrices = (Set<ProductUnitAndPrice>) nebulaToolkitService.copyCollectionByWhiteList(productUnitAndPrices, ProductUnitAndPrice.class, ProductUnitAndPrice.class, HashSet.class, ArrayList.class, "productUnit");
    dto.setProductUnitAndPrices(productUnitAndPrices);
    Map<String, ProductUnitAndPriceFlatVo> productUnitAndPriceFlatDtoMap = Maps.newHashMap();
    for (ProductUnitAndPrice productUnitAndPrice : productUnitAndPrices) {
      ProductUnitAndPriceFlatVo productUnitAndPriceFlatVo = nebulaToolkitService.copyObjectByWhiteList(productUnitAndPrice, ProductUnitAndPriceFlatVo.class, HashSet.class, ArrayList.class);
      productUnitAndPriceFlatVo.setUnitCode(productUnitAndPrice.getProductUnit().getUnitCode());
      productUnitAndPriceFlatVo.setUnitName(productUnitAndPrice.getProductUnit().getUnitName());
      productUnitAndPriceFlatDtoMap.put(productUnitAndPrice.getProductUnit().getUnitCode(), productUnitAndPriceFlatVo);
    }
    dto.setProductUnitAndPriceMap(productUnitAndPriceFlatDtoMap);

    Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices = product.getProductUnitSpecificationAndPrices();
    productUnitSpecificationAndPrices = (Set<ProductUnitSpecificationAndPrice>) nebulaToolkitService.copyCollectionByWhiteList(productUnitSpecificationAndPrices, ProductUnitSpecificationAndPrice.class, ProductUnitSpecificationAndPrice.class, HashSet.class, ArrayList.class, "productUnit", "productSpecification");
    dto.setProductUnitSpecificationAndPrices(productUnitSpecificationAndPrices);
    Map<String, ProductUnitSpecificationAndPriceFlatVo> productUnitSpecificationAndPriceFlatDtoMap = Maps.newHashMap();
    for (ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice : productUnitSpecificationAndPrices) {
      ProductUnitSpecificationAndPriceFlatVo productUnitSpecificationAndPriceFlatVo = nebulaToolkitService.copyObjectByWhiteList(productUnitSpecificationAndPrice, ProductUnitSpecificationAndPriceFlatVo.class, HashSet.class, ArrayList.class);
      productUnitSpecificationAndPriceFlatVo.setUnitCode(productUnitSpecificationAndPrice.getProductUnit().getUnitCode());
      productUnitSpecificationAndPriceFlatVo.setUnitName(productUnitSpecificationAndPrice.getProductUnit().getUnitName());
      productUnitSpecificationAndPriceFlatVo.setProductSpecificationCode(productUnitSpecificationAndPrice.getProductSpecification().getProductSpecificationCode());
      productUnitSpecificationAndPriceFlatVo.setProductSpecificationName(productUnitSpecificationAndPrice.getProductSpecification().getProductSpecificationName());
      productUnitSpecificationAndPriceFlatDtoMap.put(StringUtils.join(productUnitSpecificationAndPriceFlatVo.getProductSpecificationCode(), productUnitSpecificationAndPriceFlatVo.getUnitCode()), productUnitSpecificationAndPriceFlatVo);
    }
    dto.setProductUnitSpecificationAndPriceFlatDtoMap(productUnitSpecificationAndPriceFlatDtoMap);

    Set<ProductPricing> productPricings = product.getProductPricings();
    productPricings = (Set<ProductPricing>) nebulaToolkitService.copyCollectionByWhiteList(productPricings, ProductPricing.class, ProductPricing.class, HashSet.class, ArrayList.class, "productSpecification", "productPricingUnitSpecifications");
    dto.setProductPricings(productPricings);

    Map<String, ProductPricingFlatVo> productPricingFlatDtoHashMap = Maps.newHashMap();
    for (ProductPricing productPricing : productPricings) {
      ProductPricingFlatVo productPricingFlatVo = nebulaToolkitService.copyObjectByWhiteList(productPricing, ProductPricingFlatVo.class, HashSet.class, ArrayList.class, "productPricingUnits");
      productPricingFlatDtoHashMap.put(productPricingFlatVo.getCustomerCode(), productPricingFlatVo);
    }
    dto.setProductPricingMap(productPricingFlatDtoHashMap);

    Map<String, ProductSpecificationFlatVo> productSpecificationFlatDtoMap = Maps.newHashMap();
    Set<ProductSpecification> productSpecifications = product.getProductSpecifications();
    for (ProductSpecification specification : productSpecifications) {
      ProductSpecificationFlatVo specDto = new ProductSpecificationFlatVo();
      specDto.setProductSpecificationCode(specification.getProductSpecificationCode());
      specDto.setProductSpecificationName(specification.getProductSpecificationName());
      specDto.setMinimumOrderQuantity(specification.getMinimumOrderQuantity());
      specDto.setMaximumOrderQuantity(specification.getMaximumOrderQuantity());
      specDto.setMainImagePath(specification.getMainImagePath());
      specDto.setMainImageName(specification.getMainImageName());
      Set<ProductBarCodeInfo> productBarCodeInfos = specification.getProductBarCodeInfos();
      productBarCodeInfos = (Set<ProductBarCodeInfo>) nebulaToolkitService.copyCollectionByWhiteList(productBarCodeInfos, ProductBarCodeInfo.class, ProductBarCodeInfo.class, HashSet.class, ArrayList.class);
      specDto.setProductBarCodeInfos(productBarCodeInfos);
      Set<ProductUnitSpecificationAndPrice> specificationAndPrices = specification.getProductUnitSpecificationAndPrices();
      specificationAndPrices = (Set<ProductUnitSpecificationAndPrice>) nebulaToolkitService.copyCollectionByWhiteList(specificationAndPrices, ProductUnitSpecificationAndPrice.class, ProductUnitSpecificationAndPrice.class, HashSet.class, ArrayList.class, "productUnit", "productSpecification");
      specDto.setProductUnitSpecificationAndPrices(specificationAndPrices);
      specDtoSet.add(specDto);
      productSpecificationFlatDtoMap.put(specDto.getProductSpecificationCode(), specDto);
    }
    dto.setProductSpecificationMap(productSpecificationFlatDtoMap);
    dto.setProductSpecifications(specDtoSet);
    return dto;
  }

  /**
   * 通过指定的经销商业务编号，获取经销商对缓存读取的读锁
   *
   * @param tenantCode
   * @return
   */
  private ReadLock getReadLockByTenantCode(String tenantCode) {
    while (productInfoLockMapping.get(tenantCode) == null) {
      synchronized (productInfoLockMapping) {
        if (productInfoLockMapping.get(tenantCode) == null) {
          productInfoLockMapping.put(tenantCode, new ReentrantReadWriteLock());
        }
      }
    }
    ReentrantReadWriteLock lock = productInfoLockMapping.get(tenantCode);
    return lock.readLock();
  }

  /**
   * 通过指定的经销商业务编号，获取经销商对缓存操作的写锁
   *
   * @param tenantCode
   * @return
   */
  private WriteLock getWriteLockByTenantCode(String tenantCode) {
    while (productInfoLockMapping.get(tenantCode) == null) {
      synchronized (productInfoLockMapping) {
        if (productInfoLockMapping.get(tenantCode) == null) {
          productInfoLockMapping.put(tenantCode, new ReentrantReadWriteLock());
        }
      }
    }
    ReentrantReadWriteLock lock = productInfoLockMapping.get(tenantCode);
    return lock.writeLock();
  }


}
