package com.bizunited.empower.business.product.optimize.service.task;

import com.bizunited.empower.business.product.optimize.vo.ProductFlatVo;
import com.bizunited.empower.business.product.optimize.service.ProductFlatService;
import com.bizunited.platform.common.util.tenant.TenantContextHolder;
import com.google.common.collect.Lists;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.orm.jpa.EntityManagerFactoryUtils;
import org.springframework.orm.jpa.EntityManagerHolder;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.util.CollectionUtils;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.PersistenceException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.RecursiveTask;

/**
 * ProductFlatRecursiveTask 商品改善查询，forkjoin框架的task类
 *
 * @Author: hefan
 * @Date: 2021/6/17 15:07
 */
public class ProductFlatRecursiveTask extends RecursiveTask<List<ProductFlatVo>> {

  /**
   * 
   */
  private static final long serialVersionUID = -8775852263420340143L;

  /**
   * 入参：要查询的商品code集合
   */
  private List<String> codeList;

  /**
   * 单个task处理的商品查询数量，不得超过上限
   */
  private int upperLimit;

  /**
   * 经销商code
   */
  private String tenantCode;

  /**
   * 厂商code
   */
  private String appCode;

  /**
   * 构造实例的时候，记得注入IOC容器内的托管实例
   */
  private ProductFlatService productFlatService;

  /**
   * JPA实体管理器工厂，由外部spring线程传入
   */
  private EntityManagerFactory emFactory;

  public ProductFlatRecursiveTask(List<String> codeList, int upperLimit, ProductFlatService productFlatService, String tenantCode, String appCode, EntityManagerFactory emFactory) {
    this.codeList = codeList;
    this.upperLimit = upperLimit;
    this.productFlatService = productFlatService;
    this.tenantCode = tenantCode;
    this.appCode = appCode;
    this.emFactory = emFactory;
  }

  @SuppressWarnings("unchecked")
  @Override
  protected List<ProductFlatVo> compute() {
    /**
     * 边界拦截
     * 任务大小拆分
     *    超过处理量上限的入参，对半分割，丢入子任务
     *    合并子任务结果
     * 任务的主要业务逻辑
     *    设置租户上下文
     *    在子线程中添加数据库session，绑定管理工厂
     *    业务处理
     *    关闭数据库session，解绑管理工厂
     *    返回结果
     */
    if (CollectionUtils.isEmpty(codeList)) {
      return Lists.newArrayList();
    }

    int size = codeList.size();
    if (size > upperLimit) {
      // 如果超过上限，对半分割集合
      List<String> codeListLeft = new ArrayList<>();
      List<String> codeListRight = new ArrayList<>();
      int half = size >> 1;
      for (int i = 0; i < size; i++) {
        if (i < half){
          codeListLeft.add(codeList.get(i));
        } else {
          codeListRight.add(codeList.get(i));
        }
      }
      // 拆分任务
      ProductFlatRecursiveTask taskLeft = new ProductFlatRecursiveTask(codeListLeft, this.upperLimit, this.productFlatService, tenantCode, appCode, emFactory);
      ProductFlatRecursiveTask taskRight = new ProductFlatRecursiveTask(codeListRight, this.upperLimit, this.productFlatService, tenantCode, appCode, emFactory);
      invokeAll(taskLeft,taskRight);
      List<ProductFlatVo> left = taskLeft.join();
      List<ProductFlatVo> right = taskRight.join();
      if (!CollectionUtils.isEmpty(left) && !CollectionUtils.isEmpty(right)) {
        return (List<ProductFlatVo>) org.apache.commons.collections.CollectionUtils.union(left, right);
      } else if (!CollectionUtils.isEmpty(left) && CollectionUtils.isEmpty(right)) {
        return left;
      } else {
        return right;
      }
    } else {
      // 设置租户上下文，在下一步【添加线程的数据库session】时，会去加载租户数据源。所以，这两行代码必须在前执行。否则，获取不到正确的数据源
      TenantContextHolder.setTenant(tenantCode);
      TenantContextHolder.setApp(appCode);
      //在线程中添加数据库Session  （为了让子线程可以使用懒加载）
      boolean participate = false;
      if (TransactionSynchronizationManager.hasResource(emFactory)) {
        participate = true;
      }else{
        try {
          EntityManager em = emFactory.createEntityManager();
          EntityManagerHolder emHolder = new EntityManagerHolder(em);
          TransactionSynchronizationManager.bindResource(emFactory, emHolder);
        }
        catch (PersistenceException ex) {
          throw new DataAccessResourceFailureException("Could not create JPA EntityManager at Thread", ex);
        }
      }
      // 业务处理
      List<ProductFlatVo> resultList = productFlatService.findByProductCodeList(codeList, tenantCode);
      //如果添加过数据库Session，在结束时关闭Session （关闭后，不能使用懒加载，即对象导航查询）
      if(!participate){
        EntityManagerHolder emHolder = (EntityManagerHolder)TransactionSynchronizationManager.unbindResource(emFactory);
        EntityManagerFactoryUtils.closeEntityManager(emHolder.getEntityManager());
      }
      return resultList;
    }
  }
}
