package com.bizunited.empower.business.product.repository;

import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.repository.internal.ProductRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

/**
 * Product业务模型的数据库方法支持
 *
 * @author saturn
 */
@Repository("_ProductRepository")
public interface ProductRepository
    extends
    JpaRepository<Product, String>
    , JpaSpecificationExecutor<Product>
    , ProductRepositoryCustom {
  /**
   * 按照商品分类进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  @Query("select distinct product from Product product "
      + " left join fetch product.productCategory product_productCategory "
      + " left join fetch product.productBrand product_productBrand "
      + " left join fetch product.tags product_tags "
      + " where product_productCategory.id = :id and product.tenantCode=:tenantCode")
  Set<Product> findDetailsByProductCategory(@Param("id") String id, @Param("tenantCode") String tenantCode);

  /**
   * 按照品牌进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  @Query("select distinct product from Product product "
      + " left join fetch product.productCategory product_productCategory "
      + " left join fetch product.productBrand product_productBrand "
      + " left join fetch product.tags product_tags "
      + " where product_productBrand.id = :id and product.tenantCode=:tenantCode")
  Set<Product> findDetailsByProductBrand(@Param("id") String id, @Param("tenantCode") String tenantCode);

  /**
   * 按照品牌进行详情查询（包括关联信息）
   *
   * @param codeList
   */
  @Query("select distinct product from Product product "
      + " left join fetch product.productBrand product_productBrand "
      + " left join fetch product.productUnitSpecificationAndPrices product_puap "
      + " left join fetch product.productUnitAndPrices product_unit_specification_prices"
      + " left join fetch product.productSpecifications product_spec "
      + " left join fetch product_puap.productUnit product_unit "
      + " where product.tenantCode= :tenantCode "
      + " and product_productBrand.brandCode in (:codeList)")
  Set<Product> findByProductBrandCodeList(@Param("codeList") List<String> codeList, @Param("tenantCode") String tenantCode);

  /**
   * 按照品牌进行详情查询（包括关联信息）
   *
   * @param codeList
   */
  @Query("select distinct product from Product product "
      + " left join fetch product.productCategory product_category "
      + " left join fetch product.productUnitSpecificationAndPrices product_puap "
      + " left join fetch product.productUnitAndPrices product_unit_specification_prices"
      + " left join fetch product.productSpecifications product_spec "
      + " left join fetch product_puap.productUnit product_unit "
      + " where product.tenantCode= :tenantCode "
      + " and product_category.code in (:codeList) ")
  Set<Product> findByProductCategoryCodeList(@Param("codeList") List<String> codeList, @Param("tenantCode") String tenantCode);


  @Query("select distinct product from Product product "
      + " left join fetch product.productCategory product_category "
      + " left join fetch product.productUnitSpecificationAndPrices product_puap "
      + " left join fetch product.productUnitAndPrices product_unit_specification_prices"
      + " left join fetch product.productSpecifications product_spec "
      + " left join fetch product_puap.productUnit product_unit "
      + " where product.tenantCode= :tenantCode "
      + " and product_category.flatCode like :flatCode% ")
  Set<Product> findByProductCategoryFlatCodeAndTenantCode(@Param("flatCode") String flatCode, @Param("tenantCode") String tenantCode);

  /**
   * 按照商品标签进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  @Query("select distinct product from Product product "
      + " left join fetch product.productCategory product_productCategory "
      + " left join fetch product.productBrand product_productBrand "
      + " left join fetch product.tags product_tags "
      + " where product_tags.id = :id  and product.tenantCode=:tenantCode")
  Set<Product> findDetailsByTags(@Param("id") String id, @Param("tenantCode") String tenantCode);

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  @Query("select distinct product from Product product "
      + " left join fetch product.productCategory product_productCategory "
      + " left join fetch product.productShowCategory product_productShowCategory "
      + " left join fetch product.productBrand product_productBrand "
      + " left join fetch product.tags product_tags "
      + " where product.id=:id ")
  Product findDetailsById(@Param("id") String id);

  /**
   * 按照商品编码进行查询
   *
   * @param productCode 商品编码
   */
  @Query(" from Product f where f.productCode = :productCode and f.tenantCode =:tenantCode ")
  Product findByProductCode(@Param("productCode") String productCode, @Param("tenantCode") String tenantCode);

  /**
   * 按照商品编码进行查询(关联详细信息)
   *
   * @param productCode 商品编码
   */
  @Query("select distinct product from Product product "
      + " left join fetch product.productCategory product_productCategory "
      + " left join fetch product.productBrand product_productBrand "
      + " left join fetch product.tags product_tags "
      + " where product.productCode = :productCode and product.tenantCode =:tenantCode ")
  Product findDetailsByProductCode(@Param("productCode") String productCode, @Param("tenantCode") String tenantCode);

  /**
   * 按照商品编码集合进行查询(关联详细信息)
   * 注：目前只包含品牌、分类、规格、单位与价格、商品文件
   *
   * @param productCodes 商品编码
   */
  @Query("select distinct product from Product product "
      + " left join fetch product.productCategory product_productCategory "
      + " left join fetch product.productBrand product_productBrand "
      + " left join fetch product.productSpecifications product_specifications"
      + " left join fetch product.productUnitSpecificationAndPrices product_unit_specification_prices"
      + " left join fetch product.productUnitAndPrices product_unit_prices"
      + " left join fetch product_unit_prices.productUnit unit"
      + " where product.productCode in (:productCodes) and product.tenantCode =:tenantCode ")
  Set<Product> findDetailsByProductCodes(@Param("productCodes") Set<String> productCodes, @Param("tenantCode") String tenantCode);

  /**
   * 根据租户、商品code修改上架状态
   *
   * @param shelfStatus
   * @param productCode
   * @param tenantCode
   * @return
   */
  @Modifying
  @Query("update Product product set product.shelfStatus=:shelfStatus "
      + " where product.productCode = :productCode and product.tenantCode =:tenantCode ")
  void updateShelfStatus(@Param("shelfStatus") Integer shelfStatus, @Param("productCode") String productCode, @Param("tenantCode") String tenantCode);

  /**
   * 根据商品编号集合查询
   *
   * @param tenantCode
   * @param productCodeList
   * @return
   */
  List<Product> findByTenantCodeAndProductCodeIn(String tenantCode, List<String> productCodeList);


  @Query("from Product product " +
      "left join fetch product.productSpecifications productSpecifications " +
      "left join fetch productSpecifications.productBarCodeInfos productBarCodeInfos " +
      "where productBarCodeInfos.barCode =:barCode and product.tenantCode=:tenantCode")
  Product findByBarCodeAndTenantCode(@Param("barCode") String barCode, @Param("tenantCode") String tenantCode);


  /**
   * 通过规格编码集合查询商品信息
   * @param specificationCodes
   * @param tenantCode
   * @return
   */
  @Query("select distinct product from Product product " +
      "left join fetch product.productSpecifications productSpecifications " +
      "where productSpecifications.productSpecificationCode in (:specificationCodes) and product.tenantCode=:tenantCode")
  List<Product> findBySpecificationCodes(@Param("specificationCodes") List<String> specificationCodes, @Param("tenantCode") String tenantCode);
}
