package com.bizunited.empower.business.product.repository;

import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.repository.internal.ProductSpecificationRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

/**
 * ProductSpecification业务模型的数据库方法支持
 *
 * @author saturn
 */
@Repository("_ProductSpecificationRepository")
public interface ProductSpecificationRepository
    extends
    JpaRepository<ProductSpecification, String>
    , JpaSpecificationExecutor<ProductSpecification>
    , ProductSpecificationRepositoryCustom {
  /**
   * 按照商品编号进行详情查询（包括关联信息）
   *
   * @param product 商品编号
   */
  @Query("select distinct productSpecification from ProductSpecification productSpecification "
      + " left join fetch productSpecification.product productSpecification_product "
      + " left join fetch productSpecification.productUnitSpecificationAndPrices productSpecification_productUnitSpecificationAndPrices "
      + " where productSpecification_product.id = :id")
  Set<ProductSpecification> findDetailsByProduct(@Param("id") String id);

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  @Query("select distinct productSpecification from ProductSpecification productSpecification "
      + " left join fetch productSpecification.product productSpecification_product "
      + " left join fetch productSpecification.productUnitSpecificationAndPrices productSpecification_productUnitSpecificationAndPrices "
      + " where productSpecification.id=:id ")
  ProductSpecification findDetailsById(@Param("id") String id);

  /**
   * 经销商的这些商品规格在数据库中的数量
   *
   * @param productSpecificationCodes
   * @param tenantCode
   * @return
   */
  Integer countByProductSpecificationCodeInAndTenantCode(List<String> productSpecificationCodes, String tenantCode);

  /**
   * 根据租户和规格编码查询规格
   *
   * @param tenantCode
   * @param specificationCode
   * @return
   */
  ProductSpecification findByTenantCodeAndProductSpecificationCode(String tenantCode, String specificationCode);

  /**
   * 根据租户和规格编码集合查询规格
   *
   * @param tenantCode
   * @param specificationCodes
   * @return
   */
  @Query("select distinct p from ProductSpecification p " +
      " left join fetch p.productBarCodeInfos bar " +
      " where p.tenantCode=:tenantCode and p.productSpecificationCode in (:specificationCodes)")
  List<ProductSpecification> findByTenantCodeAndProductSpecificationCodeIn(@Param("tenantCode") String tenantCode, @Param("specificationCodes") List<String> specificationCodes);

  /**
   * 查询该租户下的所有规格编码
   *
   * @param tenantCode
   * @return
   */
  @Query("select distinct productSpecification.productSpecificationCode from ProductSpecification productSpecification "
      + " where productSpecification.tenantCode=:tenantCode ")
  List<String> findProductSpecificationCodeByTenantCode(@Param("tenantCode") String tenantCode);

  /**
   * 查询，租户，某个商品，规格集合之外的规格集合
   *
   * @param tenantCode
   * @param productCode
   * @param ids
   * @return
   */
  @Query("select distinct productSpecification from ProductSpecification productSpecification "
      + " where productSpecification.tenantCode=:tenantCode " +
      " and productSpecification.product.productCode =:productCode " +
      " and productSpecification.id not in(:ids)")
  Set<ProductSpecification> findByProductAndIdNotIn(@Param("tenantCode") String tenantCode, @Param("productCode") String productCode, @Param("ids") List<String> ids);

  /**
   * 查询，租户，某个商品 规格集合
   *
   * @param tenantCode
   * @param productCode
   * @return
   */
  @Query("select distinct productSpecification from ProductSpecification productSpecification "
      + " where productSpecification.tenantCode=:tenantCode " +
      " and productSpecification.product.productCode =:productCode ")
  Set<ProductSpecification> findByProduct(@Param("tenantCode") String tenantCode, @Param("productCode") String productCode);

  /**
   * 根据规格编码查询规格并关联商品和商品价格单位
   *
   * @param productSpecificationCode
   * @param tenantCode
   * @return
   */
  @Query("select distinct ps from ProductSpecification ps " +
      " left join fetch ps.product pro " +
      " left join fetch pro.productUnitSpecificationAndPrices pup " +
      " where ps.tenantCode=:tenantCode " +
      " and ps.productSpecificationCode =:productSpecificationCode ")
  ProductSpecification findByProductSpecificationCodeAndTenantCode(@Param("productSpecificationCode") String productSpecificationCode,
                                                                   @Param("tenantCode") String tenantCode);


  /**
   * 查询资源图是否已使用
   * @param resourceId
   * @return
   */
  int countByProductImageResourceId(@Param("resourceId") String resourceId);

  /**
   * 查询资源图是否已使用 排除传入的ID
   * @param resourceId
   * @param id
   * @return
   */
  int countByProductImageResourceIdAndIdNot(@Param("resourceId") String resourceId, @Param("id") String id);


  /**
   * 通过条形码和租户编码查询规格信息
   *
   * @param tenantCode
   * @param barCode
   * @return
   */
  @Query("from ProductSpecification ps " +
      "left join fetch ps.productBarCodeInfos productBarCode " +
      "left join fetch ps.product product " +
      "left join fetch ps.productUnitSpecificationAndPrices productUnitSpecificationAndPrices " +
      "left join fetch productUnitSpecificationAndPrices.productUnit productUnitSpecificationAndPrices_productUnit " +
      "where ps.tenantCode =:tenantCode and productBarCode.barCode =:barCode")
  ProductSpecification findByTenantCodeAndBarCode(@Param("tenantCode") String tenantCode, @Param("barCode") String barCode);

  /**
   * 根据规格编码集合 查询 商品编码集合
   * @param tenantCode
   * @param specCodeSet
   * @return
   */
  @Query("select distinct pro.productCode from ProductSpecification ps " +
      " left join ps.product pro " +
      " where ps.tenantCode=:tenantCode " +
      " and ps.productSpecificationCode in (:specCodeSet ) ")
  Set<String> findProductCodeBySpecCodes(@Param("tenantCode") String tenantCode, @Param("specCodeSet") Set<String> specCodeSet);
}