package com.bizunited.empower.business.product.repository;

import com.bizunited.empower.business.product.entity.ProductTag;

import java.lang.String;
import java.util.List;
import java.util.Set;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

/**
 * ProductTag业务模型的数据库方法支持
 *
 * @author saturn
 */
@Repository("_ProductTagRepository")
public interface ProductTagRepository
    extends
    JpaRepository<ProductTag, String>
    , JpaSpecificationExecutor<ProductTag> {
  /**
   * 按照商品信息进行详情查询（包括关联信息）
   *
   * @param id 商品信息
   */
  @Query("select distinct productTag from ProductTag productTag "
      + " left join fetch productTag.products productTag_products "
      + " where productTag_products.id = :id")
  Set<ProductTag> findDetailsByProducts(@Param("id") String id);

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  @Query("select distinct productTag from ProductTag productTag "
      + " left join fetch productTag.products productTag_products "
      + " where productTag.id=:id ")
  ProductTag findDetailsById(@Param("id") String id);

  /**
   * 查询：条件：标签名称，租户
   * @param tagName
   * @param tenantCode
   * @return
   */
  List<ProductTag> findByTagNameAndTenantCode(String tagName, String tenantCode);

  /**
   * 查询：条件：id集合
   * @param ids
   * @return
   */
  Set<ProductTag> findByIdIn(String[] ids);

  /**
   * 查询：条件：租户，标签名称（模糊）；排序：标签编码
   * @param tenantCode
   * @param tagName
   * @param pageable
   * @return
   */
  Page<ProductTag> findByTenantCodeAndTagNameContainingOrderByCreateTimeDescTagCode(String tenantCode,String tagName, Pageable pageable);

  /**
   * 根据层级，租户查询
   * @param i
   * @param tenantCode
   * @return
   */
  List<ProductTag> findByLevelAndTenantCode(int i, String tenantCode);

  /**
   * 根据租户，父级code查找
   * @param parentCode
   * @param tenantCode
   * @return
   */
  @Query(" from ProductTag f where f.parentCode = :parentCode and f.tenantCode = :tenantCode order by f.tagCode")
  List<ProductTag> findByParentCode(@Param("parentCode")String parentCode, @Param("tenantCode") String tenantCode);

  /**
   * 根据层级，租户查询 并以标签编码正序
   * @param level 层级
   * @param tenantCode
   * @return
   */
  List<ProductTag> findByLevelAndTenantCodeOrderByTagCode(int level, String tenantCode);

  /**
   * 根据启用状态查询
   * @param status
   * @return
   */
  List<ProductTag> findByTstatusAndTenantCode(Integer status, String tenantCode);
}