package com.bizunited.empower.business.product.repository.internal;

import com.bizunited.empower.business.product.common.enumerate.ProductImageStatus;
import com.bizunited.empower.business.product.common.enumerate.SpecificationImageStatus;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.platform.common.repository.PageRepositoryImpl;
import com.bizunited.platform.script.context.InvokeParams;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Repository;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Product模型的数据层自定义接口实现，可以由程序员根据实际情况完善
 *
 * @author saturn
 */
@Repository("_ProductRepositoryImpl")
public class ProductRepositoryImpl implements ProductRepositoryCustom, PageRepositoryImpl {
  @Autowired
  @PersistenceContext
  private EntityManager entityManager;

  final static String SELECT_PRODUCT_HQL = "select distinct t from Product t  left join fetch t.productSpecifications ps  left join fetch t.productUnitSpecificationAndPrices pus where 1=1";
  final static String SELECT_PRODUCT_COUNT_HQL = "select count( distinct t.id) FROM Product t  left join t.productSpecifications ps  left join t.productUnitSpecificationAndPrices pus where 1=1";


  @Override
  public Page<Product> queryPage(Pageable pageable, InvokeParams conditions) {
    StringBuilder hql = new StringBuilder(SELECT_PRODUCT_HQL);
    StringBuilder countHql = new StringBuilder(SELECT_PRODUCT_COUNT_HQL);
    StringBuilder condition = new StringBuilder();
    Map<String, Object> parameters = new HashMap<>();
    if (conditions != null) {
      Map<String, Object> params = conditions.getInvokeParams();
      // 租户
      String tenantCode = (String) params.get("tenantCode");
      condition.append(" and t.tenantCode =:tenantCode ");
      parameters.put("tenantCode", tenantCode);
      // 允销
      String productCodeList = (String) params.get("productCodeList");
      if (StringUtils.isNotBlank(productCodeList)) {
        String[] split = productCodeList.split(",");
        if (ArrayUtils.isNotEmpty(split)) {
          List<String> productSpecificationCodeList = Arrays.asList(split);
          condition.append(" and ps.productSpecificationCode in( :productSpecificationCodeList )");
          parameters.put("productSpecificationCodeList", productSpecificationCodeList);
        }
      }

      // 商品编码/商品名称
      String productCode = (String) params.get("productCode");
      if (StringUtils.isNotBlank(productCode)) {
        condition.append(" and (t.productCode like concat('%',:productCode,'%') " +
            "or t.productName like concat('%',:productCode,'%') ) ");
        parameters.put("productCode", productCode);
      }
      // 上架状态
      String shelfStatus = (String) params.get("shelfStatus");
      if (StringUtils.isNotBlank(shelfStatus)) {
        condition.append(" and t.shelfStatus =:shelfStatus ");
        parameters.put("shelfStatus", Integer.valueOf(shelfStatus));
      }
      // 商品图库状态
      String productImageStatus = (String) params.get("productImageStatus");
      if (StringUtils.isNotBlank(productImageStatus)) {
        //当查询未配图状态的数据时，需要查询出数据库中状态为空的旧数据，否则两种状态数据之和与全部数据的数量不匹配
        if(productImageStatus.equals(ProductImageStatus.UNFINISHED.getStatus().toString())){
          condition.append(" and (t.productImageStatus =:productImageStatus or t.productImageStatus is null or t.productImageStatus = '')");
        }else{
          condition.append(" and t.productImageStatus =:productImageStatus ");
        }
        parameters.put("productImageStatus", Integer.valueOf(productImageStatus));
      }
      // 规格配图状态
      String specificationImageStatus = (String) params.get("specificationImageStatus");
      if (StringUtils.isNotBlank(specificationImageStatus)) {
        //当查询未配图状态的数据时，需要查询出数据库中状态为空的旧数据，否则两种状态数据之和与全部数据的数量不匹配
        if(specificationImageStatus.equals(SpecificationImageStatus.UNFINISHED.getStatus().toString())){
          condition.append(" and (t.specificationImageStatus =:specificationImageStatus or t.specificationImageStatus is null or t.specificationImageStatus = '')");
        }else{
          condition.append(" and t.specificationImageStatus =:specificationImageStatus ");
        }
        parameters.put("specificationImageStatus", Integer.valueOf(specificationImageStatus));
      }

      // 品牌编码
      String brandCode = (String) params.get("brandCode");
      if (StringUtils.isNotBlank(brandCode)) {
        condition.append(" and t.productBrand.brandCode =:brandCode ");
        parameters.put("brandCode", brandCode);
      }
      // 分类编码
      String categoryCode = (String) params.get("categoryCode");
      if (StringUtils.isNotBlank(categoryCode)) {
        condition.append(" and t.productCategory.code =:categoryCode ");
        parameters.put("categoryCode", categoryCode);
      }
      // 分类快速编号
      String categoryFlatCode = (String) params.get("categoryFlatCode");
      if (StringUtils.isNotBlank(categoryFlatCode)) {
        condition.append(" and t.productCategory.flatCode like concat(:categoryFlatCode,'%') ");
        parameters.put("categoryFlatCode", categoryFlatCode);
      }

      // 前端分类编码
      String showCategoryCode = (String) params.get("showCategoryCode");
      if (StringUtils.isNotBlank(showCategoryCode)) {
        condition.append(" and t.productShowCategory.code =:showCategoryCode ");
        parameters.put("showCategoryCode", showCategoryCode);
      }
      // 前端分类快速编号
      String showCategoryFlatCode = (String) params.get("showCategoryFlatCode");
      if (StringUtils.isNotBlank(showCategoryFlatCode)) {
        condition.append(" and t.productShowCategory.flatCode like concat(:showCategoryFlatCode,'%') ");
        parameters.put("showCategoryFlatCode", showCategoryFlatCode);
      }
      // 商品价格
      String sellingPriceLeft = (String) params.get("sellingPriceLeft");
      String sellingPriceRight = (String) params.get("sellingPriceRight");
      if (StringUtils.isNotBlank(sellingPriceLeft) && StringUtils.isNotBlank(sellingPriceRight)) {
        condition.append(" and (pus.sellingPrice >:sellingPriceLeft and pus.sellingPrice <=:sellingPriceRight )");
        parameters.put("sellingPriceLeft", new BigDecimal(sellingPriceLeft));
        parameters.put("sellingPriceRight", new BigDecimal(sellingPriceRight));
      }else if (StringUtils.isNotBlank(sellingPriceLeft) && StringUtils.isBlank(sellingPriceRight)) {
        condition.append(" and (pus.sellingPrice >:sellingPriceLeft )");
        parameters.put("sellingPriceLeft", new BigDecimal(sellingPriceLeft));
      }if (StringUtils.isBlank(sellingPriceLeft) && StringUtils.isNotBlank(sellingPriceRight)) {
        condition.append(" and (pus.sellingPrice <=:sellingPriceRight )");
        parameters.put("sellingPriceRight", new BigDecimal(sellingPriceRight));
      }
      // 商品标签
      String tagCode = (String) params.get("tagCode");
      if (StringUtils.isNotBlank(tagCode)) {
        condition.append(" and t.tags.tagCode =:tagCode ");
        parameters.put("tagCode", tagCode);
      }
      //所属仓库
      String warehouseCode = (String) params.get("warehouseCode");
      if(StringUtils.isNotBlank(warehouseCode)){
        condition.append(" and t.defaultWarehouseCode =:warehouseCode");
        parameters.put("warehouseCode",warehouseCode);
      }
    }
    hql.append(condition);
    // ===========排序============
    Sort sort = pageable.getSort();
    List<Sort.Order> orders = sort.get().collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(orders)) {
      hql.append(" order by ");
      for (int i = 0; i < orders.size(); i++) {
        Sort.Order order = orders.get(i);
        if (i > 0) {
          hql.append(",");
        }
        hql.append(" t." + order.getProperty() + " ");
        hql.append(order.getDirection().toString());
      }
    } else {
      hql.append(" order by t.createTime desc, t.productCode desc");
    }
    // =========================

    countHql.append(condition);
    return queryByConditions(entityManager, hql.toString(), countHql.toString(), parameters, pageable, false, null);
  }
}
