package com.bizunited.empower.business.product.service;

import com.bizunited.empower.business.product.dto.BrandAndUnitMapBarCodeDto;
import com.bizunited.empower.business.product.dto.ProductSpecificationOrderDto;
import com.bizunited.empower.business.product.vo.ProductSpecificationVo;
import com.bizunited.platform.script.context.InvokeParams;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.math.BigDecimal;
import java.util.List;

public interface ProductSpecificationVoService {
  /**
   * 根据商品规格查询：商品，品牌，分类，具体规格
   * @param specificationCode
   * @return
   */
  ProductSpecificationVo findBySpecificationCode(String specificationCode);
  /**
   * 分页查询，获取规格的信息，包括规格的产品延申信息
   * @param pageable
   * @param conditions
   * @return
   */
  Page<ProductSpecificationVo> findByConditions(Pageable pageable, InvokeParams conditions);
  /**
   * 查询当前租户下的所有规格编号
   * @return
   */
  List<String> findAllSpecificationCode();
  /**
   * 根据规格编号集合，查询规格详细信息
   * @param specificationCodes
   * @return
   */
  List<ProductSpecificationVo> findBySpecificationCodeList(List<String> specificationCodes);

  /**
   * 根据商品规格集合 客户编号 查询：商品，品牌，分类，具体规格 以及客户关联仓库的库存
   * @param specificationCodes
   * @param customerCode
   * @return
   */
  List<ProductSpecificationVo> findBySpecificationCodesAndCustomerCode(List<String> specificationCodes, String customerCode);

  /**
   * 校验这些订单是否满足起订量，限定量productSpecificationVo/findBySpecificationCodeList
   * @param productSpecificationOrderDtoList 传入一批需要进行验证最小订购数量和最大订购数量的商品规格编码信息
   */
  void validateOrderQuantity(List<ProductSpecificationOrderDto> productSpecificationOrderDtoList);

  /**
   * 查询单位转换率
   * @param productSpecificationCode
   * @param unitCode
   * @return
   */
  BigDecimal findConvertRate(String productSpecificationCode, String unitCode);

  /**
   * 该私有方法用于完成指定商品规格的单位转换率计算。</br>
   * 从业务的角度出发，sourceUnitCode为输入的用户购买商品规格的单位编号，targetUnitCode为（营销活动中）设定的商品规格的单位编号
   * @param productSpecificationCode
   * @param sourceUnitCode
   * @param targetUnitCode
   * @return
   */
  float transformationUnitRate(String productSpecificationCode , String sourceUnitCode , String targetUnitCode);


    /**
     * 根据商品 品牌集合 查询：商品规格，基本单位
     * @param brandCodeList
     * @return 返回值中，第一个为规格编号，另一个为基础单位编号
     */
  List<Pair<String , String>> findSpecAndBasicUnitByBrandCodeList(List<String> brandCodeList);

  /**
   * 根据商品 分类集合 查询：商品规格，基本单位
   * @param categoryCodeList
   * @return 返回值中，第一个为规格编号，另一个为基础单位编号
   */
  List<Pair<String , String>> findSpecAndBasicUnitByCategoryCodeList(List<String> categoryCodeList);

  /**
   * 根据商品 品牌集合 查询
   * @param brandCodeList
   * @return
   */
  List<ProductSpecificationVo> findByBrandCodeList(List<String> brandCodeList);

  /**
   * 根据商品 分类集合 查询
   * @param categoryCodeList
   * @return
   */
  List<ProductSpecificationVo> findByCategoryCodeList(List<String> categoryCodeList);

  /**
   * 条件：商品规格编码集合
   * 结果：品牌code和单位code 映射的条码
   * @param specificationCodes
   * @return
   */
  BrandAndUnitMapBarCodeDto findBrandAndUnitMapBarCodeBySpecCodes(List<String> specificationCodes);

  /**
   * 通过条形码查询规格信息
   * @param barCode
   * @param customerCode
   * @return
   */
  ProductSpecificationVo findByBarCode(String barCode, String customerCode);


  /**
   * 商品基本单位数量转换
   * @param productCode
   * @param unitCode
   * @param quantity
   * @return
   */
  BigDecimal convertProductBasicUnitQuantity(String productCode,String unitCode,BigDecimal quantity);

  /**
   * 商品单位数量转换
   * @param productCode
   * @param sourceUnitCode
   * @param targetUnitCode
   * @param quantity
   * @return
   */
  BigDecimal covertProductUnitQuantity(String productCode,String sourceUnitCode,String targetUnitCode,BigDecimal quantity);

  /**
   * 规格基本单位数量转换
   * @param specificationCode
   * @param unitCode
   * @param quantity
   * @return
   */
  BigDecimal convertSpecificationBasicUnitQuantity(String specificationCode,String unitCode,BigDecimal quantity);


  /**
   * 规格单位数量转换
   * @param specificationCode
   * @param sourceUnitCode
   * @param targetUnitCode
   * @param quantity
   * @return
   */
  BigDecimal convertSpecificationUnitQuantity(String specificationCode,String sourceUnitCode,String targetUnitCode,BigDecimal quantity);

}
