package com.bizunited.empower.business.product.service.handler;

import com.bizunited.empower.business.customer.entity.Customer;
import com.bizunited.empower.business.customer.entity.CustomerLevel;
import com.bizunited.empower.business.customer.service.CustomerService;
import com.bizunited.empower.business.product.dto.DiscountResultDto;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductUnitSpecificationAndPrice;
import com.bizunited.empower.business.product.service.ProductService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 根据客户级别打折
 */
@Component("CustomerLevelDiscountHandler")
public class CustomerLevelDiscountHandler extends DiscountHandler {

  @Autowired
  CustomerService customerService;
  @Autowired
  ProductService productService;

  @Override
  public List<DiscountResultDto> basedOnOriginalPrice(String customerCode, String productCode) {
    /**
     * 1.获取当前客户的等级的打折率
     * 2.获取当前商品的交易价
     * 3.计算本轮价格
     */
    Customer customer = this.customerService.findByTenantCodeAndCustomerCode(TenantUtils.getTenantCode(), customerCode);
    Validate.notNull(customer, "%s未找到客户", customerCode);
    CustomerLevel customerLevel = customer.getCustomerLevel();
    Validate.notNull(customerLevel, "%s未找到客户的等级", customerCode);
    BigDecimal rebateDecimal = customerLevel.getRebate();

    BigDecimal discountPrice;
    Product product = this.productService.findByProductCode(productCode);
    Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices = product.getProductUnitSpecificationAndPrices();
    List<DiscountResultDto> currentBase =new ArrayList<>(productUnitSpecificationAndPrices.size());
    for (ProductUnitSpecificationAndPrice unitAndPrice : productUnitSpecificationAndPrices) {
      DiscountResultDto current = new DiscountResultDto(customerCode, productCode);
      BigDecimal sellingPrice = unitAndPrice.getSellingPrice();
      discountPrice = sellingPrice.multiply(rebateDecimal).divide(new BigDecimal(100), 4, RoundingMode.HALF_UP);
      current.setAfterDiscountPrice(discountPrice);
      current.setUnitCode(unitAndPrice.getProductUnit().getUnitCode());
      current.setProductSpecificationCode(unitAndPrice.getProductSpecification().getProductSpecificationCode());
      currentBase.add(current);
    }
    return currentBase;
  }

  @Override
  public List<DiscountResultDto> addUpDiscountResult(List<DiscountResultDto> prevResult, List<DiscountResultDto> currentBase) {
    if (CollectionUtils.isEmpty(prevResult)) {
      return currentBase;
    }
    Map<String, List<DiscountResultDto>> puMapDiscountResult = prevResult.stream().collect(Collectors.groupingBy(discountResultDto -> {
      String productCode = discountResultDto.getProductCode();
      String unitCode = discountResultDto.getUnitCode();
      String specificationCode = discountResultDto.getProductSpecificationCode();
      return StringUtils.join(productCode, unitCode,specificationCode);
    }));
    for (DiscountResultDto current : currentBase) {
      String productCode = current.getProductCode();
      String unitCode = current.getUnitCode();
      String specificationCode = current.getProductSpecificationCode();
      List<DiscountResultDto> prev = puMapDiscountResult.get(StringUtils.join(productCode, unitCode,specificationCode));
      if (CollectionUtils.isEmpty(prev)) {
        continue;
      }
      BigDecimal prevPrice = prev.get(0).getAfterDiscountPrice();
      BigDecimal currentPrice = current.getAfterDiscountPrice();
      BigDecimal result = prevPrice.add(currentPrice);
      current.setAfterDiscountPrice(result);
    }
    return currentBase;
  }

  @Override
  public Boolean isOver() {
    return false;
  }
}
